// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/chromeos/drive/resource_metadata_storage.h"

#include <algorithm>

#include "base/file_util.h"
#include "base/files/scoped_temp_dir.h"
#include "chrome/browser/chromeos/drive/drive.pb.h"
#include "chrome/browser/chromeos/drive/test_util.h"
#include "content/public/test/test_browser_thread_bundle.h"
#include "testing/gtest/include/gtest/gtest.h"
#include "third_party/leveldatabase/src/include/leveldb/db.h"

namespace drive {
namespace internal {

class ResourceMetadataStorageTest : public testing::Test {
 protected:
  virtual void SetUp() OVERRIDE {
    ASSERT_TRUE(temp_dir_.CreateUniqueTempDir());

    storage_.reset(new ResourceMetadataStorage(
        temp_dir_.path(), base::MessageLoopProxy::current()));
    ASSERT_TRUE(storage_->Initialize());
  }

  // Overwrites |storage_|'s version.
  void SetDBVersion(int version) {
    ResourceMetadataHeader header;
    ASSERT_TRUE(storage_->GetHeader(&header));
    header.set_version(version);
    EXPECT_TRUE(storage_->PutHeader(header));
  }

  bool CheckValidity() {
    return storage_->CheckValidity();
  }

  // Puts a child entry.
  void PutChild(const std::string& parent_resource_id,
                const std::string& child_base_name,
                const std::string& child_resource_id) {
    storage_->resource_map_->Put(
        leveldb::WriteOptions(),
        ResourceMetadataStorage::GetChildEntryKey(parent_resource_id,
                                                  child_base_name),
        child_resource_id);
  }

  // Removes a child entry.
  void RemoveChild(const std::string& parent_resource_id,
                   const std::string& child_base_name) {
    storage_->resource_map_->Delete(
        leveldb::WriteOptions(),
        ResourceMetadataStorage::GetChildEntryKey(parent_resource_id,
                                                  child_base_name));
  }

  content::TestBrowserThreadBundle thread_bundle_;
  base::ScopedTempDir temp_dir_;
  scoped_ptr<ResourceMetadataStorage,
             test_util::DestroyHelperForTests> storage_;
};

TEST_F(ResourceMetadataStorageTest, LargestChangestamp) {
  const int64 kLargestChangestamp = 1234567890;
  EXPECT_TRUE(storage_->SetLargestChangestamp(kLargestChangestamp));
  EXPECT_EQ(kLargestChangestamp, storage_->GetLargestChangestamp());
}

TEST_F(ResourceMetadataStorageTest, PutEntry) {
  const std::string key1 = "abcdefg";
  const std::string key2 = "abcd";
  const std::string key3 = "efgh";
  const std::string name2 = "ABCD";
  const std::string name3 = "EFGH";

  ResourceEntry entry1;
  entry1.set_resource_id(key1);

  // key1 not found.
  ResourceEntry result;
  EXPECT_FALSE(storage_->GetEntry(key1, &result));

  // Put entry1.
  EXPECT_TRUE(storage_->PutEntry(entry1));

  // key1 found.
  ASSERT_TRUE(storage_->GetEntry(key1, &result));
  EXPECT_EQ(key1, result.resource_id());

  // key2 not found.
  EXPECT_FALSE(storage_->GetEntry(key2, &result));

  // Put entry2 as a child of entry1.
  ResourceEntry entry2;
  entry2.set_parent_resource_id(key1);
  entry2.set_resource_id(key2);
  entry2.set_base_name(name2);
  EXPECT_TRUE(storage_->PutEntry(entry2));

  // key2 found.
  EXPECT_TRUE(storage_->GetEntry(key2, &result));
  EXPECT_EQ(key2, storage_->GetChild(key1, name2));

  // Put entry3 as a child of entry2.
  ResourceEntry entry3;
  entry3.set_parent_resource_id(key2);
  entry3.set_resource_id(key3);
  entry3.set_base_name(name3);
  EXPECT_TRUE(storage_->PutEntry(entry3));

  // key3 found.
  EXPECT_TRUE(storage_->GetEntry(key3, &result));
  EXPECT_EQ(key3, storage_->GetChild(key2, name3));

  // Change entry3's parent to entry1.
  entry3.set_parent_resource_id(key1);
  EXPECT_TRUE(storage_->PutEntry(entry3));

  // entry3 is a child of entry1 now.
  EXPECT_TRUE(storage_->GetChild(key2, name3).empty());
  EXPECT_EQ(key3, storage_->GetChild(key1, name3));

  // Remove entries.
  EXPECT_TRUE(storage_->RemoveEntry(key3));
  EXPECT_FALSE(storage_->GetEntry(key3, &result));
  EXPECT_TRUE(storage_->RemoveEntry(key2));
  EXPECT_FALSE(storage_->GetEntry(key2, &result));
  EXPECT_TRUE(storage_->RemoveEntry(key1));
  EXPECT_FALSE(storage_->GetEntry(key1, &result));
}

TEST_F(ResourceMetadataStorageTest, Iterator) {
  // Prepare data.
  std::vector<ResourceEntry> entries;
  ResourceEntry entry;

  entry.set_resource_id("entry1");
  entries.push_back(entry);
  entry.set_resource_id("entry2");
  entries.push_back(entry);
  entry.set_resource_id("entry3");
  entries.push_back(entry);
  entry.set_resource_id("entry4");
  entries.push_back(entry);

  for (size_t i = 0; i < entries.size(); ++i)
    EXPECT_TRUE(storage_->PutEntry(entries[i]));

  // Insert some dummy cache entries.
  FileCacheEntry cache_entry;
  EXPECT_TRUE(storage_->PutCacheEntry(entries[0].resource_id(), cache_entry));
  EXPECT_TRUE(storage_->PutCacheEntry(entries[1].resource_id(), cache_entry));

  // Iterate and check the result.
  std::map<std::string, ResourceEntry> result;
  scoped_ptr<ResourceMetadataStorage::Iterator> it = storage_->GetIterator();
  ASSERT_TRUE(it);
  for (; !it->IsAtEnd(); it->Advance()) {
    const ResourceEntry& entry = it->Get();
    result[entry.resource_id()] = entry;
  }
  EXPECT_FALSE(it->HasError());

  EXPECT_EQ(entries.size(), result.size());
  for (size_t i = 0; i < entries.size(); ++i)
    EXPECT_EQ(1U, result.count(entries[i].resource_id()));
}

TEST_F(ResourceMetadataStorageTest, PutCacheEntry) {
  FileCacheEntry entry;
  const std::string key1 = "abcdefg";
  const std::string key2 = "abcd";
  const std::string md5_1 = "foo";
  const std::string md5_2 = "bar";

  // Put cache entries.
  entry.set_md5(md5_1);
  EXPECT_TRUE(storage_->PutCacheEntry(key1, entry));
  entry.set_md5(md5_2);
  EXPECT_TRUE(storage_->PutCacheEntry(key2, entry));

  // Get cache entires.
  EXPECT_TRUE(storage_->GetCacheEntry(key1, &entry));
  EXPECT_EQ(md5_1, entry.md5());
  EXPECT_TRUE(storage_->GetCacheEntry(key2, &entry));
  EXPECT_EQ(md5_2, entry.md5());

  // Remove cache entries.
  EXPECT_TRUE(storage_->RemoveCacheEntry(key1));
  EXPECT_FALSE(storage_->GetCacheEntry(key1, &entry));

  EXPECT_TRUE(storage_->RemoveCacheEntry(key2));
  EXPECT_FALSE(storage_->GetCacheEntry(key2, &entry));
}

TEST_F(ResourceMetadataStorageTest, CacheEntryIterator) {
  // Prepare data.
  std::map<std::string, FileCacheEntry> entries;
  FileCacheEntry cache_entry;

  cache_entry.set_md5("aA");
  entries["entry1"] = cache_entry;
  cache_entry.set_md5("bB");
  entries["entry2"] = cache_entry;
  cache_entry.set_md5("cC");
  entries["entry3"] = cache_entry;
  cache_entry.set_md5("dD");
  entries["entry4"] = cache_entry;

  for (std::map<std::string, FileCacheEntry>::iterator it = entries.begin();
       it != entries.end(); ++it)
    EXPECT_TRUE(storage_->PutCacheEntry(it->first, it->second));

  // Insert some dummy entries.
  ResourceEntry entry;
  entry.set_resource_id("entry1");
  EXPECT_TRUE(storage_->PutEntry(entry));
  entry.set_resource_id("entry2");
  EXPECT_TRUE(storage_->PutEntry(entry));

  // Iterate and check the result.
  scoped_ptr<ResourceMetadataStorage::CacheEntryIterator> it =
      storage_->GetCacheEntryIterator();
  ASSERT_TRUE(it);
  size_t num_entries = 0;
  for (; !it->IsAtEnd(); it->Advance()) {
    EXPECT_EQ(1U, entries.count(it->GetID()));
    EXPECT_EQ(entries[it->GetID()].md5(), it->GetValue().md5());
    ++num_entries;
  }
  EXPECT_FALSE(it->HasError());
  EXPECT_EQ(entries.size(), num_entries);
}

TEST_F(ResourceMetadataStorageTest, GetChildren) {
  const std::string parents_id[] = { "mercury", "venus", "mars", "jupiter",
                                     "saturn" };
  std::vector<std::vector<std::pair<std::string, std::string> > >
      children_name_id(arraysize(parents_id));
  // Skip children_name_id[0/1] here because Mercury and Venus have no moon.
  children_name_id[2].push_back(std::make_pair("phobos", "mars_i"));
  children_name_id[2].push_back(std::make_pair("deimos", "mars_ii"));
  children_name_id[3].push_back(std::make_pair("io", "jupiter_i"));
  children_name_id[3].push_back(std::make_pair("europa", "jupiter_ii"));
  children_name_id[3].push_back(std::make_pair("ganymede", "jupiter_iii"));
  children_name_id[3].push_back(std::make_pair("calisto", "jupiter_iv"));
  children_name_id[4].push_back(std::make_pair("mimas", "saturn_i"));
  children_name_id[4].push_back(std::make_pair("enceladus", "saturn_ii"));
  children_name_id[4].push_back(std::make_pair("tethys", "saturn_iii"));
  children_name_id[4].push_back(std::make_pair("dione", "saturn_iv"));
  children_name_id[4].push_back(std::make_pair("rhea", "saturn_v"));
  children_name_id[4].push_back(std::make_pair("titan", "saturn_vi"));
  children_name_id[4].push_back(std::make_pair("iapetus", "saturn_vii"));

  // Put parents.
  for (size_t i = 0; i < arraysize(parents_id); ++i) {
    ResourceEntry entry;
    entry.set_resource_id(parents_id[i]);
    EXPECT_TRUE(storage_->PutEntry(entry));
  }

  // Put children.
  for (size_t i = 0; i < children_name_id.size(); ++i) {
    for (size_t j = 0; j < children_name_id[i].size(); ++j) {
      ResourceEntry entry;
      entry.set_parent_resource_id(parents_id[i]);
      entry.set_base_name(children_name_id[i][j].first);
      entry.set_resource_id(children_name_id[i][j].second);
      EXPECT_TRUE(storage_->PutEntry(entry));
    }
  }

  // Put some dummy cache entries.
  for (size_t i = 0; i < arraysize(parents_id); ++i) {
    FileCacheEntry cache_entry;
    EXPECT_TRUE(storage_->PutCacheEntry(parents_id[i], cache_entry));
  }

  // Try to get children.
  for (size_t i = 0; i < children_name_id.size(); ++i) {
    std::vector<std::string> children;
    storage_->GetChildren(parents_id[i], &children);
    EXPECT_EQ(children_name_id[i].size(), children.size());
    for (size_t j = 0; j < children_name_id[i].size(); ++j) {
      EXPECT_EQ(1, std::count(children.begin(),
                              children.end(),
                              children_name_id[i][j].second));
    }
  }
}

TEST_F(ResourceMetadataStorageTest, OpenExistingDB) {
  const std::string parent_id1 = "abcdefg";
  const std::string child_name1 = "WXYZABC";
  const std::string child_id1 = "qwerty";

  ResourceEntry entry1;
  entry1.set_resource_id(parent_id1);
  ResourceEntry entry2;
  entry2.set_resource_id(child_id1);
  entry2.set_parent_resource_id(parent_id1);
  entry2.set_base_name(child_name1);

  // Put some data.
  EXPECT_TRUE(storage_->PutEntry(entry1));
  EXPECT_TRUE(storage_->PutEntry(entry2));

  // Close DB and reopen.
  storage_.reset(new ResourceMetadataStorage(
      temp_dir_.path(), base::MessageLoopProxy::current()));
  ASSERT_TRUE(storage_->Initialize());

  // Can read data.
  ResourceEntry result;
  ASSERT_TRUE(storage_->GetEntry(parent_id1, &result));
  EXPECT_EQ(parent_id1, result.resource_id());

  ASSERT_TRUE(storage_->GetEntry(child_id1, &result));
  EXPECT_EQ(child_id1, result.resource_id());
  EXPECT_EQ(parent_id1, result.parent_resource_id());
  EXPECT_EQ(child_name1, result.base_name());

  EXPECT_EQ(child_id1, storage_->GetChild(parent_id1, child_name1));
}

TEST_F(ResourceMetadataStorageTest, IncompatibleDB) {
  const int64 kLargestChangestamp = 1234567890;
  const std::string key1 = "abcd";

  ResourceEntry entry;
  entry.set_resource_id(key1);

  // Put some data.
  ResourceEntry result;
  EXPECT_TRUE(storage_->SetLargestChangestamp(kLargestChangestamp));
  EXPECT_TRUE(storage_->PutEntry(entry));
  EXPECT_TRUE(storage_->GetEntry(key1, &result));

  // Set incompatible version and reopen DB.
  SetDBVersion(ResourceMetadataStorage::kDBVersion - 1);
  storage_.reset(new ResourceMetadataStorage(
      temp_dir_.path(), base::MessageLoopProxy::current()));
  ASSERT_TRUE(storage_->Initialize());

  // Data is erased because of the incompatible version.
  EXPECT_EQ(0, storage_->GetLargestChangestamp());
  EXPECT_FALSE(storage_->GetEntry(key1, &result));
}

TEST_F(ResourceMetadataStorageTest, WrongPath) {
  // Create a file.
  base::FilePath path;
  ASSERT_TRUE(file_util::CreateTemporaryFileInDir(temp_dir_.path(), &path));

  storage_.reset(new ResourceMetadataStorage(
      path, base::MessageLoopProxy::current()));
  // Cannot initialize DB beacause the path does not point a directory.
  ASSERT_FALSE(storage_->Initialize());
}

TEST_F(ResourceMetadataStorageTest, CheckValidity) {
  const std::string key1 = "foo";
  const std::string name1 = "hoge";
  const std::string key2 = "bar";
  const std::string name2 = "fuga";
  const std::string key3 = "boo";
  const std::string name3 = "piyo";

  // Empty storage is valid.
  EXPECT_TRUE(CheckValidity());

  // Put entry with key1.
  ResourceEntry entry;
  entry.set_resource_id(key1);
  entry.set_base_name(name1);
  EXPECT_TRUE(storage_->PutEntry(entry));
  EXPECT_TRUE(CheckValidity());

  // Put entry with key2 under key1.
  entry.set_resource_id(key2);
  entry.set_parent_resource_id(key1);
  entry.set_base_name(name2);
  EXPECT_TRUE(storage_->PutEntry(entry));
  EXPECT_TRUE(CheckValidity());

  RemoveChild(key1, name2);
  EXPECT_FALSE(CheckValidity());  // Missing parent-child relationship.

  // Add back parent-child relationship between key1 and key2.
  PutChild(key1, name2, key2);
  EXPECT_TRUE(CheckValidity());

  // Add parent-child relationship between key2 and key3.
  PutChild(key2, name3, key3);
  EXPECT_FALSE(CheckValidity());  // key3 is not stored in the storage.

  // Put entry with key3 under key2.
  entry.set_resource_id(key3);
  entry.set_parent_resource_id(key2);
  entry.set_base_name(name3);
  EXPECT_TRUE(storage_->PutEntry(entry));
  EXPECT_TRUE(CheckValidity());

  // Parent-child relationship with wrong name.
  RemoveChild(key2, name3);
  EXPECT_FALSE(CheckValidity());
  PutChild(key2, name2, key3);
  EXPECT_FALSE(CheckValidity());

  // Fix up the relationship between key2 and key3.
  RemoveChild(key2, name2);
  EXPECT_FALSE(CheckValidity());
  PutChild(key2, name3, key3);
  EXPECT_TRUE(CheckValidity());

  // Add some cache entries.
  FileCacheEntry cache_entry;
  EXPECT_TRUE(storage_->PutCacheEntry(key1, cache_entry));
  EXPECT_TRUE(storage_->PutCacheEntry(key2, cache_entry));

  // Remove key2.
  RemoveChild(key1, name2);
  EXPECT_FALSE(CheckValidity());
  EXPECT_TRUE(storage_->RemoveEntry(key2));
  EXPECT_FALSE(CheckValidity());

  // Remove key3.
  RemoveChild(key2, name3);
  EXPECT_FALSE(CheckValidity());
  EXPECT_TRUE(storage_->RemoveEntry(key3));
  EXPECT_TRUE(CheckValidity());

  // Remove key1.
  EXPECT_TRUE(storage_->RemoveEntry(key1));
  EXPECT_TRUE(CheckValidity());
}

}  // namespace internal
}  // namespace drive
