// Copyright (c) 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef NET_QUIC_CRYPTO_P256_KEY_EXCHANGE_H_
#define NET_QUIC_CRYPTO_P256_KEY_EXCHANGE_H_

#include <string>

#include "base/memory/scoped_ptr.h"
#include "base/strings/string_piece.h"
#include "net/base/net_export.h"
#include "net/quic/crypto/key_exchange.h"

#if defined(USE_OPENSSL)
#include "crypto/openssl_util.h"
// Forward declaration for openssl/*.h
typedef struct ec_key_st EC_KEY;
extern "C" void EC_KEY_free(EC_KEY* key);
#else
#include "crypto/ec_private_key.h"
#include "crypto/scoped_nss_types.h"
#endif

namespace net {

// P256KeyExchange implements a KeyExchange using elliptic-curve
// Diffie-Hellman on NIST P-256.
class NET_EXPORT_PRIVATE P256KeyExchange : public KeyExchange {
 public:
  virtual ~P256KeyExchange();

  // New creates a new key exchange object from a private key. If
  // |private_key| is invalid, NULL is returned.
  static P256KeyExchange* New(base::StringPiece private_key);

  // |NewPrivateKey| returns a private key, suitable for passing to |New|.
  // If |NewPrivateKey| can't generate a private key, it returns an empty
  // string.
  static std::string NewPrivateKey();

  // KeyExchange interface.
  virtual bool CalculateSharedKey(const base::StringPiece& peer_public_value,
                                  std::string* shared_key) const OVERRIDE;
  virtual base::StringPiece public_value() const OVERRIDE;
  virtual CryptoTag tag() const OVERRIDE;

 private:
  enum {
    // A P-256 field element consists of 32 bytes.
    kP256FieldBytes = 32,
    // A P-256 point in uncompressed form consists of 0x04 (to denote
    // that the point is uncompressed) followed by two, 32-byte field
    // elements.
    kUncompressedP256PointBytes = 1 + 2 * kP256FieldBytes,
    // The first byte in an uncompressed P-256 point.
    kUncompressedECPointForm = 0x04,
  };

#if defined(USE_OPENSSL)
  // P256KeyExchange takes ownership of |private_key|, and expects
  // |public_key| consists of |kUncompressedP256PointBytes| bytes.
  P256KeyExchange(EC_KEY* private_key, const uint8* public_key);

  crypto::ScopedOpenSSL<EC_KEY, EC_KEY_free> private_key_;
#else
  // P256KeyExchange takes ownership of |key_pair|, and expects
  // |public_key| consists of |kUncompressedP256PointBytes| bytes.
  P256KeyExchange(crypto::ECPrivateKey* key_pair, const uint8* public_key);

  scoped_ptr<crypto::ECPrivateKey> key_pair_;
#endif
  // The public key stored as an uncompressed P-256 point.
  uint8 public_key_[kUncompressedP256PointBytes];
};

}  // namespace net
#endif  // NET_QUIC_CRYPTO_P256_KEY_EXCHANGE_H_

