/* Copyright (c) 2012 The Chromium Authors. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be
 * found in the LICENSE file.
 */

/** @file hello_world.c
 * This example, is a modified version of hello world.  It will start a second
 * thread and cause that thread to crash via a NULL dereference.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ppapi/c/pp_errors.h"
#include "ppapi/c/pp_module.h"
#include "ppapi/c/pp_var.h"
#include "ppapi/c/ppb.h"
#include "ppapi/c/ppb_core.h"
#include "ppapi/c/ppb_instance.h"
#include "ppapi/c/ppb_messaging.h"
#include "ppapi/c/ppb_var.h"
#include "ppapi/c/ppp.h"
#include "ppapi/c/ppp_instance.h"
#include "ppapi/c/ppp_messaging.h"

#include <pthread.h>

#include "error_handling/error_handling.h"

PPB_Messaging* ppb_messaging_interface = NULL;
PPB_Var* ppb_var_interface = NULL;
PPB_Core* ppb_core_interface = NULL;

pthread_t g_NexeThread;
pthread_t g_PPAPIThread;
PP_Instance g_Instance;

volatile int g_CrashTime = 0;

void PostMessage(const char* str);

void layer5(int x, int y) {
  if (g_CrashTime) {
    *(volatile int*)x = y;
  }
}

void layer4(int x) { layer5(x, 1); }

void layer3(int a, int b, int c) { layer4(a + b + c); }

void layer2(int i, int j) { layer3(i, j, 7); }

void layer1(int s, int t) {
  int* junk = (int*)alloca(sizeof(int) * 1234);
  junk[0] = s + 5;
  layer2(junk[0], t + 1);
}

void* NexeMain(void* data) {
  PostMessage("Running Boom thread.");
  while (1) {
    layer1(2, 9);
  }
  return NULL;
}

/**
 * Creates new string PP_Var from C string. The resulting object will be a
 * refcounted string object. It will be AddRef()ed for the caller. When the
 * caller is done with it, it should be Release()d.
 * @param[in] str C string to be converted to PP_Var
 * @return PP_Var containing string.
 */
static struct PP_Var CStrToVar(const char* str) {
  if (ppb_var_interface != NULL) {
    return ppb_var_interface->VarFromUtf8(str, strlen(str));
  }
  return PP_MakeUndefined();
}

static void PostCompletionCallback(void* user_data, int32_t result) {
  const char* str = (const char*)user_data;
  ppb_messaging_interface->PostMessage(g_Instance, CStrToVar(str));
  free(user_data);
}

void PostMessage(const char* str) {
  struct PP_CompletionCallback cb;

  if (NULL == str)
    return;
  if (NULL == ppb_messaging_interface)
    return;
  if (0 == g_Instance)
    return;

  if (strncmp(str, "ERR:", 4)) {
    fprintf(stderr, "%s\n", str);
    fflush(stderr);
  } else {
    fprintf(stdout, "%s\n", str);
    fflush(stdout);
  }

  /* If on Main Pepper thread, then call interface directly. */
  if (pthread_self() == g_PPAPIThread) {
    ppb_messaging_interface->PostMessage(g_Instance, CStrToVar(str));
    return;
  }

  /* Otherwise use call on main thread. */
  cb = PP_MakeCompletionCallback(PostCompletionCallback, strdup(str));
  ppb_core_interface->CallOnMainThread(0, cb, 0);
}

void DumpJson(const char* json) {
  char* out = (char*)malloc(strlen(json) + 5);
  strcpy(out, "TRC: ");
  strcat(out, json);

  PostMessage(out);
  free(out);
}
/**
 * Called when the NaCl module is instantiated on the web page. The identifier
 * of the new instance will be passed in as the first argument (this value is
 * generated by the browser and is an opaque handle).  This is called for each
 * instantiation of the NaCl module, which is each time the <embed> tag for
 * this module is encountered.
 *
 * If this function reports a failure (by returning @a PP_FALSE), the NaCl
 * module will be deleted and DidDestroy will be called.
 * @param[in] instance The identifier of the new instance representing this
 *     NaCl module.
 * @param[in] argc The number of arguments contained in @a argn and @a argv.
 * @param[in] argn An array of argument names.  These argument names are
 *     supplied in the <embed> tag, for example:
 *       <embed id="nacl_module" dimensions="2">
 *     will produce two arguments, one named "id" and one named "dimensions".
 * @param[in] argv An array of argument values.  These are the values of the
 *     arguments listed in the <embed> tag.  In the above example, there will
 *     be two elements in this array, "nacl_module" and "2".  The indices of
 *     these values match the indices of the corresponding names in @a argn.
 * @return @a PP_TRUE on success.
 */
static PP_Bool Instance_DidCreate(PP_Instance instance,
                                  uint32_t argc,
                                  const char* argn[],
                                  const char* argv[]) {
  g_Instance = instance;
  g_PPAPIThread = pthread_self();

  PostMessage("LOG: DidCreate");

  /* Request exception callbacks with JSON. */
  EHRequestExceptionsJson(DumpJson);

  /* Report back if the request was honored. */
  if (!EHHanderInstalled()) {
    PostMessage("LOG: Stack traces not available, so don't expect them.\n");
  } else {
    PostMessage("LOG: Stack traces are on.");
  }
  pthread_create(&g_NexeThread, NULL, NexeMain, NULL);
  return PP_TRUE;
}

/**
 * Called when the NaCl module is destroyed. This will always be called,
 * even if DidCreate returned failure. This routine should deallocate any data
 * associated with the instance.
 * @param[in] instance The identifier of the instance representing this NaCl
 *     module.
 */
static void Instance_DidDestroy(PP_Instance instance) {}

/**
 * Called when the position, the size, or the clip rect of the element in the
 * browser that corresponds to this NaCl module has changed.
 * @param[in] instance The identifier of the instance representing this NaCl
 *     module.
 * @param[in] position The location on the page of this NaCl module. This is
 *     relative to the top left corner of the viewport, which changes as the
 *     page is scrolled.
 * @param[in] clip The visible region of the NaCl module. This is relative to
 *     the top left of the plugin's coordinate system (not the page).  If the
 *     plugin is invisible, @a clip will be (0, 0, 0, 0).
 */
static void Instance_DidChangeView(PP_Instance instance,
                                   PP_Resource view_resource) {}

/**
 * Notification that the given NaCl module has gained or lost focus.
 * Having focus means that keyboard events will be sent to the NaCl module
 * represented by @a instance. A NaCl module's default condition is that it
 * will not have focus.
 *
 * Note: clicks on NaCl modules will give focus only if you handle the
 * click event. You signal if you handled it by returning @a true from
 * HandleInputEvent. Otherwise the browser will bubble the event and give
 * focus to the element on the page that actually did end up consuming it.
 * If you're not getting focus, check to make sure you're returning true from
 * the mouse click in HandleInputEvent.
 * @param[in] instance The identifier of the instance representing this NaCl
 *     module.
 * @param[in] has_focus Indicates whether this NaCl module gained or lost
 *     event focus.
 */
static void Instance_DidChangeFocus(PP_Instance instance, PP_Bool has_focus) {}

/**
 * Handler that gets called after a full-frame module is instantiated based on
 * registered MIME types.  This function is not called on NaCl modules.  This
 * function is essentially a place-holder for the required function pointer in
 * the PPP_Instance structure.
 * @param[in] instance The identifier of the instance representing this NaCl
 *     module.
 * @param[in] url_loader A PP_Resource an open PPB_URLLoader instance.
 * @return PP_FALSE.
 */
static PP_Bool Instance_HandleDocumentLoad(PP_Instance instance,
                                           PP_Resource url_loader) {
  /* NaCl modules do not need to handle the document load function. */
  return PP_FALSE;
}

/**
 * Handles message from JavaScript.
 *
 * Any message from JS is a request to cause the main thread to crash.
 */
static void Messaging_HandleMessage(PP_Instance instance,
                                    struct PP_Var message) {
  PostMessage("LOG: Got BOOM");
  g_CrashTime = 1;
}

/**
 * Entry points for the module.
 * Initialize needed interfaces: PPB_Core, PPB_Messaging and PPB_Var.
 * @param[in] a_module_id module ID
 * @param[in] get_browser pointer to PPB_GetInterface
 * @return PP_OK on success, any other value on failure.
 */
PP_EXPORT int32_t PPP_InitializeModule(PP_Module a_module_id,
                                       PPB_GetInterface get_browser) {
  ppb_messaging_interface =
      (PPB_Messaging*)(get_browser(PPB_MESSAGING_INTERFACE));
  ppb_var_interface = (PPB_Var*)(get_browser(PPB_VAR_INTERFACE));
  ppb_core_interface = (PPB_Core*)(get_browser(PPB_CORE_INTERFACE));
  return PP_OK;
}

/**
 * Returns an interface pointer for the interface of the given name, or NULL
 * if the interface is not supported.
 * @param[in] interface_name name of the interface
 * @return pointer to the interface
 */
PP_EXPORT const void* PPP_GetInterface(const char* interface_name) {
  if (strcmp(interface_name, PPP_INSTANCE_INTERFACE) == 0) {
    static PPP_Instance instance_interface = {
      &Instance_DidCreate,
      &Instance_DidDestroy,
      &Instance_DidChangeView,
      &Instance_DidChangeFocus,
      &Instance_HandleDocumentLoad,
    };
    return &instance_interface;
  }
  if (strcmp(interface_name, PPP_MESSAGING_INTERFACE) == 0) {
    static PPP_Messaging messaging_interface = {
      &Messaging_HandleMessage,
    };
    return &messaging_interface;
  }
  return NULL;
}

/**
 * Called before the plugin module is unloaded.
 */
PP_EXPORT void PPP_ShutdownModule() {}
