// Copyright 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#import "chrome/browser/ui/cocoa/tab_contents/overlayable_contents_controller.h"

#include "base/mac/bundle_locations.h"
#include "chrome/browser/ui/cocoa/browser_window_controller.h"
#include "chrome/browser/ui/cocoa/tab_contents/instant_overlay_controller_mac.h"
#include "chrome/browser/ui/cocoa/tab_contents/overlay_separator_view.h"
#include "content/public/browser/web_contents.h"
#include "content/public/browser/web_contents_view.h"

@interface OverlayableContentsController()
- (void)viewDidResize:(NSNotification*)note;
- (void)layoutViews;
- (CGFloat)overlayHeightInPixels;
- (BOOL)shouldShowTopSeparator;
@end

@implementation OverlayableContentsController

@synthesize drawDropShadow = drawDropShadow_;
@synthesize activeContainerOffset = activeContainerOffset_;
@synthesize overlayContentsOffset = overlayContentsOffset_;

- (id)initWithBrowser:(Browser*)browser
     windowController:(BrowserWindowController*)windowController {
  if ((self = [super init])) {
    windowController_ = windowController;
    scoped_nsobject<NSView> view([[NSView alloc] initWithFrame:NSZeroRect]);
    [view setAutoresizingMask:NSViewHeightSizable | NSViewWidthSizable];
    [view setAutoresizesSubviews:NO];
    [[NSNotificationCenter defaultCenter]
        addObserver:self
           selector:@selector(viewDidResize:)
               name:NSViewFrameDidChangeNotification
             object:view];
    [self setView:view];

    activeContainer_.reset([[NSView alloc] initWithFrame:NSZeroRect]);
    [view addSubview:activeContainer_];

    instantOverlayController_.reset(
        new InstantOverlayControllerMac(browser, windowController, self));
    topSeparatorView_.reset(
        [[OverlayTopSeparatorView alloc] initWithFrame:NSZeroRect]);
    [[self view] addSubview:topSeparatorView_];
  }
  return self;
}

- (void)dealloc {
  [[NSNotificationCenter defaultCenter] removeObserver:self];
  [super dealloc];
}

- (void)setOverlay:(content::WebContents*)overlay
            height:(CGFloat)height
       heightUnits:(InstantSizeUnits)heightUnits
    drawDropShadow:(BOOL)drawDropShadow {
  // If drawing drop shadow, clip the bottom 1-px-thick separator out of
  // overlay.
  // TODO(sail): remove this when GWS gives chrome the height without the
  // separator.
  if (drawDropShadow && heightUnits != INSTANT_SIZE_PERCENT)
    --height;

  if (overlayContents_ == overlay &&
      overlayHeight_ == height &&
      overlayHeightUnits_ == heightUnits &&
      drawDropShadow_ == drawDropShadow) {
    return;
  }

  // Remove any old overlay contents before showing the new one.
  if (overlayContents_) {
    if (overlayContents_ != overlay)
      overlayContents_->WasHidden();
    [overlayContents_->GetView()->GetNativeView() removeFromSuperview];
  }

  overlayContents_ = overlay;
  overlayHeight_ = height;
  overlayHeightUnits_ = heightUnits;
  drawDropShadow_ = drawDropShadow;

  // Add the overlay contents.
  if (overlayContents_) {
    [[[self view] window] disableScreenUpdatesUntilFlush];
    [[self view] addSubview:overlayContents_->GetView()->GetNativeView()];
  }

  if (drawDropShadow_) {
    if (!dropShadowView_) {
      dropShadowView_.reset(
          [[OverlayBottomSeparatorView alloc] initWithFrame:NSZeroRect]);
      [[self view] addSubview:dropShadowView_];
    }
  } else {
    [dropShadowView_ removeFromSuperview];
    dropShadowView_.reset();
  }

  [self layoutViews];

  if (overlayContents_)
    overlayContents_->WasShown();
}

- (void)onActivateTabWithContents:(content::WebContents*)contents {
  if (overlayContents_ == contents) {
    if (overlayContents_) {
      [overlayContents_->GetView()->GetNativeView() removeFromSuperview];
      overlayContents_ = NULL;
    }
    [self setOverlay:NULL
              height:0
         heightUnits:INSTANT_SIZE_PIXELS
      drawDropShadow:NO];
  }
}

- (BOOL)isShowingOverlay {
  return overlayContents_ != nil;
}

- (InstantOverlayControllerMac*)instantOverlayController {
  return instantOverlayController_.get();
}

- (void)activeContentsCompositingIOSurfaceCreated {
  if (!overlayContents_)
    return;

  // If the active tab becomes composited the the overlay will no longer be
  // visible. Workaround this by re-adding the overlay to the view hierarchy.
  // See http://crbug.com/222122
  [overlayContents_->GetView()->GetNativeView() removeFromSuperview];
  [[self view] addSubview:overlayContents_->GetView()->GetNativeView()];
}

- (NSView*)activeContainer {
  return activeContainer_.get();
}

- (NSView*)dropShadowView {
  return dropShadowView_.get();
}

- (void)setActiveContainerOffset:(CGFloat)activeContainerOffset {
  if (activeContainerOffset_ == activeContainerOffset)
    return;

  activeContainerOffset_ = activeContainerOffset;
  [self layoutViews];
}

- (void)setOverlayContentsOffset:(CGFloat)overlayContentsOffset {
  if (overlayContentsOffset_ == overlayContentsOffset)
    return;

  overlayContentsOffset_ = overlayContentsOffset;
  [self layoutViews];
}

- (void)viewDidResize:(NSNotification*)note {
  [self layoutViews];
}

- (void)layoutViews {
  NSRect bounds = [[self view] bounds];

  // Layout the separator at the top of the view.
  NSRect separatorRect = bounds;
  if ([self shouldShowTopSeparator])
    separatorRect.size.height = [OverlayTopSeparatorView preferredHeight];
  else
    separatorRect.size.height = 0;
  separatorRect.origin.y = NSMaxY(bounds) - NSHeight(separatorRect);
  [topSeparatorView_ setFrame:separatorRect];

  // Layout the overlay.
  if (overlayContents_) {
    NSRect overlayFrame = bounds;
    overlayFrame.size.height = [self overlayHeightInPixels];
    overlayFrame.origin.y = NSMinY([topSeparatorView_ frame]) -
                            NSHeight(overlayFrame) - overlayContentsOffset_;
    [overlayContents_->GetView()->GetNativeView() setFrame:overlayFrame];

    if (dropShadowView_) {
      NSRect dropShadowFrame = bounds;
      dropShadowFrame.size.height =
          [OverlayBottomSeparatorView preferredHeight];
      dropShadowFrame.origin.y =
          NSMinY(overlayFrame) - NSHeight(dropShadowFrame);
      [dropShadowView_ setFrame:dropShadowFrame];
    }
  }

  // Layout the active tab contents.
  NSRect activeFrame = bounds;
  if (activeContainerOffset_)
    activeFrame.size.height -= activeContainerOffset_;
  else
    activeFrame.size.height -= NSHeight([topSeparatorView_ frame]);
  if (!NSEqualRects(activeFrame, [activeContainer_ frame])) {
    [[activeContainer_ window] disableScreenUpdatesUntilFlush];
    [activeContainer_ setFrame:activeFrame];
  }
}

- (CGFloat)overlayHeightInPixels {
  CGFloat height = NSHeight([[self view] bounds]) -
                   NSHeight([topSeparatorView_ frame]) - overlayContentsOffset_;
  switch (overlayHeightUnits_) {
    case INSTANT_SIZE_PERCENT:
      return std::min(height, (height * overlayHeight_) / 100);
    case INSTANT_SIZE_PIXELS:
      return std::min(height, overlayHeight_);
  }
}

- (BOOL)shouldShowTopSeparator {
  // In presentation mode tab contents are flush with the top of the screen
  // so there's no need for a separator.
  if ([windowController_ inPresentationMode])
    return NO;

  if (![windowController_ hasToolbar])
    return NO;

  // Show a separator is the overlay or the tab contents will be shown right
  // next to the omnibox.
  return activeContainerOffset_ == 0 || overlayContents_;
}

@end
