// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/first_run/first_run.h"

#include "base/file_util.h"
#include "base/files/file_path.h"
#include "base/message_loop.h"
#include "base/path_service.h"
#include "base/prefs/pref_service.h"
#include "base/utf_string_conversions.h"
#include "chrome/browser/browser_process.h"
#include "chrome/browser/first_run/first_run_dialog.h"
#include "chrome/browser/first_run/first_run_internal.h"
#include "chrome/browser/importer/importer_host.h"
#include "chrome/browser/importer/importer_list.h"
#include "chrome/browser/profiles/profile_manager.h"
#include "chrome/common/chrome_constants.h"
#include "chrome/common/chrome_paths.h"
#include "chrome/common/pref_names.h"
#include "chrome/common/startup_metric_utils.h"
#include "chrome/installer/util/google_update_settings.h"
#include "chrome/installer/util/master_preferences.h"
#include "chrome/installer/util/master_preferences_constants.h"

namespace first_run {
namespace internal {

void DoPostImportPlatformSpecificTasks(Profile* profile) {
#if !defined(OS_CHROMEOS)
  // Aura needs a views implementation of the first run dialog for Linux.
  // http://crbug.com/234637
#if !defined(USE_AURA)
  base::FilePath local_state_path;
  PathService::Get(chrome::FILE_LOCAL_STATE, &local_state_path);
  bool local_state_file_exists = file_util::PathExists(local_state_path);
  // Launch the first run dialog only for certain builds, and only if the user
  // has not already set preferences.
  if (internal::IsOrganicFirstRun() && !local_state_file_exists) {
    if (ShowFirstRunDialog(profile))
      startup_metric_utils::SetNonBrowserUIDisplayed();
  }
#endif

  // If stats reporting was turned on by the first run dialog then toggle
  // the pref (on Windows, the download is tagged with enable/disable stats so
  // this is POSIX-specific).
  if (GoogleUpdateSettings::GetCollectStatsConsent()) {
    g_browser_process->local_state()->SetBoolean(
        prefs::kMetricsReportingEnabled, true);
  }
#endif
}

bool GetFirstRunSentinelFilePath(base::FilePath* path) {
  base::FilePath first_run_sentinel;

  if (!PathService::Get(chrome::DIR_USER_DATA, &first_run_sentinel))
    return false;

  *path = first_run_sentinel.Append(chrome::kFirstRunSentinel);
  return true;
}

bool ImportSettings(Profile* profile,
                    ImporterHost* importer_host,
                    scoped_refptr<ImporterList> importer_list,
                    int items_to_import) {
  const importer::SourceProfile& source_profile =
      importer_list->GetSourceProfileAt(0);

  // Ensure that importers aren't requested to import items that they do not
  // support. If there is no overlap, skip.
  items_to_import &= source_profile.services_supported;
  if (items_to_import == 0)
    return true;

  ImportEndedObserver observer;
  importer_host->SetObserver(&observer);
  importer_host->StartImportSettings(source_profile,
                                     profile,
                                     items_to_import,
                                     new ProfileWriter(profile));
  // If the import process has not errored out, block on it.
  if (!observer.ended()) {
    observer.set_should_quit_message_loop();
    MessageLoop::current()->Run();
  }

  // Unfortunately there's no success/fail signal in ImporterHost.
  return true;
}

void SetImportPreferencesAndLaunchImport(
    MasterPrefs* out_prefs,
    installer::MasterPreferences* install_prefs) {
  std::string import_bookmarks_path;
  install_prefs->GetString(
      installer::master_preferences::kDistroImportBookmarksFromFilePref,
      &import_bookmarks_path);
  if (!import_bookmarks_path.empty()) {
    // There are bookmarks to import from a file.
    base::FilePath path = base::FilePath::FromWStringHack(UTF8ToWide(
        import_bookmarks_path));
    if (!ImportBookmarks(path)) {
      LOG(WARNING) << "silent bookmark import failed";
    }
  }
}

bool ShowPostInstallEULAIfNeeded(installer::MasterPreferences* install_prefs) {
  // The EULA is only handled on Windows.
  return true;
}

}  // namespace internal
}  // namespace first_run

namespace first_run {

// TODO(port): Import switches need to be ported to both Mac and Linux. Not all
// import switches here are implemented for Linux. None are implemented for Mac
// (as this function will not be called on Mac).
int ImportNow(Profile* profile, const CommandLine& cmdline) {
  return internal::ImportBookmarkFromFileIfNeeded(profile, cmdline);
}

}  // namespace first_run
