/*
 * Copyright (C) 2009 Google Inc. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 
 *     * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above
 * copyright notice, this list of conditions and the following disclaimer
 * in the documentation and/or other materials provided with the
 * distribution.
 *     * Neither the name of Google Inc. nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "V8HTMLInputElement.h"

#include "core/html/HTMLInputElement.h"

#include "bindings/v8/V8Binding.h"

namespace WebCore {

v8::Handle<v8::Value> V8HTMLInputElement::selectionStartAttrGetterCustom(v8::Local<v8::String> name, const v8::AccessorInfo& info)
{
    v8::Handle<v8::Object> holder = info.Holder();
    HTMLInputElement* imp = V8HTMLInputElement::toNative(holder);

    if (!imp->canHaveSelection())
        return throwTypeError("Accessing selectionStart on an input element that cannot have a selection.", info.GetIsolate());

    int v = imp->selectionStart();
    return v8Integer(v, info.GetIsolate());
}

void V8HTMLInputElement::selectionStartAttrSetterCustom(v8::Local<v8::String> name, v8::Local<v8::Value> value, const v8::AccessorInfo& info)
{
    v8::Handle<v8::Object> holder = info.Holder();
    HTMLInputElement* imp = V8HTMLInputElement::toNative(holder);

    if (!imp->canHaveSelection()) {
        throwTypeError("Accessing selectionStart on an input element that cannot have a selection.", info.GetIsolate());
        return;
    }
    imp->setSelectionStart(value->Int32Value());
}

v8::Handle<v8::Value> V8HTMLInputElement::selectionEndAttrGetterCustom(v8::Local<v8::String> name, const v8::AccessorInfo& info)
{
    v8::Handle<v8::Object> holder = info.Holder();
    HTMLInputElement* imp = V8HTMLInputElement::toNative(holder);

    if (!imp->canHaveSelection())
        return throwTypeError("Accessing selectionEnd on an input element that cannot have a selection.", info.GetIsolate());

    int v = imp->selectionEnd();
    return v8Integer(v, info.GetIsolate());
}

void V8HTMLInputElement::selectionEndAttrSetterCustom(v8::Local<v8::String> name, v8::Local<v8::Value> value, const v8::AccessorInfo& info)
{
    v8::Handle<v8::Object> holder = info.Holder();
    HTMLInputElement* imp = V8HTMLInputElement::toNative(holder);

    if (!imp->canHaveSelection()) {
        throwTypeError("Accessing selectionEnd on an input element that cannot have a selection.", info.GetIsolate());
        return;
    }

    imp->setSelectionEnd(value->Int32Value());
}

v8::Handle<v8::Value> V8HTMLInputElement::selectionDirectionAttrGetterCustom(v8::Local<v8::String> name, const v8::AccessorInfo& info)
{
    v8::Handle<v8::Object> holder = info.Holder();
    HTMLInputElement* imp = V8HTMLInputElement::toNative(holder);

    if (!imp->canHaveSelection())
        return throwTypeError("Accessing selectionDirection on an input element that cannot have a selection.", info.GetIsolate());

    return v8String(imp->selectionDirection(), info.GetIsolate());
}

void V8HTMLInputElement::selectionDirectionAttrSetterCustom(v8::Local<v8::String> name, v8::Local<v8::Value> value, const v8::AccessorInfo& info)
{
    v8::Handle<v8::Object> holder = info.Holder();
    HTMLInputElement* imp = V8HTMLInputElement::toNative(holder);

    if (!imp->canHaveSelection()) {
        throwTypeError("Accessing selectionDirection on an input element that cannot have a selection.", info.GetIsolate());
        return;
    }

    imp->setSelectionDirection(toWebCoreString(value));
}

v8::Handle<v8::Value> V8HTMLInputElement::setSelectionRangeMethodCustom(const v8::Arguments& args)
{
    v8::Handle<v8::Object> holder = args.Holder();
    HTMLInputElement* imp = V8HTMLInputElement::toNative(holder);

    if (!imp->canHaveSelection())
        return throwTypeError("Calling setSelectionRange on an input element that cannot have a selection.", args.GetIsolate());

    int start = args[0]->Int32Value();
    int end = args[1]->Int32Value();
    String direction = toWebCoreString(args[2]);

    imp->setSelectionRange(start, end, direction);
    return v8::Undefined();
}

} // namespace WebCore
