// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/search/instant_test_utils.h"

#include "base/command_line.h"
#include "base/prefs/pref_service.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/search_engines/template_url_service.h"
#include "chrome/browser/search_engines/template_url_service_factory.h"
#include "chrome/browser/ui/omnibox/omnibox_view.h"
#include "chrome/browser/ui/search/instant_ntp.h"
#include "chrome/browser/ui/search/instant_overlay.h"
#include "chrome/common/chrome_notification_types.h"
#include "chrome/common/chrome_switches.h"
#include "chrome/common/pref_names.h"
#include "chrome/test/base/interactive_test_utils.h"
#include "chrome/test/base/ui_test_utils.h"
#include "content/public/browser/notification_service.h"
#include "content/public/browser/render_process_host.h"
#include "content/public/browser/web_contents.h"
#include "content/public/common/result_codes.h"
#include "content/public/test/browser_test_utils.h"

namespace {

std::string WrapScript(const std::string& script) {
  return "domAutomationController.send(" + script + ")";
}

}  // namespace

// InstantTestModelObserver --------------------------------------------------

InstantTestModelObserver::InstantTestModelObserver(
    InstantOverlayModel* model,
    SearchMode::Type expected_mode_type)
    : model_(model),
      expected_mode_type_(expected_mode_type),
      observed_mode_type_(static_cast<SearchMode::Type>(-1)) {
  model_->AddObserver(this);
}

InstantTestModelObserver::~InstantTestModelObserver() {
  model_->RemoveObserver(this);
}

SearchMode::Type InstantTestModelObserver::WaitForExpectedOverlayState() {
  run_loop_.Run();
  return observed_mode_type_;
}

void InstantTestModelObserver::OverlayStateChanged(
    const InstantOverlayModel& model) {
  observed_mode_type_ = model.mode().mode;
  run_loop_.Quit();
}

// InstantTestBase -----------------------------------------------------------

void InstantTestBase::SetupInstant(Browser* browser) {
  browser_ = browser;
  TemplateURLService* service =
      TemplateURLServiceFactory::GetForProfile(browser_->profile());
  ui_test_utils::WaitForTemplateURLServiceToLoad(service);

  TemplateURLData data;
  // Necessary to use exact URL for both the main URL and the alternate URL for
  // search term extraction to work in InstantExtended.
  data.SetURL(instant_url_.spec() + "q={searchTerms}");
  data.instant_url = instant_url_.spec();
  data.alternate_urls.push_back(instant_url_.spec() + "#q={searchTerms}");
  data.search_terms_replacement_key = "strk";

  TemplateURL* template_url = new TemplateURL(browser_->profile(), data);
  service->Add(template_url);  // Takes ownership of |template_url|.
  service->SetDefaultSearchProvider(template_url);

  browser_->profile()->GetPrefs()->SetBoolean(prefs::kInstantEnabled, true);

  // TODO(shishir): Fix this ugly hack.
  instant()->SetInstantEnabled(false, true);
  instant()->SetInstantEnabled(true, false);
}

void InstantTestBase::SetInstantURL(const std::string& url) {
  TemplateURLService* service =
      TemplateURLServiceFactory::GetForProfile(browser_->profile());
  ui_test_utils::WaitForTemplateURLServiceToLoad(service);

  TemplateURLData data;
  data.SetURL(url);
  data.instant_url = url;

  TemplateURL* template_url = new TemplateURL(browser_->profile(), data);
  service->Add(template_url);  // Takes ownership of |template_url|.
  service->SetDefaultSearchProvider(template_url);
}

void InstantTestBase::Init(const GURL& instant_url) {
  instant_url_ = instant_url;
}

void InstantTestBase::KillInstantRenderView() {
  base::KillProcess(
      instant()->GetOverlayContents()->GetRenderProcessHost()->GetHandle(),
      content::RESULT_CODE_KILLED,
      false);
}

void InstantTestBase::FocusOmnibox() {
  // If the omnibox already has focus, just notify Instant.
  if (omnibox()->model()->has_focus()) {
    instant()->OmniboxFocusChanged(OMNIBOX_FOCUS_VISIBLE,
                                   OMNIBOX_FOCUS_CHANGE_EXPLICIT, NULL);
  } else {
    browser_->window()->GetLocationBar()->FocusLocation(false);
  }
}

void InstantTestBase::FocusOmniboxAndWaitForInstantOverlaySupport() {
  content::WindowedNotificationObserver observer(
      chrome::NOTIFICATION_INSTANT_OVERLAY_SUPPORT_DETERMINED,
      content::NotificationService::AllSources());
  FocusOmnibox();
  if (!instant()->overlay() ||
      !instant()->overlay()->instant_support_determined()) {
    observer.Wait();
  }
}

void InstantTestBase::FocusOmniboxAndWaitForInstantOverlayAndNTPSupport() {
  content::WindowedNotificationObserver ntp_observer(
      chrome::NOTIFICATION_INSTANT_NTP_SUPPORT_DETERMINED,
      content::NotificationService::AllSources());
  content::WindowedNotificationObserver overlay_observer(
      chrome::NOTIFICATION_INSTANT_OVERLAY_SUPPORT_DETERMINED,
      content::NotificationService::AllSources());
  FocusOmnibox();
  if (!instant()->ntp() ||
      !instant()->ntp()->instant_support_determined()) {
    ntp_observer.Wait();
  }
  if (!instant()->overlay() ||
      !instant()->overlay()->instant_support_determined()) {
    overlay_observer.Wait();
  }
}

void InstantTestBase::SetOmniboxText(const std::string& text) {
  FocusOmnibox();
  omnibox()->SetUserText(UTF8ToUTF16(text));
}

bool InstantTestBase::SetOmniboxTextAndWaitForOverlayToShow(
    const std::string& text) {
  // The order of events may be:
  //   { hide, show } or just { show } depending on the order things
  // flow in from GWS and Chrome's response to hiding the infobar and/or
  // bookmark bar.  Note, the GWS response is relevant because of the
  // Instant "MANUAL_*" tests.
  InstantTestModelObserver first_observer(
      instant()->model(), SearchMode::MODE_DEFAULT);
  SetOmniboxText(text);

  SearchMode::Type observed = first_observer.WaitForExpectedOverlayState();
  if (observed == SearchMode::MODE_DEFAULT) {
    InstantTestModelObserver second_observer(
        instant()->model(), SearchMode::MODE_SEARCH_SUGGESTIONS);
    observed = second_observer.WaitForExpectedOverlayState();
  }
  EXPECT_EQ(SearchMode::MODE_SEARCH_SUGGESTIONS, observed);
  return observed == SearchMode::MODE_SEARCH_SUGGESTIONS;
}

void InstantTestBase::SetOmniboxTextAndWaitForSuggestion(
    const std::string& text) {
  content::WindowedNotificationObserver observer(
      chrome::NOTIFICATION_INSTANT_SET_SUGGESTION,
      content::NotificationService::AllSources());
  SetOmniboxText(text);
  observer.Wait();
}

bool InstantTestBase::GetBoolFromJS(content::WebContents* contents,
                                    const std::string& script,
                                    bool* result) {
  return content::ExecuteScriptAndExtractBool(
      contents, WrapScript(script), result);
}

bool InstantTestBase::GetIntFromJS(content::WebContents* contents,
                                   const std::string& script,
                                   int* result) {
  return content::ExecuteScriptAndExtractInt(
      contents, WrapScript(script), result);
}

bool InstantTestBase::GetStringFromJS(content::WebContents* contents,
                                      const std::string& script,
                                      std::string* result) {
  return content::ExecuteScriptAndExtractString(
      contents, WrapScript(script), result);
}

bool InstantTestBase::ExecuteScript(const std::string& script) {
  return content::ExecuteScript(instant()->GetOverlayContents(), script);
}

bool InstantTestBase::CheckVisibilityIs(content::WebContents* contents,
                                        bool expected) {
  bool actual = !expected;  // Purposely start with a mis-match.
  // We can only use ASSERT_*() in a method that returns void, hence this
  // convoluted check.
  return GetBoolFromJS(contents, "!document.webkitHidden", &actual) &&
      actual == expected;
}

bool InstantTestBase::HasUserInputInProgress() {
  return omnibox()->model()->user_input_in_progress_;
}

bool InstantTestBase::HasTemporaryText() {
  return omnibox()->model()->has_temporary_text_;
}

bool InstantTestBase::LoadImage(content::RenderViewHost* rvh,
                                const std::string& image,
                                bool* loaded) {
  std::string js_chrome =
      "var img = document.createElement('img');"
      "img.onerror = function() { domAutomationController.send(false); };"
      "img.onload  = function() { domAutomationController.send(true); };"
      "img.src = '" + image + "';";
  return content::ExecuteScriptAndExtractBool(rvh, js_chrome, loaded);
}

string16 InstantTestBase::GetBlueText() {
  size_t start = 0, end = 0;
  omnibox()->GetSelectionBounds(&start, &end);
  if (start > end)
    std::swap(start, end);
  return omnibox()->GetText().substr(start, end - start);
}

string16 InstantTestBase::GetGrayText() {
  return omnibox()->GetInstantSuggestion();
}
