# -*- coding: utf-8 -*-
#
#    Copyright (C) 2009 Patrick Ulbrich <zulu99@gmx.net>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

import gtk
import gobject
from lib.common import *
from lib import resources
from lib.junkfilter import *

UI_FILE = resources.get_data_file("junkfilter_dialog.ui")
POPUPMENU_UI_FILE = resources.get_data_file("junkfilter_popupmenu.ui")

(
 ENABLED_COL,
 MAILFIELD_COL,
 COMPARETYPE_COL,
 STRING_COL,
 FILTER_COL
) = range(5)

class JunkFilterDialog:
	def __init__(self):
		self.filters_changed = False
		
		# Instanciate junkfilter manager.
		self.__filter_man = JunkFilterManager.get_default()
		
		# Build popup menu.
		self.builder2 = gtk.Builder()
		self.builder2.add_from_file(POPUPMENU_UI_FILE)
		self.builder2.set_translation_domain("cgmail")
		
		dict = {
			"on_top_activate"		: self.on_top_activate,
			"on_up_activate"		: self.on_up_activate,
			"on_down_activate"		: self.on_down_activate,
			"on_bottom_activate"	: self.on_bottom_activate,
			"on_remove_activate"	: self.on_remove_activate
		}
		self.builder2.connect_signals(dict)
		self.popupmenu = self.builder2.get_object("menu")
		
		# Build gui.
		self.builder = gtk.Builder()
		self.builder.add_from_file(UI_FILE)
		self.builder.set_translation_domain("cgmail")
		
		dict = {
			"on_add_button_clicked"				: self.on_add_button_clicked,
			"on_filters_treeview_button_press"	: self.on_filters_treeview_button_press,
			"on_string_entry_key_press"			: self.on_string_entry_key_press,
			"on_filters_treeview_key_press"		: self.on_filters_treeview_key_press
		}
		self.builder.connect_signals(dict)
		
		self.dialog = self.builder.get_object("dialog")
		self.filters_treeview = self.builder.get_object("filters_treeview")
		self.mailfield_box = self.builder.get_object("mailfield_box")
		self.comparetype_box = self.builder.get_object("comparetype_box")
		self.string_entry = self.builder.get_object("string_entry")
		self.add_button = self.builder.get_object("add_button")
		
		
		# Filters treeview.
		self.filters_store = gtk.ListStore(
						 gobject.TYPE_BOOLEAN,
						 gobject.TYPE_STRING,
						 gobject.TYPE_STRING,
						 gobject.TYPE_STRING,
						 gobject.TYPE_PYOBJECT)
		
		renderer = gtk.CellRendererToggle()
		renderer.connect("toggled", self.on_filter_set_active)
		self.enabled_column = gtk.TreeViewColumn( _("Enabled"),
			renderer, active = ENABLED_COL)
		self.filters_treeview.append_column(self.enabled_column)

		self.mailfield_column = gtk.TreeViewColumn( _("Field"),
			gtk.CellRendererText(), text = MAILFIELD_COL)
		self.filters_treeview.append_column(self.mailfield_column)

		self.comparetype_column = gtk.TreeViewColumn( _("Comparision"),
			gtk.CellRendererText(), text = COMPARETYPE_COL)
		self.filters_treeview.append_column(self.comparetype_column)
		
		self.string_column = gtk.TreeViewColumn( _("Value"),
			gtk.CellRendererText(), text = STRING_COL)
		self.filters_treeview.append_column(self.string_column)
		
		self.filters_treeview.set_model(self.filters_store)
		
		
		textcell = gtk.CellRendererText()
		# Mailfield combobox.
		self.mailfield_box.set_model(gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_INT))
		self.mailfield_box.pack_end(textcell, True)
		self.mailfield_box.add_attribute(textcell, 'text', 0)
		
		# Comparetype combobox.
		self.comparetype_box.set_model(gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_INT))
		self.comparetype_box.pack_end(textcell, True)
		self.comparetype_box.add_attribute(textcell, 'text', 0)
		
		
		# Fill in data.
		self.__fill_filters()
		self.__fill_mailfields()
		self.__fill_comparetypes()		
		
		
		# Additional signal handlers.
		# Make sure they are connected _after_ gui creation!
		
		# Connect signal handler for row reordering.
		self.filters_store.connect("row-inserted", self.on_row_inserted)
		
		self.run()
	
	def run(self):
		result = self.dialog.run()

		if self.filters_changed:
			self.__save_filters()

		self.dialog.destroy()
	
	def on_filter_set_active(self, cell, path):
		model = self.filters_store
		iter = model.get_iter((int(path),))
		enabled = model.get_value(iter, ENABLED_COL)
		f = model.get_value(iter, FILTER_COL)
		
		enabled = not enabled
		
		f.enabled = enabled
		model.set(iter, ENABLED_COL, enabled)
		
		self.filters_changed = True
	
	def on_add_button_clicked(self, butt):
		self.__add_new_filter()
	
	def on_top_activate(self, arg):
		selection = self.filters_treeview.get_selection()
		model, selected = selection.get_selected()		
		assert not selected is None
		model.move_after(selected, None)
		
		self.filters_changed = True
	
	def on_up_activate(self, arg):
		selection = self.filters_treeview.get_selection()
		model, selected = selection.get_selected()		
		assert not selected is None
		path = model.get_path(selected)
		pos = path[-1]
		if pos == 0:
			return

		prev_path = list(path)[:-1]
		prev_path.append(pos - 1)
		prev = model.get_iter(tuple(prev_path))
		
		model.swap(selected, prev)
		
		self.filters_changed = True
	
	def on_down_activate(self, arg):
		selection = self.filters_treeview.get_selection()
		model, selected = selection.get_selected()		
		assert not selected is None
		next = model.iter_next(selected)
		if next is None:
			return
				
		model.swap(selected, next)
		
		self.filters_changed = True
	
	def on_bottom_activate(self, arg):
		selection = self.filters_treeview.get_selection()
		model, selected = selection.get_selected()		
		assert not selected is None

		model.move_before(selected, None)
		
		self.filters_changed = True
	
	def on_row_inserted(self, model, path, iter):
		self.filters_changed = True
	
	def on_remove_activate(self, arg):
		self.__remove_filter()
	
	def on_filters_treeview_button_press(self, treeview, event):
		if event.button == 3:
			x = int(event.x)
			y = int(event.y)
			time = event.time
			pthinfo = treeview.get_path_at_pos(x, y)
			if pthinfo is not None:
				self.popupmenu.popup(None, None, None, event.button, time)
	
	def on_string_entry_key_press(self, widget, event):
		key_name = gtk.gdk.keyval_name(event.keyval)
		if key_name == "Return":
			self.__add_new_filter()
		
	def on_filters_treeview_key_press(self, widget, event):
		key_name = gtk.gdk.keyval_name(event.keyval)
		if key_name == "Delete":
			self.__remove_filter()
	
	def __fill_filters(self):
		filters = self.__filter_man.get_filters()
		
		if len(filters) == 0: return
		
		for f in filters:			
			self.__append_filter(f)
	
	def __fill_mailfields(self):
		model = self.mailfield_box.get_model()
		for k, v in MailField.names.iteritems():
			iter = model.append()
			model.set_value(iter, 0, v)
			model.set_value(iter, 1, k)

		self.mailfield_box.set_active(0)
	
	def __fill_comparetypes(self):
		model = self.comparetype_box.get_model()
		for k, v in CompareType.names.iteritems():
			iter = model.append()
			model.set_value(iter, 0, v)
			model.set_value(iter, 1, k)

		self.comparetype_box.set_active(0)
		
	def __add_new_filter(self):
		strn = unicode(self.string_entry.get_text().strip())

		if len(strn) == 0:
			self.string_entry.grab_focus()
			return
		
		model = self.mailfield_box.get_model()
		iter = self.mailfield_box.get_active_iter()
		mailfield = model.get_value(iter, 1)
		
		model = self.comparetype_box.get_model()
		iter = self.comparetype_box.get_active_iter()
		comparetype = model.get_value(iter, 1)
		
		f = JunkFilter(mailfield,
						comparetype,
						strn)
		
		self.__append_filter(f)
		
		self.string_entry.set_text("")
		
		self.filters_changed = True
	
	def __remove_filter(self):
		selection = self.filters_treeview.get_selection()
		model, path_list = selection.get_selected_rows()
		
		if len(path_list) == 0:
			return
		
		path = path_list[0]
		iter = model.get_iter(path)
		model.remove(iter)
		
		self.filters_changed = True
	
	def __append_filter(self, f):
		iter = self.filters_store.append()
		
		if f.enabled:
			self.filters_store.set_value(iter, ENABLED_COL, True)
		else:
			self.filters_store.set_value(iter, ENABLED_COL, False)
		
		self.filters_store.set_value(iter, MAILFIELD_COL, MailField.names[f.mail_field])
		self.filters_store.set_value(iter, COMPARETYPE_COL, CompareType.names[f.compare_type])
		self.filters_store.set_value(iter, STRING_COL, f.string)
		self.filters_store.set_value(iter, FILTER_COL, f)
	
	def __save_filters(self):
		self.__filter_man.clear_filters()
		
		for row in self.filters_store:
			f =  row[FILTER_COL]
			self.__filter_man.add_filter(f)
		
		self.__filter_man.save_filters()

