// Copyright (c) 2017 GeometryFactory
//
// This file is part of CGAL (www.cgal.org)
//
// $URL: https://github.com/CGAL/cgal/blob/releases/CGAL-5.0.1/Polyhedron_IO/include/CGAL/IO/PLY_writer.h $
// $Id: PLY_writer.h 52164b1 2019-10-19T15:34:59+02:00 Sébastien Loriot
// SPDX-License-Identifier: LGPL-3.0-or-later OR LicenseRef-Commercial
//
// Author(s)     : Simon Giraudot

#ifndef CGAL_IO_PLY_WRITER_H
#define CGAL_IO_PLY_WRITER_H

#include <CGAL/IO/PLY.h>

namespace CGAL{

  template <class Point_3, class Polygon_3>
  bool
  write_PLY(std::ostream& out,
            const std::vector< Point_3 >& points,
            const std::vector< Polygon_3 >& polygons,
            bool /* verbose */ = false)
  {

    if(!out)
    {
      std::cerr << "Error: cannot open file" << std::endl;
      return false;
    }

    // Write header
    out << "ply" << std::endl
        << ((get_mode(out) == IO::BINARY) ? "format binary_little_endian 1.0" : "format ascii 1.0") << std::endl
        << "comment Generated by the CGAL library" << std::endl
        << "element vertex " << points.size() << std::endl;
    
    internal::PLY::output_property_header (out,
                                           make_ply_point_writer (CGAL::Identity_property_map<Point_3>()));

    out << "element face " << polygons.size() << std::endl;
  
    internal::PLY::output_property_header (out,
                                           std::make_pair (CGAL::Identity_property_map<Polygon_3>(),
                                                           PLY_property<std::vector<int> >("vertex_indices")));
    
    out << "end_header" << std::endl;
  
    for (std::size_t i = 0; i < points.size(); ++ i)
      internal::PLY::output_properties (out, points.begin() + i,
                                        make_ply_point_writer (CGAL::Identity_property_map<Point_3>()));

    for (std::size_t i = 0; i < polygons.size(); ++ i)
      internal::PLY::output_properties (out, polygons.begin() + i,
                                        std::make_pair (CGAL::Identity_property_map<Polygon_3>(),
                                                        PLY_property<std::vector<int> >("vertex_indices")));

    return out.good();
  }

  template <class SurfaceMesh>
  bool
  write_PLY(std::ostream& out,
            const SurfaceMesh& mesh,
            bool /* verbose */ = false)
  {
    typedef typename boost::graph_traits<SurfaceMesh>::face_descriptor face_descriptor;
    typedef typename boost::graph_traits<SurfaceMesh>::halfedge_descriptor halfedge_descriptor;
    typedef typename boost::graph_traits<SurfaceMesh>::vertex_descriptor vertex_descriptor;
    typedef typename boost::property_map<SurfaceMesh, boost::vertex_point_t>::type::value_type Point_3;
    typedef typename SurfaceMesh::template Property_map<halfedge_descriptor,std::pair<float, float> > UV_map;
    UV_map h_uv;
    bool has_texture;
    boost::tie(h_uv, has_texture) = mesh.template property_map<halfedge_descriptor,std::pair<float, float> >("h:uv");
    if(!out)
    {
      std::cerr << "Error: cannot open file" << std::endl;
      return false;
    }

    // Write header
    out << "ply" << std::endl
        << ((get_mode(out) == IO::BINARY) ? "format binary_little_endian 1.0" : "format ascii 1.0") << std::endl
        << "comment Generated by the CGAL library" << std::endl
        << "element vertex " << num_vertices(mesh) << std::endl;
    
    internal::PLY::output_property_header (out,
                                           make_ply_point_writer (CGAL::Identity_property_map<Point_3>()));

    out << "element face " << num_faces(mesh) << std::endl;
  
    internal::PLY::output_property_header (out,
                                           std::make_pair (CGAL::Identity_property_map<std::vector<std::size_t> >(),
                                                           PLY_property<std::vector<int> >("vertex_indices")));
    
    if(has_texture)
    {
      out << "element halfedge " << num_halfedges(mesh) << std::endl;
      
      internal::PLY::output_property_header (out,
                                             std::make_pair (CGAL::Identity_property_map<std::size_t >(),
                                                             PLY_property<unsigned int >("source")));
      
      internal::PLY::output_property_header (out,
                                             std::make_pair (CGAL::Identity_property_map<std::size_t >(),
                                                             PLY_property<unsigned int >("target")));
      internal::PLY::output_property_header (out,
                                             std::make_tuple (h_uv,
                                                              PLY_property<float>("u"),
                                                              PLY_property<float>("v")));
    }
    out << "end_header" << std::endl;

    for(vertex_descriptor vd : vertices(mesh))
    {
      Point_3 p = get(get(CGAL::vertex_point, mesh), vd);
      internal::PLY::output_properties (out, &p,
                                        make_ply_point_writer (CGAL::Identity_property_map<Point_3>()));
    }
    

    std::vector<std::size_t> polygon;
    for(face_descriptor fd : faces(mesh))
    {
      polygon.clear();
      for(halfedge_descriptor hd : halfedges_around_face(halfedge(fd, mesh), mesh))
        polygon.push_back (get(get(boost::vertex_index, mesh), target(hd,mesh)));

      internal::PLY::output_properties (out, &polygon,
                                        std::make_pair (CGAL::Identity_property_map<std::vector<std::size_t> >(),
                                                        PLY_property<std::vector<int> >("vertex_indices")));
    }
    
    if(has_texture)
    {
      for(halfedge_descriptor hd : halfedges(mesh))
      {
        typedef std::tuple<unsigned int, unsigned int, float, float> Super_tuple;
         Super_tuple t = 
            std::make_tuple(source(hd, mesh),target(hd, mesh),
                            h_uv[hd].first,
                            h_uv[hd].second);
        
        internal::PLY::output_properties (out, &t,
                                          std::make_pair (Nth_of_tuple_property_map<0,Super_tuple>(),
                                                          PLY_property<unsigned int >("source")),
                                          std::make_pair (Nth_of_tuple_property_map<1,Super_tuple>(),
                                                          PLY_property<unsigned int >("target")),
                                          std::make_pair (Nth_of_tuple_property_map<2,Super_tuple>(),
                                                          PLY_property<float>("u")),
                                          std::make_pair (Nth_of_tuple_property_map<3,Super_tuple>(),
                                                          PLY_property<float>("v")));
      }
    }
    return out.good();
  }

} // namespace CGAL

#endif // CGAL_IO_PLY_WRITER_H
