#ifndef CEPH_LIB_H
#define CEPH_LIB_H
#include <netinet/in.h>
#include <sys/statvfs.h>
#include <utime.h>
#include <sys/stat.h>
#include <stdbool.h>
#include <sys/types.h>
#include <unistd.h>
#include <dirent.h>

#ifdef __cplusplus
# define CEPH_DEFAULT_ARG(v) =v
#else
# define CEPH_DEFAULT_ARG(v)
#endif

struct stat_precise {
  ino_t st_ino;
  dev_t st_dev;
  mode_t st_mode;
  nlink_t st_nlink;
  uid_t st_uid;
  gid_t st_gid;
  dev_t st_rdev;
  off_t st_size;
  blksize_t st_blksize;
  blkcnt_t st_blocks;
  time_t st_atime_sec;
  time_t st_atime_micro;
  time_t st_mtime_sec;
  time_t st_mtime_micro;
  time_t st_ctime_sec;
  time_t st_ctime_micro;
};

#ifdef __cplusplus
extern "C" {
#endif

const char *ceph_version(int *major, int *minor, int *patch);

int ceph_initialize(int argc, const char **argv);
void ceph_deinitialize();

/* Sets a configuration value from a string.
 * Returns 0 on success, error code otherwise. */
int ceph_conf_set(const char *option, const char *value);

/* Returns a configuration value as a string.
 * If len is positive, that is the maximum number of bytes we'll write into the
 * buffer. If len == -1, we'll call malloc() and set *buf.
 * Returns 0 on success, error code otherwise. Returns ENAMETOOLONG if the
 * buffer is too short. */
int ceph_conf_get(const char *option, char *buf, size_t len);

int ceph_mount();
int ceph_umount();

int ceph_statfs(const char *path, struct statvfs *stbuf);
int ceph_get_local_osd();

int ceph_getcwd(char *buf, int buflen);
int ceph_chdir(const char *s);

int ceph_opendir(const char *name, DIR **dirpp);
int ceph_closedir(DIR *dirp);
int ceph_readdir_r(DIR *dirp, struct dirent *de);
int ceph_readdirplus_r(DIR *dirp, struct dirent *de, struct stat *st, int *stmask);
int ceph_getdents(DIR *dirp, char *name, int buflen);
int ceph_getdnames(DIR *dirp, char *name, int buflen);
void ceph_rewinddir(DIR *dirp); 
loff_t ceph_telldir(DIR *dirp);
void ceph_seekdir(DIR *dirp, loff_t offset);

int ceph_link (const char *existing, const char *newname);
int ceph_unlink (const char *path);
int ceph_rename(const char *from, const char *to);

// dirs
int ceph_mkdir(const char *path, mode_t mode);
int ceph_mkdirs(const char *path, mode_t mode);
int ceph_rmdir(const char *path);

// symlinks
int ceph_readlink(const char *path, char *buf, loff_t size);
int ceph_symlink(const char *existing, const char *newname);

// inode stuff
int ceph_lstat(const char *path, struct stat *stbuf);
int ceph_lstat_precise(const char *path, struct stat_precise *stbuf);

int ceph_setattr(const char *relpath, struct stat *attr, int mask);
int ceph_setattr_precise (const char *relpath, struct stat_precise *stbuf, int mask);
int ceph_chmod(const char *path, mode_t mode);
int ceph_chown(const char *path, uid_t uid, gid_t gid);
int ceph_utime(const char *path, struct utimbuf *buf);
int ceph_truncate(const char *path, loff_t size);

// file ops
int ceph_mknod(const char *path, mode_t mode, dev_t rdev CEPH_DEFAULT_ARG(0));
int ceph_open(const char *path, int flags, mode_t mode CEPH_DEFAULT_ARG(0));
int ceph_close(int fd);
loff_t ceph_lseek(int fd, loff_t offset, int whence);
int ceph_read(int fd, char *buf, loff_t size, loff_t offset CEPH_DEFAULT_ARG(-1));
int ceph_write(int fd, const char *buf, loff_t size, loff_t offset CEPH_DEFAULT_ARG(-1));
int ceph_ftruncate(int fd, loff_t size);
int ceph_fsync(int fd, bool syncdataonly);
int ceph_fstat(int fd, struct stat *stbuf);

int ceph_sync_fs();
int ceph_get_file_stripe_unit(int fh);
int ceph_get_file_replication(const char *path);
int ceph_get_default_preferred_pg(int fd);
int ceph_get_file_stripe_address(int fd, loff_t offset, char *buf, int buflen);
int ceph_set_default_file_stripe_unit(int stripe);
int ceph_set_default_file_stripe_count(int count);
int ceph_set_default_object_size(int size);
int ceph_set_default_file_replication(int replication);
int ceph_set_default_preferred_pg(int pg);
int ceph_localize_reads(int val);

#ifdef __cplusplus
}
#endif

#endif
