/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

/**
   \file cdw_mkudffs_helpers.c

   \brief Functions creating command strings for mkudffs helpers,
   i.e. for external tools that accompany mkudffs in creating UDF
   image file: truncate, sudo mount, rsync, sudo umount.
*/

#define _GNU_SOURCE /* asprintf() */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdbool.h>
#include <inttypes.h>

#include "gettext.h"
#include "cdw_thread.h"
#include "cdw_string.h"
#include "cdw_debug.h"
#include "cdw_ext_tools.h"
#include "cdw_mkudffs_helpers.h"
#include "cdw_file.h"
#include "cdw_file_manager.h"
#include "cdw_processwin.h"






/**

   \param task - variable describing current task

   \return CDW_ERROR on errors in the function or while running a task
   \return CDW_OK on success
*/
cdw_rv_t cdw_truncate_run_task(cdw_task_t *task)
{
	cdw_assert (task->id == CDW_TASK_CREATE_IMAGE_UDF,
		    "ERROR: incorrect task id is not CDW_TASK_CREATE_IMAGE_UDF: %lld\n", task->id);
	/* This function is for "truncate", but the main tool used for
	   creating UDF file system is mkudffs. This is a kind of
	   umbrella ID for all tools involved in creating UDF
	   image. */
	cdw_assert (task->create_image.tool.id == CDW_TOOL_MKUDFFS,
		    "ERROR: incorrect tool id is not CDW_TOOL_MKUDFFS in \"create image\" task: %lld\n", task->create_image.tool.id);
	cdw_assert (task->create_image.tool.label,
		    "ERROR: \"create image\" tool fullpath is not initialized (is NULL)\n");
	cdw_assert (task->create_image.udf.volume_size_bytes > 0,
		    "ERROR: volume size is zero bytes\n");

	char *size_string = (char *) NULL;
	int n = asprintf(&size_string, "%"PRIu64"", task->create_image.udf.volume_size_bytes);
	if (n == -1) {
		cdw_vdm ("ERROR: failed to allocate memory for size string\n");
		return CDW_ERROR;
	} else {
		cdw_vdm ("INFO: successfully created size string: \"%s\"\n",
			 size_string);
	}

	/* I'm using the explicit tool's name. Even if there are more
	   instances, I will use the default one. Instances of helpers
	   for mkudffs are not configurable. */
	char *command = cdw_string_concat("truncate ",
					  " --size=",
					  size_string,
					  /* Path may contain spaces, so use opening and closing quotes. */
					  " \"", task->image_file_fullpath, "\" ", (char *) NULL);

	free(size_string);
	size_string = (char *) NULL;

	if (!command) {
		cdw_vdm ("ERROR: failed to create command for truncating file\n");
		return CDW_ERROR;
	} else {
		cdw_sdm ("INFO: command string for \"truncate\" is \"%s\"\n", command);
		int rv = cdw_thread_run_command(command, task);
		free(command);
		command = (char *) NULL;

		cdw_vdm ("INFO: exit status of \"truncate\" is %d\n", task->tool_status.child_exit_status);

		if (rv == 0
		    && task->tool_status.mkudffs_general == CDW_TOOL_STATUS_OK
		    && task->tool_status.truncate == CDW_TOOL_STATUS_OK
		    && task->tool_status.child_exit_status == 0) {     /* Some other error that wasn't caught by regex code, so task->tool_status.mkudffs has not been set. */
			return CDW_OK;
		} else {
			return CDW_ERROR;
		}
	}
}





/**

   \param task - variable describing current task

   \return CDW_ERROR on errors in the function or while running a task
   \return CDW_OK on success
*/
cdw_rv_t cdw_mount_run_task(cdw_task_t *task)
{
	cdw_assert (task->id == CDW_TASK_CREATE_IMAGE_UDF,
		    "ERROR: incorrect task id is not CDW_TASK_CREATE_IMAGE_UDF: %lld\n", task->id);
	/* This function is for "mount", but the main tool used for
	   creating UDF file system is mkudffs. This is a kind of
	   umbrella ID for all tools involved in creating UDF
	   image. */
	cdw_assert (task->create_image.tool.id == CDW_TOOL_MKUDFFS,
		    "ERROR: incorrect tool id is not CDW_TOOL_MKUDFFS in \"create image\" task: %lld\n", task->create_image.tool.id);
	cdw_assert (task->create_image.tool.label,
		    "ERROR: \"create image\" tool fullpath is not initialized (is NULL)\n");


	/* uid option is used to ensure proper permissions for mount
	   point, so that copying files into mounted file system is
	   possible. */
	char *uid_string = (char *) NULL;
	int n = asprintf(&uid_string, "uid=%d", (int) getuid());
	if (n == -1) {
		cdw_vdm ("ERROR: asprintf() failed\n");
		return CDW_ERROR;
	} else {
		cdw_vdm ("INFO: mount uid option = \"%s\"\n", uid_string);
	}

	/* I'm using the explicit tool's name. Even if there are more
	   instances, I will use the default one. Instances of helpers
	   for mkudffs are not configurable. */
	/*
	  man sudo(8):
          "
	  -n, --non-interactive
	         Avoid prompting the user for input of any kind.  If a
                 password is required for the command to run, sudo
                 will display an error message and exit.
	  "

	  Otherwise sudo would print prompt to current tty, and no
	  amount of redirecting stdout/stderr output by cdw would
	  help.

	  The error message mentioned in man page is
	  "sudo: a password is required"
	*/
	char *command = cdw_string_concat("sudo -n mount ",
					  " -t udf ",
					  " -oloop,rw,",   /* No space between this string and uid string! */
					  uid_string,
					  /* Paths may contain spaces, so use opening and closing quotes. */
					  " \"", task->image_file_fullpath, "\" ",
					  " \"", task->create_image.udf.mount_point, "\" ",
					  (char *) NULL);
	free(uid_string);
	uid_string = (char *) NULL;

	if (!command) {
		cdw_vdm ("ERROR: failed to create command for mounting\n");
		return CDW_ERROR;
	} else {
		cdw_sdm ("INFO: command string for \"sudo mount\" is \"%s\"\n", command);
		int rv = cdw_thread_run_command(command, task);
		free(command);
		command = (char *) NULL;

		cdw_vdm ("INFO: exit status of \"sudo mount\" is %d\n", task->tool_status.child_exit_status);

		if (rv == 0
		    && task->tool_status.mkudffs_general == CDW_TOOL_STATUS_OK
		    && task->tool_status.sudo == CDW_TOOL_STATUS_OK
		    && task->tool_status.mount == CDW_TOOL_STATUS_OK
		    && task->tool_status.child_exit_status == 0) {     /* Some other error that wasn't caught by regex code, so task->tool_status.mkudffs has not been set. */

			return CDW_OK;
		} else {
			return CDW_ERROR;
		}
	}
}





/**

   \param task - variable describing current task

   \return CDW_ERROR on errors in the function or while running a task
   \return CDW_OK on success
*/
cdw_rv_t cdw_umount_run_task(cdw_task_t *task)
{
	cdw_assert (task->id == CDW_TASK_CREATE_IMAGE_UDF,
		    "ERROR: incorrect task id is not CDW_TASK_CREATE_IMAGE_UDF: %lld\n", task->id);
	/* This function is for "umount", but the main tool used for
	   creating UDF file system is mkudffs. This is a kind of
	   umbrella ID for all tools involved in creating UDF
	   image. */
	cdw_assert (task->create_image.tool.id == CDW_TOOL_MKUDFFS,
		    "ERROR: incorrect tool id is not CDW_TOOL_MKUDFFS in \"create image\" task: %lld\n", task->create_image.tool.id);
	cdw_assert (task->create_image.tool.label,
		    "ERROR: \"create image\" tool fullpath is not initialized (is NULL)\n");


	/* I'm using the explicit tool's name. Even if there are more
	   instances, I will use the default one. Instances of helpers
	   for mkudffs are not configurable. */
	/*
	  man sudo(8):
          "
	  -n, --non-interactive
	         Avoid prompting the user for input of any kind.  If a
                 password is required for the command to run, sudo
                 will display an error message and exit.
	  "

	  Otherwise sudo would print prompt to current tty, and no
	  amount of redirecting stdout/stderr output by cdw would
	  help.

	  The error message mentioned in man page is
	  "sudo: a password is required"
	*/
	char *command = cdw_string_concat("sudo -n umount ",
					  /* Path may contain spaces, so use opening and closing quotes. */
					  " \"", task->create_image.udf.mount_point, "\" ",
					  (char *) NULL);

	if (!command) {
		cdw_vdm ("ERROR: failed to create command for unmounting\n");
		return CDW_ERROR;
	} else {
		cdw_sdm ("INFO: command string for \"sudo umount\" is \"%s\"\n", command);
		int rv = cdw_thread_run_command(command, task);
		free(command);
		command = (char *) NULL;

		cdw_vdm ("INFO: exit status of \"umount\" is %d\n", task->tool_status.child_exit_status);

		if (rv == 0
		    && task->tool_status.mkudffs_general == CDW_TOOL_STATUS_OK
		    && task->tool_status.sudo == CDW_TOOL_STATUS_OK
		    && task->tool_status.umount == CDW_TOOL_STATUS_OK
		    && task->tool_status.child_exit_status == 0) {     /* Some other error that wasn't caught by regex code, so task->tool_status.mkudffs has not been set. */

			return CDW_OK;
		} else {
			return CDW_ERROR;
		}
	}
}





/**

   \param task - variable describing current task

   \return CDW_ERROR on errors in the function or while running a task
   \return CDW_OK on success
*/
cdw_rv_t cdw_rsync_run_task(cdw_task_t *task)
{
	cdw_assert (task->id == CDW_TASK_CREATE_IMAGE_UDF,
		    "ERROR: incorrect task id is not CDW_TASK_CREATE_IMAGE_UDF: %lld\n", task->id);
	/* This function is for "rsync", but the main tool used for
	   creating UDF file system is mkudffs. This is a kind of
	   umbrella ID for all tools involved in creating UDF
	   image. */
	cdw_assert (task->create_image.tool.id == CDW_TOOL_MKUDFFS,
		    "ERROR: incorrect tool id is not CDW_TOOL_MKUDFFS in \"create image\" task: %lld\n", task->create_image.tool.id);
	cdw_assert (task->create_image.tool.label,
		    "ERROR: \"create image\" tool fullpath is not initialized (is NULL)\n");


	cdw_dll_item_t *f = cdw_file_manager_get_list_of_selected_files(); /* head */
	cdw_assert (f, "ERROR: called the function when list of selected files is empty\n")

	size_t files_i = 1;
	size_t files_n = cdw_file_manager_number_of_selected_files();

	for (; f; f = f->next) {
		cdw_file_t *file = (cdw_file_t *) f->data;
		cdw_assert (file->fullpath, "ERROR: file has no fullpath\n");
		cdw_assert (strlen(file->fullpath) != 0, "ERROR: file full path length == 0\n");
		cdw_assert (file->name_start > 0, "ERROR: file has no name\n");
		cdw_assert (strlen(file->fullpath + file->name_start) != 0, "ERROR: file name length == 0\n");

		/* I'm using the explicit tool's name. Even if there are more
		   instances, I will use the default one. Instances of helpers
		   for mkudffs are not configurable. */
		char *command = cdw_string_concat("rsync ",
						  /* These two options are hardwired because I don't think that any user would object having them. And I find them useful and needed. */
						  "--verbose ",
						  "--progress "

						  /* Any other options that user needs, including options for symbolic links. */
						  " ", task->create_image.udf.rsync_options, " ",

						  /* SOURCE */
						  /* Path may contain spaces, so use opening and closing quotes. */
						  " \"", file->fullpath, "\" ",

						  /* TARGET */
						  /* Paths may contain spaces, so use opening and closing quotes. */
						  /* The "/" is used to be sure that a slash exists at the end of mount point dir. */
						  //" \"", task->create_image.udf.mount_point,"/",file->fullpath + file->name_start, "\" ",
						  " \"", task->create_image.udf.mount_point,"/", "\" ",
						  (char *) NULL);

		if (!command) {
			cdw_vdm ("ERROR: failed to create command for copying file \"%s\"\n", file->fullpath);
			return CDW_ERROR;
		} else {
			/* -(2 * 2) for spaces on left and right side of string, +1 for NULL. */
#define PROGRESS_LEN (PROCESSWIN_COLS - (2 * 2))
			char files_progress[PROGRESS_LEN + 1];
			snprintf(files_progress, PROGRESS_LEN + 1, "Selected file %zd/%zd", files_i, files_n);
			cdw_processwin_display_sub_info(files_progress);


			cdw_sdm ("INFO: command string for \"rsync\" is \"%s\"\n", command);
			int rv = cdw_thread_run_command(command, task);
			free(command);
			command = (char *) NULL;

			cdw_vdm ("INFO: exit status of \"rsync\" is %d\n", task->tool_status.child_exit_status);

			if (rv == 0
			    && task->tool_status.mkudffs_general == CDW_TOOL_STATUS_OK
			    && task->tool_status.rsync == CDW_TOOL_STATUS_OK
			    && task->tool_status.child_exit_status == 0) {     /* Some other error that wasn't caught by regex code, so task->tool_status.mkudffs has not been set. */
			} else {
				return CDW_ERROR;
			}
		}

		files_i++;
	}

	/* Hide progress information. */
	cdw_processwin_display_sub_info("");
	cdw_processwin_delete_progress_bar();
	cdw_processwin_wrefresh();

	return CDW_OK;
}





/**

   \param task - variable describing current task

   \return CDW_ERROR on errors in the function or while running a task
   \return CDW_OK on success
*/
cdw_rv_t cdw_sync_run_task(cdw_task_t *task)
{
	cdw_assert (task->id == CDW_TASK_CREATE_IMAGE_UDF,
		    "ERROR: incorrect task id is not CDW_TASK_CREATE_IMAGE_UDF: %lld\n", task->id);
	/* This function is for "sync", but the main tool used for
	   creating UDF file system is mkudffs. This is a kind of
	   umbrella ID for all tools involved in creating UDF
	   image. */
	cdw_assert (task->create_image.tool.id == CDW_TOOL_MKUDFFS,
		    "ERROR: incorrect tool id is not CDW_TOOL_MKUDFFS in \"create image\" task: %lld\n", task->create_image.tool.id);
	cdw_assert (task->create_image.tool.label,
		    "ERROR: \"create image\" tool fullpath is not initialized (is NULL)\n");


	/* I'm using the explicit tool's name. Even if there are more
	   instances, I will use the default one. Instances of helpers
	   for mkudffs are not configurable. */
	char *command = cdw_string_concat("sync",
					  (char *) NULL);

	if (!command) {
		cdw_vdm ("ERROR: failed to create command for syncing\n");
		return CDW_ERROR;
	} else {
		cdw_sdm ("INFO: command string for \"sync\" is \"%s\"\n", command);
		int rv = cdw_thread_run_command(command, task);
		free(command);
		command = (char *) NULL;

		cdw_vdm ("INFO: exit status of \"sync\" is %d\n", task->tool_status.child_exit_status);

		if (rv == 0
		    && task->tool_status.mkudffs_general == CDW_TOOL_STATUS_OK
		    && task->tool_status.sync == CDW_TOOL_STATUS_OK
		    && task->tool_status.child_exit_status == 0) {     /* Some other error that wasn't caught by regex code, so task->tool_status.mkudffs has not been set. */

			return CDW_OK;
		} else {
			return CDW_ERROR;
		}
	}
}
