/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2011 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "gettext.h"
#include "cdw_processwin.h"
#include "cdw_widgets.h"
#include "cdw_digest.h"
#include "cdw_logging.h"
#include "cdw_debug.h"
#include "cdw_main_window.h"
#include "cdw_read_disc.h"
#include "cdw_fs.h"
#include "cdw_calculate_digest.h"
#include "cdw_file_picker.h"
#include "cdw_verify_wizard.h"
#include "cdw_config.h"

extern cdw_config_t global_config;

/* digest regex code is called twice, this is counter allowing
   the regex code to recognize in which table store digest */
int digest_index = CDW_CALCULATE_DIGEST_MODE_NONE;

/** \brief Buffer for digest of optical disc track */
char digest_disc[CDW_DIGEST_LEN_MAX + 1];
/** \brief Buffer for digest of regular file */
char digest_file[CDW_DIGEST_LEN_MAX + 1];


static bool using_local_processwin = false;
/* id used for look up of tool name used in summary printed to log file */
static cdw_id_t digest_tool_id = CDW_TOOL_NONE;


static cdw_rv_t cdw_calculate_digest_disc(cdw_task_t *task);
static cdw_rv_t cdw_calculate_digest_set_up_reading_disc(cdw_task_t *task);
static void    *cdw_calculate_digest_read_and_write_disc(void *thread_task);
static void     cdw_calculate_digest_clean_up_reading_disc(void);

static cdw_rv_t cdw_calculate_digest_file(cdw_task_t *task);
static cdw_rv_t cdw_calculate_digest_set_up_reading_file(cdw_task_t *task, const char *file_fullpath);
static void    *cdw_calculate_digest_read_and_write_file(void *thread_task);
static void     cdw_calculate_digest_clean_up_reading_file(cdw_task_t *task);

static void cdw_calculate_digest_summary(cdw_id_t mode, cdw_id_t tool_id);
static void cdw_calculate_digest_processwin_create(cdw_id_t mode);
static void cdw_calculate_digest_processwin_destroy(const char *message, bool wait);
static void cdw_calculate_digest_processwin_update(long long i, long long n, long long size);




cdw_rv_t cdw_calculate_digest(cdw_id_t mode, const char *file_fullpath)
{
	if (!cdw_ext_tools_digest_tool_available()) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   _("Can't perform the task, it seems that there are no digest tools (like md5sum or sha1sum) installed in your system."),
				   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
		return CDW_NO;
	}

	digest_disc[0] = '\0';
	digest_file[0] = '\0';

	cdw_calculate_digest_processwin_create(mode);

	cdw_task_t *task = cdw_task_new(CDW_TASK_CALCULATE_DIGEST, (cdw_disc_t *) NULL);
	if (task == (cdw_task_t *) NULL) {
		cdw_vdm ("ERROR: failed to create a task\n");
		return CDW_ERROR;
	}
	digest_tool_id = task->calculate_digest.tool.id;
	task->calculate_digest.mode = mode;

	/* setup */

	task->calculate_digest.data_size_sectors = -1;
	if (mode == CDW_CALCULATE_DIGEST_MODE_FILE || mode == CDW_CALCULATE_DIGEST_MODE_DISC_FILE) {
		/* There is one difficulty when calculating digest of
		   track on a disc: how many sectors from the end of the
		   track to read? How many of the empty sectors at the end
		   of the track come from source ISO image file, and how
		   many of them are a padding added by burning tool?

		   This is a bit easier when you are comparing disc track
		   against some ISO9660 file - you can take size of the file
		   as probable size of disc track, and attempt to read as
		   many sectors in the file as there are in ISO image file.

		   The universal, common data size when comparing disc track
		   against ISO image file (or when only reading ISO file) is
		   data_size_sectors.

		   When cdw only calculates digest for disc track (and it
		   doesn't have any source ISO image file to use as reference)
		   then size of data to read is discovered by cdio module,
		   and - due to empty sectors at the end of track - this may
		   produce digest different than digest of source ISO file
		   (which we don't have). */
		cdw_rv_t crv = cdw_calculate_digest_set_up_reading_file(task, file_fullpath);
		if (crv != CDW_OK) {
			cdw_task_delete(&task);
			return CDW_ERROR;
		} else {
			task->calculate_digest.data_size_sectors = task->calculate_digest.file_size / 2048;
			cdw_vdm ("INFO: setting data size to read to %lld sectors (%lld bytes)\n",
				 task->calculate_digest.data_size_sectors,
				 task->calculate_digest.file_size);
		}
	}


	if (mode == CDW_CALCULATE_DIGEST_MODE_DISC || mode == CDW_CALCULATE_DIGEST_MODE_DISC_FILE) {

		cdw_rv_t crv = cdw_calculate_digest_disc(task);

		/* when cdw is run in login console and there are some
		   disc read errors, OS prints some error messages to the
		   console that overwrite cdw ui; this line fixes it */
		cdw_main_ui_main_window_wrefresh();
		cdw_processwin_wrefresh();

		if (crv != CDW_OK) {
			cdw_calculate_digest_processwin_destroy(_("Error"), true);
			return CDW_ERROR;
		}
	}
	if (mode == CDW_CALCULATE_DIGEST_MODE_FILE || mode == CDW_CALCULATE_DIGEST_MODE_DISC_FILE) {
		cdw_processwin_reset_progress();
		cdw_rv_t crv = cdw_calculate_digest_file(task);
		if (crv != CDW_OK) {
			cdw_calculate_digest_processwin_destroy(_("Error"), true);
			return CDW_ERROR;
		}
	}

	cdw_task_delete(&task);
	cdw_calculate_digest_summary(mode, digest_tool_id);
	cdw_calculate_digest_processwin_destroy("", false);

	return CDW_OK;
}





cdw_rv_t cdw_calculate_digest_file(cdw_task_t *task)
{
	digest_index = CDW_CALCULATE_DIGEST_MODE_FILE;

	//cdw_rv_t crv = cdw_calculate_digest_set_up_reading_file(task, file_fullpath);

	/* 2TRANS: this is message in process window */
	cdw_processwin_display_main_info(_("Calculating digest of input file"));

	task->calculate_digest.read_and_write = cdw_calculate_digest_read_and_write_file;
	cdw_rv_t crv = cdw_digest_run_task(task);
	cdw_calculate_digest_clean_up_reading_file(task);

	cdw_rv_t tool_status = cdw_task_check_tool_status(task);

	if (tool_status != CDW_OK || crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to check digest of input file\n");
		if (tool_status != CDW_OK) {
			cdw_vdm ("ERROR: tool_status.ok flag was set to \"false\" after checking data file\n");
		}
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}





cdw_rv_t cdw_calculate_digest_disc(cdw_task_t *task)
{
	digest_index = CDW_CALCULATE_DIGEST_MODE_DISC;

	cdw_rv_t crv = cdw_calculate_digest_set_up_reading_disc(task);
	if (crv != CDW_OK) {
		cdw_task_delete(&task);
		return CDW_ERROR;
	}

	/* 2TRANS: this is message in progress window:
	   description of task currently performed */
	cdw_processwin_display_main_info(_("Calculating digest of disc track"));
	/* this erases stale message in line 2 */
	cdw_processwin_display_sub_info("");
	cdw_processwin_wrefresh();

	task->calculate_digest.read_and_write = cdw_calculate_digest_read_and_write_disc;
	crv = cdw_digest_run_task(task);
	cdw_calculate_digest_clean_up_reading_disc();

	cdw_rv_t tool_status = cdw_task_check_tool_status(task);

	if (tool_status != CDW_OK || crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to check digest of track on disc\n");
		if (tool_status != CDW_OK) {
			cdw_vdm ("ERROR: tool_status.ok flag was set to \"false\" after checking track on disc\n");
		}
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}





void cdw_calculate_digest_summary(cdw_id_t mode, cdw_id_t tool_id)
{
	/* only md5sum digest would fit into dialog window without wrapping;
	   I could wrap digest strings for other algorithms and print them
	   in dialog window, but that wouldn't look good;
	   I could print md5sum digest in dialog window and redirect user
	   to log file in rest of the cases to log file, but that would be
	   inconsistent.
	   Therefore digest for every algorithm is printed to log file only.

	   On the other hand, most often used algorithm is md5sum, so the
	   inconsistency mentioned above wouldn't be that visible. */


	if (mode == CDW_CALCULATE_DIGEST_MODE_FILE || mode == CDW_CALCULATE_DIGEST_MODE_DISC) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Message"), _("Finished. Digest value is in log file ('L' hotkey in main window)."),
				   CDW_BUTTONS_OK, CDW_COLORS_DIALOG);
		/* 2TRANS: this is text written to log file: a header printed before
		   a digest for data file is printed;
		   "%s" is a name of tool/algorithm used to calculate digest, like
		   "md5sum" or "sha256" */
		cdw_logging_write(_("\n\nDigest (%s):\n"), cdw_ext_tools_get_tool_name(tool_id));
		if (mode == CDW_CALCULATE_DIGEST_MODE_FILE) {
			/* 2TRANS: this is text written to log file: %s is a digest of
			   data burned to first track of optical disc;
			   keep '\n' chars in place; keep " chars around %s */
			cdw_logging_write(_("Digest of input file:\n\"%s\"\n"), digest_file);
		} else {
			/* 2TRANS: this is text written to log file: %s is a digest of
			   data burned to first track of optical disc;
			   keep '\n' chars in place; keep " chars around %s */
			cdw_logging_write(_("Digest of first track on optical disc:\n\"%s\"\n"), digest_disc);
		}

		cdw_logging_write("\n\n");
		return;
	} else {
		bool success = !strcmp(digest_disc, digest_file);

		/* 2TRANS: this is message in dialog window: it describes result of verification */
		const char *message_the_same = _("Verification finished, data file and first track on disc are the same.\n\nDigest values have been printed to log file ('L' key in main window).");
		/* 2TRANS: this is message in dialog window: it describes result of verification */
		const char *message_different = _("Verification finished, data file and first track on disc are different.\n\nDigest values have been printed to log file ('L' key in main window).");

		if (success) {
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Message"), message_the_same,
					   CDW_BUTTONS_OK, CDW_COLORS_DIALOG);
		} else {
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Error"), message_different,
				   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
		}

		cdw_main_ui_main_window_wrefresh();

		/* Add some '\n' chars to keep strings grouped and aligned;
		   especially important for digest strings - if they start in the
		   same column it is easier to compare them */

		/* 2TRANS: this is text written to log file: a header printed before
		   two digests for track and for input file are printed;
		   "%s" is a name of tool/algorithm used to calculate digest, like
		   "md5sum" or "sha256" */
		cdw_logging_write(_("\n\nDigests (%s):\n"), cdw_ext_tools_get_tool_name(tool_id));
		/* 2TRANS: this is text written to log file: %s is a digest of
		   data burned to first track of optical disc;
		   keep '\n' chars in place; keep " chars around %s */
		cdw_logging_write(_("Digest of burned data:\n\"%s\"\n"), digest_disc);
		/* 2TRANS: this is text written to log file: %s is digest of
		   input data file; keep '\n' chars in place;  keep " chars around %s */
		cdw_logging_write(_("Digest of input data file:\n\"%s\"\n"), digest_file);
		cdw_logging_write("\n\n");
	}

	return;
}





/**
   \brief Initialize cdio disc for reading of data track

   Open cdio disc and get its metadata needed for further reading of
   sectors from first track. Function will return CDW_ERROR if
   disc cannot be open, if function cannot read metainformation from that
   disc or if the track is of wrong kind.

   \para task - current task

   \return CDW_OK on success
   \return CDW_ERROR on other problems
*/
cdw_rv_t cdw_calculate_digest_set_up_reading_disc(cdw_task_t *task)
{
	cdw_rv_t crv = cdw_read_disc_check_preconditions();
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to get disc\n");
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window:
				      some preconditions of reading a CD were not met */
				   _("Can't open disc."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_ERROR;
	}

	crv = cdw_cdio_open_disc();

	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to open cdio disc\n");
		task->tool_status.libcdio |= CDW_TOOL_STATUS_LIBCDIO_SETUP_FAIL;
		cdw_vdm ("ERROR: task->tool_status |= CDW_TOOL_STATUS_LIBCDIO_SETUP_FAIL\n");
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window: disc
				      is in drive, but cdw cannot open it using a
				      library API */
				   _("Cannot open disc."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_ERROR;
	}

	/* we can verify only first track of type supported by cdw */
	track_t n_tracks = cdw_cdio_get_number_of_tracks();
	track_t first_track = cdw_cdio_get_first_track();

	if (n_tracks > 1) {
		cdw_vdm ("WARNING: the disc has more than one track, cdw will check only first track\n");
	}
	if (cdw_cdio_is_data_track(first_track)) {
		return CDW_OK;
	} else {
		cdw_vdm ("ERROR: failed to set up reading optical disc\n");
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window:
				      it describes result of verification */
				   _("Failed to set up reading from optical disc."),
				   CDW_BUTTONS_OK, CDW_COLORS_DIALOG);
		return CDW_ERROR;
	}
}





/**
   \brief Read first track of optical disc

   \param thread_task - current task

   \return NULL pointer
*/
void *cdw_calculate_digest_read_and_write_disc(void *thread_task)
{
	cdw_task_t *task = (cdw_task_t *) thread_task;

	//cdw_vdm ("INFO: copying %ld sectors from disc\n", task->calculate_digest.data_size_sectors);

	int rv = cdw_cdio_copy_sectors_to_file(cdw_cdio_get_first_track_number(),
					       task->calculate_digest.target_fd,
					       task->calculate_digest.data_size_sectors);

	/* we need to close digest input, and we need to do this right now,
	   in read/write function, i.e. still in thread;
	   closing file equals marking end of input for digest tool, without
	   this digest tool won't exit, thread function won't return, and
	   cdw will be struck */
	close(task->calculate_digest.target_fd);

	if (rv == 0) {
		cdw_vdm ("INFO: read and write optical disc sectors ok, end\n");
	} else {
		task->tool_status.libcdio |= CDW_TOOL_STATUS_LIBCDIO_READ_FAIL;
		cdw_vdm ("ERROR: tool_status.libcdio |= CDW_TOOL_STATUS_LIBCDIO_READ_FAIL\n");
		cdw_cdio_copy_print_debug_on_error(rv);
	}
	return (void *) NULL;
}





/**
   \brief Close disc that you have opened with cdw_calculate_digest_set_up_reading_disc()

   Function closes disc using proper cdio call.
*/
void cdw_calculate_digest_clean_up_reading_disc(void)
{
	cdw_vdm ("INFO: closing data source\n");
	cdw_cdio_close_disc();

	return;
}





cdw_rv_t cdw_calculate_digest_set_up_reading_file(cdw_task_t *task, const char *file_fullpath)
{
	cdw_vdm ("INFO: attempting to open file with fullpath %s\n", file_fullpath);
	task->calculate_digest.source_fd = open(file_fullpath, O_RDONLY);
	if (task->calculate_digest.source_fd == -1) {
		int e = errno;
		task->tool_status.calculate_digest |= CDW_TOOL_STATUS_CALCULATE_DIGEST_SETUP_FAIL;
		cdw_vdm ("ERROR: tool_status.calculate_digest |= CDW_TOOL_STATUS_CALCULATE_DIGEST_SETUP_FAIL\n");
		cdw_vdm ("ERROR: failed to open data file, reason: %s\n", strerror(e));
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Failed to open given file."),
				   CDW_BUTTONS_OK, CDW_COLORS_DIALOG);
		return CDW_ERROR;
	} else {
		cdw_vdm ("INFO: file open ok\n");

		task->calculate_digest.file_size = cdw_fs_get_file_size(file_fullpath);
		cdw_vdm ("INFO: fs_size = %lld\n", task->calculate_digest.file_size);
		if (task->calculate_digest.file_size % 2048) {
			cdw_vdm ("WARNING: file size is not multiple of 2048\n");
		}

		return CDW_OK;
	}
}





void *cdw_calculate_digest_read_and_write_file(void *thread_task)
{
	cdw_task_t *task = (cdw_task_t *) thread_task;

	/* WARNING: the buffer can't be too large; there were some
	   problems when n was 100k, the code could read that much,
	   but was unable to write that much in single function call
	   (e.g. w = 31744 r = 100000); this problem happened for
	   reading data file one in two times, but never for reading
	   optical disc; so it's better to keep the value low,
	   e.g. 10k, and do *multiple* reads to be sure
	   that the buffer is not too large;

	   EDIT: this problem may occur only when running cdw in gdb,
	   but still...

	*/

	long long buf_size = 10 * 1000;
	char read_buffer[buf_size];

	long long int n_reads = task->calculate_digest.file_size / buf_size;
	if (task->calculate_digest.file_size % buf_size) {
		n_reads++;
	}

	ssize_t r = 0, w = 0;
	int e = 0;
	long long int i = 0;
	do {
		r = read(task->calculate_digest.source_fd, read_buffer, (size_t) buf_size);
		if (r > 0) {
			w = write(task->calculate_digest.target_fd, read_buffer, (size_t) r);
			i++;
			if ((i % (3 * 1000)) == 0) {
				cdw_calculate_digest_processwin_update(i, n_reads, buf_size);
	}

		} else {
			e = errno;
		}
		cdw_sdm ("INFO: read %zd, write %zd\n", r, w);
	} while (w == r);

	/* we need to close digest input, and we need to do this right now,
	   in read/write function, i.e. still in thread;
	   closing file equals marking end of input for digest tool, without
	   this digest tool won't exit, thread function won't return, and
	   cdw will be struck */
	close(task->calculate_digest.target_fd);

	if (r == 0) {
		cdw_vdm ("INFO: data file read ok, EOF\n");
		/* even though we have read/written whole file,
		   last call to cdw_calculate_digest_processwin_update()
		   may have resulted in displaying progress < 100%; let's
		   signal reaching "100%" by this additional call */
		cdw_calculate_digest_processwin_update(n_reads - 1, n_reads, buf_size);
	} else {
		task->tool_status.calculate_digest |= CDW_TOOL_STATUS_CALCULATE_DIGEST_READ_FAIL;
		cdw_vdm ("ERROR: tool_status.calculate_digest |= CDW_TOOL_STATUS_CALCULATE_DIGEST_READ_FAIL\n");
		cdw_vdm ("ERROR: read() returns -1, reason: %s\n", strerror(e));
	}
	return (void *) NULL;
}





void cdw_calculate_digest_processwin_update(long long i, long long n, long long size)
{
	/* dividing by 4s to avoid overflows */
	long current = ((i / 4) * (size / 4)) / (256 * 256);
	long total = ((n / 4) * (size / 4)) / (256 * 256);
	char current_value_string[PROCESSWIN_MAX_RTEXT_LEN + 1];

	/* 2TRANS: this string will be displayed as message in progress window;
	   first %ld is amount of data already read from CD,
	   second %ld is total amount of data to be read */
	snprintf(current_value_string, PROCESSWIN_MAX_RTEXT_LEN + 1, _("%ld/%ld MB"), current, total);
	cdw_vdm ("INFO: current value string = \"%s\"\n", current_value_string);
	cdw_processwin_display_progress_conditional(current, total, current_value_string);

	return;
}





void cdw_calculate_digest_clean_up_reading_file(cdw_task_t *task)
{
	cdw_vdm ("INFO: closing data source\n");
	close(task->calculate_digest.source_fd);

	return;
}





void cdw_calculate_digest_processwin_create(cdw_id_t mode)
{
	if (cdw_processwin_is_active()) {
		using_local_processwin = false;
	} else {
		using_local_processwin = true;

		const char *message = (char *) NULL;
		if (mode == CDW_CALCULATE_DIGEST_MODE_DISC) {
			message = _("Verifying data on optical disc");
		} else if (mode == CDW_CALCULATE_DIGEST_MODE_FILE) {
			message = _("Verifying data in given file");
		} else { /* CDW_CALCULATE_DIGEST_MODE_DISC_FILE */
			message = _("Comparing data on disc and in given file");
		}

		/* 2TRANS: this is title of process window */
		cdw_processwin_create(_("Verify data"), message, true);
	}
	return;
}





void cdw_calculate_digest_processwin_destroy(const char *message, bool wait)
{
	if (using_local_processwin) {
		cdw_processwin_destroy(message, wait);
	} else {
		cdw_processwin_display_main_info(message);
	}
	return;
}





cdw_rv_t cdw_verify(void)
{
	if (!cdw_ext_tools_digest_tool_available()) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   _("Can't perform the task, it seems that there are no digest tools (like md5sum or sha1sum) installed in your system."),
				   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
		return CDW_NO;
	}

	cdw_id_t mode = cdw_verify_wizard();

	if (mode == CDW_CALCULATE_DIGEST_MODE_NONE) {
		/* user pressed Cancel or Escape key in wizard */
		return CDW_CANCEL;
	}

	char *fullpath = (char *) NULL;
	if (global_config.iso_image_full_path != (char *) NULL) {
		fullpath = strdup(global_config.iso_image_full_path);
	} else {
		fullpath = cdw_fs_get_initial_dirpath();
	}
	if (fullpath == (char *) NULL) {
		fullpath = strdup("/"); /* last resort */
	}

	if (mode == CDW_CALCULATE_DIGEST_MODE_FILE || mode == CDW_CALCULATE_DIGEST_MODE_DISC_FILE) {
		cdw_rv_t crv = cdw_fs_ui_file_picker(_("Select data file"),
						     _("Select existing file to verify"),
						     &fullpath, CDW_FS_FILE, R_OK, CDW_FS_EXISTING);

		if (crv == CDW_CANCEL || crv == CDW_ERROR) {
			return crv;
		}
	}

	cdw_main_ui_main_window_wrefresh();

	cdw_rv_t crv = cdw_calculate_digest(mode, fullpath);
	free(fullpath);
	fullpath = (char *) NULL;

	return crv;
}

