/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2011 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#define _BSD_SOURCE /* lstat() */

#include <unistd.h>
#include <errno.h>
#include <string.h>
/* open() */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>


#include "cdw_config.h"
#include "cdw_fs.h"
#include "cdw_iso9660.h"
#include "isosize.h"
#include "cdw_debug.h"
#include "cdw_widgets.h"
#include "gettext.h"

extern cdw_config_t global_config;      /* variable holding cdw configuration */


/**
  \brief Function returns size of ISO9660 file system in current ISO image file

  Size of the file system is counted in 2048-byte sectors.
  Size of file system may be different than size of file on a disc
*/
long cdw_iso_image_get_n_sectors(void)
{
	cdw_vdm ("INFO: inspecting file \"%s\"\n", global_config.iso_image_full_path);
	int f = open(global_config.iso_image_full_path, O_RDONLY);
	if (f == -1) {
		cdw_fs_errno_handler(errno);
		cdw_vdm ("ERROR: failed to open iso file image\n");
		return CDW_ERROR;
	}
	long long iso_size = isosize(f);
	close(f);

	long long fs_size = cdw_fs_get_file_size(global_config.iso_image_full_path);
	cdw_vdm ("INFO: iso_size = %lld, fs_size = %lld\n", iso_size, fs_size);

	if (iso_size % 2048) {
		cdw_vdm ("ERROR: iso_size is not multiple of 2048\n");
	}
	if (fs_size % 2048) {
		cdw_vdm ("WARNING: fs_size is not multiple of 2048\n");
	}

	long long size = iso_size;
	long sectors = size / 2048;

	/* WARNING: sector size is set as constant (2048), but:
	   - 2048 is at the moment the only sector size supported by cdw
	   - mkisofs man page describes "-print-size" as: "Print
	   estimated filesystem size in multiples of the sector
	   size (2048 bytes) and exit". Why only 2048? Who should
	   re-calculate the size when another size should be used?
	   cdrecord or caller of cdrecord? */

	/* 524288 * 2048-byte sectors = 1GB */
	cdw_vdm ("INFO: returning %ld sectors (%.2f%s)\n",
		 sectors,
		 sectors >= 524288 ? (double) size / (1024 * 1024 * 1024) : (double) size / (1024 * 1024),
		 sectors >= 524288 ? "GB" : "MB");
	return sectors;
}





double cdw_iso_image_get_size_mb(const char *fullpath)
{
	struct stat stbuf;
	int rv = lstat(fullpath, &stbuf);
	if (rv != 0) {
		int e = errno;
		cdw_vdm ("ERROR: can't lstat() iso file \"%s\", error = \"%s\"\n", fullpath, strerror(e));
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("There seems to be problem with iso file. Please check the file location."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		cdw_fs_errno_handler(e);

		return CDW_ERROR;
	}

	long long iso_file_size_b = (long long) stbuf.st_size;
	double iso_file_size_mb = (double) (iso_file_size_b / (1024 * 1024));
	cdw_vdm ("INFO: size in bytes = %lld, size in megabytes = %f\n", iso_file_size_b, iso_file_size_mb);

	return iso_file_size_mb;
}



