/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "cdw_task.h"
#include "cdw_disc.h"
#include "cdw_widgets.h"
#include "gettext.h"
#include "cdw_debug.h"
#include "cdw_erase_disc.h"
#include "cdw_main_window.h"
#include "cdw_utils.h"
#include "cdw_erase_wizard.h"
#include "cdw_drive.h"
#include "cdw_config.h"
#include "cdw_cdrecord.h"
#include "cdw_dvd_rw_format.h"
#include "cdw_ext_tools.h"
#include "cdw_processwin.h"
#include "cdw_read_disc_info.h"



static cdw_rv_t cdw_erase_disc_check_preconditions(cdw_disc_t *disc);
static cdw_rv_t cdw_erase_disc_dispatcher(cdw_task_t *task, cdw_disc_t *disc);



/**
   \brief Check if blanking is possible, show blanking wizard and run the process

   Check if disc is ready for blanking, do other necessary checks, display
   blankingwizard and then call blank_dispatcher() function that will take
   care of blanking.

   \return CDW_OK if command finished successfully
   \return CDW_NO if some preconditions were not met
   \return CDW_CANCEL if user cancelled blanking
   \return CDW_GEN_ERROR if some problems occurred
*/
cdw_rv_t cdw_erase_disc(void)
{
	cdw_rv_t crv = cdw_read_disc_info();
	/* FIXME: if disc is erasable, but due to some problems with disc
	   file system cdio module can't read full meta information from
	   the disc, cdw will be unable to erase the disc: no meta info
	   from "broken" disc = no possibility to erase and restore it;
	   implement getting some basic information needed by erase module:
	   knowing disc type + simple disc type (+ if disc is not empty)
	   should be enough */
	if (crv != CDW_OK) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Cannot get metainformation from disc."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		cdw_main_ui_main_window_wrefresh();
		return CDW_NO;
	}

	cdw_disc_t *current_disc = cdw_disc_get();
	cdw_main_ui_disc_info_view_display_data(current_disc);

	/* check if disc is in drive and if is erasable */
	crv = cdw_erase_disc_check_preconditions(current_disc);
	if (crv != CDW_OK) {
		return crv;
	}

	CDW_TASK_CREATE_TASK(erase_task, CDW_TASK_ERASE_DISC);
	/* check which tool should be used for given task on given disc type  */
	crv = cdw_task_select_tools_for_task(&erase_task, current_disc);
	if (crv != CDW_OK) {
		return CDW_GEN_ERROR;
	}

	/* the function will set task->blanking.format_mode for DVD-RW */
	cdw_rv_t decision = cdw_erase_wizard(&erase_task, current_disc);

	if (decision == CDW_OK) {
		cdw_main_ui_main_window_wrefresh();

		if (erase_task.erase.tool.id == CDW_TOOL_DVD_RW_FORMAT) {
			if (current_disc->type == CDW_DVD_RW
			    || current_disc->type == CDW_DVD_RW_SEQ
			    || current_disc->type == CDW_DVD_RW_RES) {

				cdw_assert (erase_task.erase.format_mode != CDW_ERASE_DVD_RW_FORMAT_NONE,
					    "ERROR: correct format mode is not set!");
			}
		}

		/* 2TRANS: this is title of process window;
		   "disc" is "optical disc" */
		crv = cdw_processwin_create(_("Erase disc"),
					    /* 2TRANS: this is message in
					       dialog window;
					       "disc" is "optical disc" */
					    _("Erasing disc"), false);

		cdw_rv_t command_return = cdw_erase_disc_dispatcher(&erase_task, current_disc);

		/* this function sets task.success according to task->tool_status,
		   and resets task->tool_status */
		cdw_task_check_tool_status(&erase_task);


		cdw_rv_t retval = CDW_OK;
		if (command_return == CDW_OK) {
			const char *drive = cdw_drive_get_drive_fullpath();
			cdw_drive_reload_tray_with_ui_update(drive, true);
			retval = CDW_OK;
		} else {
			retval = CDW_GEN_ERROR;
		}
		/* after erasing disc kernel may print to console information
		   about problems with reading data from empty disc, and this
		   information will disrupt cdw ui (main window + processwin);
		   reload_tray_with_ui_update() refreshes only disc info area,
		   let's refresh main window + process window */
		cdw_main_ui_main_window_wrefresh();
		cdw_processwin_force_refresh();

		if (command_return == CDW_OK) {
			/* TODO: find a way to display time of blanking */
			/* 2TRANS: this is message in dialog window:
			   disc is now erased */
			cdw_processwin_destroy(_("Erasing of disc finished successfully"), true);
		} else if (command_return == CDW_NO
			   || command_return == CDW_GEN_ERROR) {

			/* 2TRANS: this is message in dialog window:
			   cdw called cdrecord to perform erasing, but
			   most probably it failed and disc was not erased. */
			cdw_processwin_destroy(_("Erasing of disc probably failed. Please check your disc."), true);
		} else {
			/* 2TRANS: this is message printed in process window */
			cdw_processwin_destroy("erasing was not performed", true);
		}
		/* 2TRANS: this is title of dialog window, displaying
		   messages from program blanking disc */
		after_event(_("\"Erase\" log"), 1);
		return retval;

	} else if (decision == CDW_CANCEL) { /* selection of blanking method was cancelled by user */
		return CDW_CANCEL;
	} else {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window, user
				      cancelled erasing media by pressing Escape key */
				   _("Some error occurred when attempting to erase disc."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_GEN_ERROR;
	}
}





/**
   \brief Check which method of blanking should be used, call appropriate blanking function

   This function checks task->tool to decide which tool should handle erasing,
   and then calls appropriate blanking function.

   \param task - variable describing current task

   \return CDW_GEN_ERROR if current disc type is not erasable by cdw
   \return other, blanking-function-specific values
*/
cdw_rv_t cdw_erase_disc_dispatcher(cdw_task_t *task, cdw_disc_t *disc)
{
	if (disc->type != CDW_DVD_RWP) {
		if (task->erase.erase_mode == CDW_ERASE_MODE_FAST) {
			/* 2TRANS: this is message in process window,
			   it says about fast mode of blanking */
			cdw_processwin_display_main_info(_("Erasing in \"fast\" mode"));
		} else {
			/* 2TRANS: this is message in process window -
			   erasing a disc will be a time-consuming task */
			cdw_processwin_display_main_info(_("Erasing in \"all\" mode (long time)"));
		}
	} else { /* no fast / all parameter for erasing DVD+RW */
		/* 2TRANS: this is message in process window;
		   "disc" is optical disc */
		cdw_processwin_display_main_info(_("Erasing of disc in progress"));
	}

	/* at this point we only have to check which tool was selected by
	   external_tools_select_tool_for_task() to do the job; the function
	   made the selection taking disc type into consideration */
	if (task->erase.tool.id == CDW_TOOL_CDRECORD) {

		cdw_sdm ("INFO: dispatching to cdrecord\n");
		/* 2TRANS: this is message in progress window, displayed when
		   cdrecord blanks disc - no information about progress of
		   erasing is provided */
		cdw_processwin_display_sub_info(_("No progress info will be displayed"));
		return cdw_cdrecord_run_task(task, disc);

	} else if (task->erase.tool.id == CDW_TOOL_DVD_RW_FORMAT) {

		cdw_sdm ("INFO: dispatching to run_command_dvd_rw_format_blank()\n");
		cdw_processwin_add_progress_bar();
		/* 2TRANS: this is message displayed in process window */
		cdw_processwin_display_sub_info(_("Erasing with dvd+rw-format"));
		return cdw_dvd_rw_format_run_task(task, disc);
#if 0
	} else if (task->erase.tool.id == CDW_TOOL_GROWISOFS) {

		cdw_sdm ("INFO: dispatching to run_command_growisofs_blank_dvd()\n");
		/* 2TRANS: this is message displayed in process window */
		cdw_processwin_display_sub_info(_("Erasing with growisofs"));
		return run_command_growisofs_blank_dvd(task);
#endif
	} else {
		cdw_assert (0, "ERROR: incorrect blanking tool id: %d\n", task->erase.tool.id);
		return CDW_GEN_ERROR;
	}
}





cdw_rv_t cdw_erase_disc_check_preconditions(cdw_disc_t *current_disc)
{
	if (current_disc->type == CDW_DISC_TYPE_UNKNOWN) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window:
				      disc is not recognized as supported by
				      currently selected tools */
				   _("Can't recognize disc in drive. Try changing tools family in Configuration -> Tools or use different disc type."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

		return CDW_NO;
	}

	if (current_disc->type_erasable == CDW_TRUE) {
		return CDW_OK;

	} else if (current_disc->type_erasable == CDW_FALSE) {

		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message dialog window, user
				      tries to erase disc that is not erasable */
				   _("Disc in drive is not erasable. Cannot erase it."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_NO;
	} else { /* CDW_DISC_ERASABLE_UNKNOWN || CDW_DISC_ERASABLE_ERROR */
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Warning"),
				   /* 2TRANS: this is message dialog window, user
				      tries to erase disc that may be not erasable */
				   _("Type of disc in drive is unknown. Cannot erase it."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_NO;

	}
}

