/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#define _BSD_SOURCE /* lstat() */

#include <sys/stat.h>
#include <errno.h>
#include <unistd.h>

#include "main.h"
#include "cdw_config.h"
#include "cdw_task.h"
#include "cdw_widgets.h"
#include "gettext.h"
#include "cdw_debug.h"
#include "cdw_burn_disc.h"
#include "cdw_write_wizard.h"

#include "cdw_utils.h"
#include "cdw_drive.h"
#include "cdw_file_picker.h"
#include "cdw_file_manager.h"
#include "cdw_cdrecord.h"
#include "cdw_growisofs.h"
#include "cdw_md5sum.h"
#include "cdw_logging.h"
#include "cdw_main_window.h"
#include "cdw_iso9660.h"
#include "cdw_ext_tools.h"
#include "cdw_config_ui.h"
#include "cdw_processwin.h"
#include "cdw_read_disc_info.h"

extern cdw_config_t global_config;      /* variable holding cdw configuration */

static cdw_rv_t cdw_burn_disc_dispatcher(cdw_task_t *parent_task, cdw_disc_t *disc);
static bool cdw_burn_disc_is_disc_ready(void);
static bool cdw_burn_disc_get_source(cdw_task_t *burn_task);
static void cdw_burn_disc_release_source(cdw_task_t *task);


/**
   \brief Check if everything is ready for writing to optical disc and then perform it

   Function does pre-writing checks, displays write wizard and
   calls burn_dispatcher() to do the job.

   Supported tasks are writing directly to disc and writing image to disc.
   Supported disc types are data cd and dvd.
   When combined the two things above we have following tasks:
    - write files to dvd
    - write image to dvd
    - write files to cd
    - write image to cd

   The decision is based on value of burn_task->source and type of
   disc. Disc type is checked during this function call, before performing
   task.

   Function checks if image is available or if any files are selected for
   writing.

   \param task_id CDW_TASK_BURN_FROM_IMAGE or CDW_TASK_BURN_FROM_FILES

   \return CDW_OK on success
   \return CDW_NO if some preconditions were not met
   \return CDW_CANCEL if user cancels writing
   \return CDW_GEN_ERROR on failure or some error
*/
cdw_rv_t cdw_burn_disc(int task_id)
{
	cdw_assert (task_id == CDW_TASK_BURN_FROM_FILES || task_id == CDW_TASK_BURN_FROM_IMAGE,
		    "ERROR: called the function with incorrect task id %d\n", task_id);

	CDW_TASK_CREATE_TASK(burn_task, task_id);

	bool source_available = cdw_burn_disc_get_source(&burn_task);
	if (! source_available) {
		return CDW_NO;
	}

	/* check if there is a disc in the drive and if it is writable */
	if (! cdw_burn_disc_is_disc_ready()) {
		cdw_burn_disc_release_source(&burn_task);
		return CDW_NO;
	}

	cdw_disc_t *current_disc = cdw_disc_get();

	/* check if there is any tool that can be used for a task;
	   it is important to call this function before resolving
	   writing modes, because the resolver needs information
	   on both disc and tool selected for task */
	cdw_rv_t ts = cdw_task_select_tools_for_task(&burn_task, current_disc);
	if (ts != CDW_OK) {
		cdw_vdm ("ERROR: failed to get tool for task TASK_BURN\n");
		cdw_burn_disc_release_source(&burn_task);
		return CDW_GEN_ERROR;
	} else {
		if (task_id == CDW_TASK_BURN_FROM_IMAGE
		    && burn_task.burn.tool.id == CDW_TOOL_CDRECORD
		    && cdw_ext_tools_is_cdrecord_wodim()) {

			/* sector = 2048 bytes */
			long sectors = cdw_iso_image_get_n_sectors();
			double gigabytes = ((float) sectors * 2.0) / (1024.0 * 1024.0);
			cdw_vdm ("INFO: sectors = %ld -> %fGB\n",
				 sectors, gigabytes);
			if (sectors > 2 * 1024 * 1024) {
				cdw_vdm ("INFO: large image: %ld sectors -> %fGB\n",
					 sectors, gigabytes);
				/* 2TRANS: this is the title of dialog window */
				cdw_rv_t crv = cdw_buttons_dialog(_("Warning"),
								  /* 2TRANS: this is the message in the dialog window */
								  _("You are attempting to use wodim to burn large image. This task will probably fail. You should try using another tool (cdrecord or growisofs) for this task. Do you want to continue?"),
								  CDW_BUTTONS_OK_CANCEL, CDW_COLORS_ERROR);
				if (crv != CDW_OK) {
					cdw_burn_disc_release_source(&burn_task);
					return CDW_CANCEL;
				} else {
					/* 2TRANS: this is message printed
					   in log file; %ld is size of
					   ISO9660 image (in sectors) */
					cdw_logging_write(_("Attempting to burn ISO image of size %ld sectors with wodim, this may fail...\n"), sectors);
				}
			}
		}
	}

	/* check if current state of disc allows us to write what we want
	   (iso file or files from disc), and specify allowed modes of writing
	   (start multisession, append and close, etc.) */
	cdw_rv_t crv = cdw_task_resolve_allowed_writing_modes(&burn_task, current_disc);
	if (crv != CDW_OK) { /* CDW_GEN_ERROR or CDW_NO */
		cdw_burn_disc_release_source(&burn_task);
		return CDW_NO;
	}

	if (burn_task.id == CDW_TASK_BURN_FROM_FILES) {
		/* ask user for volume name (depends on checkbox in configuration) */
		crv = cdw_config_ui_conditional_volume_label_dialog();
		if (crv != CDW_OK) {
			cdw_vdm ("INFO: cancelled at creating volume label\n");
			cdw_burn_disc_release_source(&burn_task);
			/* e.g. "Escape" key */
			return CDW_NO;
		}
	}

	cdw_rv_t decision = write_wizard(&burn_task, current_disc);

	if (decision != CDW_OK) {
		cdw_burn_disc_release_source(&burn_task);
		if (decision == CDW_GEN_ERROR) {
			/* 2TRANS: this is the title of dialog window */
			cdw_buttons_dialog(_("Error"),
					   /* 2TRANS: this is the message in the dialog window */
					   _("There were some problems when cdw tried to prepare for writing. Try restarting cdw."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		} else { /* CDW_CANCEL */
			;
		}
		return decision;
	}
	if (current_disc->state_empty != CDW_TRUE) {
		if (burn_task.burn.session_mode == CDW_SESSION_MODE_START_MULTI
		    || burn_task.burn.session_mode == CDW_SESSION_MODE_CREATE_SINGLE) {

			/* 2TRANS: this is the title of dialog window */
			crv = cdw_buttons_dialog(_("Warning"),
						 /* 2TRANS: this is the message in the dialog window */
						 _("The disc is not blank, you will overwrite its current content. Continue?"),
						 CDW_BUTTONS_OK_CANCEL, CDW_COLORS_WARNING);
			if (crv != CDW_OK) {
				/* user decided not to write to disc with
				   "start multi" or "create single" session mode
				   so let's abandon burning completely */
				cdw_burn_disc_release_source(&burn_task);
				return CDW_CANCEL;
			}
		}
	}

	cdw_main_ui_main_window_wrefresh();

	cdw_assert (burn_task.id == CDW_TASK_BURN_FROM_FILES || burn_task.id == CDW_TASK_BURN_FROM_IMAGE,
		    "ERROR: task id has changed\n");

	/* 2TRANS: this is a title of process window;
	   "disc" is "optical disc" */
	cdw_processwin_create(_("Writing to disc"),
			      /* 2TRANS: this is message in process window;
				 "disc" is "optical disc" */
			      _("Writing to optical disc"), true);

	/* burn_dispatcher() calls actual function performing burning */
	cdw_rv_t command_return = cdw_burn_disc_dispatcher(&burn_task, current_disc);
	/* writing has been finished, erase information that was updated
	   (and up to date) only during actual writing; displaying it
	   after this point may be misleading - user might think that
	   it is somehow still valid */
	cdw_processwin_erase_fifo_and_speed();
	cdw_processwin_erase_eta();

	cdw_burn_disc_release_source(&burn_task);

	/* this function sets task.success according to task->tool_status,
	   and resets task->tool_status */
	cdw_task_check_tool_status(&burn_task);
	cdw_vdm ("INFO: burn dispatcher returns %s\n", cdw_utils_get_crv_label(crv));
	cdw_vdm ("%s: burn function sets burn_task.tool_status.ok to %s\n",
		 burn_task.tool_status.ok ? "INFO" : "ERROR",
		 burn_task.tool_status.ok ? "true" : "false");

	if (command_return == CDW_OK && burn_task.tool_status.ok) {
		const char *drive = cdw_drive_get_drive_fullpath();
		if (burn_task.burn.verify) {
			cdw_vdm ("INFO: \"verify burn\" is true, attempting verification\n");

			cdw_drive_reload_tray_with_ui_update(drive, false);
			/* this is necessary because of dispatches in pipe_regexp.c */
			burn_task.id = CDW_TASK_CHECK_MD5SUM;
			burn_task.verify.iso_fullpath = global_config.iso_image_full_path;
			/* we are passing a copy of burn_task here because
			   tools are already configured in this variable */
			cdw_rv_t md5 = cdw_verify(&burn_task);
			if (md5 != CDW_OK) {
				/* TODO: check burn_task.tool_status.ok */
				cdw_vdm ("ERROR: md5sum check function returns !CDW_OK\n");
			}
		} else {
			cdw_vdm ("INFO: \"verify burn\" is false, not attempting to verify\n");
		}
		cdw_sdm ("INFO: attempting to reload tray with ui update\n");
		cdw_drive_reload_tray_with_ui_update(drive, true);
		cdw_sdm ("INFO: reloading tray with ui update - done\n");
	} else {
		/* incorrect situation, already covered by debug messages above */
		command_return = CDW_NO;
	}

	if (command_return == CDW_OK) {
		/* 2TRANS: this is message in dialog window:
		   operation finished with unknown result, probably success */
		cdw_processwin_destroy(_("Writing finished"), true);
	} else {
		cdw_vdm ("ERROR: command_return != CDW_OK\n");
		/* 2TRANS: this is message in dialog window:
		   operation finished (most probably) with error */
		cdw_processwin_destroy(_("Writing probably failed"), true);
	}

	/* 2TRANS: this is the title of the dialog window, displaying messages
	   from the program writing iso image or selected files to CD disc */
	cdw_logging_display_log_conditional(_("\"Write\" log"));

	if (command_return == CDW_OK && burn_task.tool_status.ok) {
		cdw_sdm ("INFO: burning dispatcher returns CDW_OK and burn_task.tool_status.ok == true\n");
		return CDW_OK;
	} else {
		/* both command_return and burn_task.tool_status.ok were
		   checked and signaled before, so just use a generic message here */
		cdw_vdm ("ERROR: returning CDW_GEN_ERROR due to errors reported above\n");
		return CDW_GEN_ERROR;
	}
}





/**
   \brief Pass control to cdrecord or growisofs code to perform burning

   This function is just a dispatcher, and calls cdw_cdrecord_run_task()
   or cdw_growisofs_run_task(), depending on task->burn.tool.id. You
   should call cdw_task_select_tools_for_task() first to set the value of
   that variable.

   The function checks all important assertions before doing actual
   dispatching, but only in debug build.

   The function will be even more useful when you add another tool
   that does burning, e.g. xorriso.

   \param task - variable describing current task
   \param disc - variable describing current disc

   \return return value from cdrecord or growisofs cdw_*_run_task() function
*/
cdw_rv_t cdw_burn_disc_dispatcher(cdw_task_t *task, cdw_disc_t *disc)
{
	cdw_assert (task->burn.disc_mode != CDW_DISC_MODE_INIT, "ERROR: calling the function with uninitialized disc mode\n");
	cdw_assert (task->burn.session_mode != CDW_SESSION_MODE_INIT, "ERROR: calling the function with uninitialized session mode\n");
	cdw_assert (task->id == CDW_TASK_BURN_FROM_FILES || task->id == CDW_TASK_BURN_FROM_IMAGE,
		    "ERROR: incorrect task id %d\n", task->id);
	cdw_assert (task->burn.tool.id == CDW_TOOL_CDRECORD || task->burn.tool.id == CDW_TOOL_GROWISOFS,
		    "ERROR: incorrect task->main_tool_id %d\n", task->burn.tool.id);

	if (task->id == CDW_TASK_BURN_FROM_FILES) {
		if (disc->simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
			/* 2TRANS: this is message in process window:
			   writing selected files to CD is in progress */
			cdw_processwin_display_main_info(_("Writing files to CD..."));
		} else {
			/* 2TRANS: this is message in process window:
			   writing selected files to DVD is in progress */
			cdw_processwin_display_main_info(_("Writing files to DVD..."));
		}
	} else {
		if (disc->simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
			/* 2TRANS: this is message in process window -
			   writing iso image to CD is in progress */
			cdw_processwin_display_main_info(_("Writing image to CD..."));
		} else {
			/* 2TRANS: this is message in process window -
			   writing iso image to DVD is in progress */
			cdw_processwin_display_main_info(_("Writing image to DVD..."));
		}
	}

	/* at this point we don't care what kind of disc we have,
	   because "external tools" module already checked disc type and
	   decided which tool to use, e.g. to use cdrecord or growisofs for
	   writing to DVD; so we dispatch basing only on tool id */
	cdw_rv_t crv = CDW_OK;
	if (task->burn.tool.id == CDW_TOOL_CDRECORD) {
		cdw_vdm ("INFO: dispatching \"burn\" task to cdrecord\n");
		crv = cdw_cdrecord_run_task(task, disc);
	} else if (task->burn.tool.id == CDW_TOOL_GROWISOFS) {
		cdw_vdm ("INFO: dispatching \"burn\" task to growisofs\n");
		crv = cdw_growisofs_run_task(task, disc);
	} else {
		cdw_assert (0, "ERROR: wrong tool id '%d' for the job (1)\n", task->burn.tool.id);
	}

	/* it is too early to check task->tool_status.ok */

	return crv;
}




bool cdw_burn_disc_get_source(cdw_task_t *burn_task)
{
	bool source_available = false;

	if (burn_task->id == CDW_TASK_BURN_FROM_IMAGE) {
		/* 2TRANS: this is the title of dialog window */
		cdw_rv_t crv = cdw_fs_ui_file_picker(_("Path to iso image"),
						     /* 2TRANS: this is the message in the dialog window;
						     "writing" means writing to optical disc */
						     _("Please enter FULL path to an existing iso image file for writing:"),
						     &(global_config.iso_image_full_path),
						     CDW_FS_FILE, R_OK, CDW_FS_EXISTING);

		cdw_main_ui_main_window_wrefresh();
		if (crv == CDW_OK) {
			burn_task->burn.data_size_mb = cdw_iso_image_get_size_mb();
			source_available = true;
		}
	} else if (burn_task->id == CDW_TASK_BURN_FROM_FILES) {
		cdw_rv_t crv = cdw_file_manager_create_graftpoints_file();
		if (crv == CDW_OK) {
			long long size = cdw_selected_files_get_size();
			burn_task->burn.data_size_mb = (((double) size) / 1024.0) / 1024.0;

			source_available = true;
			if (cdw_selected_files_file_over_4gb_present()) {
				cdw_vdm ("WARNING: selected file > 4 GB\n");
				/* 2TRANS: this is title of dialog window */
				cdw_buttons_dialog(_("Warning"),
						   /* 2TRANS: this is the message in the dialog window:
						      disc is not recognized as supported by
						      currently selected tools */
						   _("One of selected files has a size over 4GB. Choose your tools wisely (Configuration -> Tools), otherwise writing to disc will give incorrect results."),
						   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
			} else {
				cdw_sdm ("INFO: no file > 4 GB\n");
			}
		} else if (crv == CDW_NO) {
			; /* probably no selected files */
		} else {
			cdw_vdm ("ERROR: failed to correctly create graftpoints file\n");
		}
	} else {
		; /* covered by caller's assert */
	}

	return source_available;
}





void cdw_burn_disc_release_source(cdw_task_t *task)
{
	if (task->id == CDW_TASK_BURN_FROM_FILES) {
		cdw_file_manager_delete_graftpoints_file();
	}

	return;
}





bool cdw_burn_disc_is_disc_ready(void)
{
	cdw_rv_t crv = cdw_read_disc_info();
	if (crv != CDW_OK) {
		return false;
	}

	cdw_disc_t *current_disc = cdw_disc_get();
	//cdw_main_ui_disc_info_view_display_data(current_disc);

	if (current_disc->type == CDW_DISC_TYPE_UNKNOWN) {
		/* 2TRANS: this is the title of the dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is the message in the dialog window:
				      disc is not recognized as supported by
				      currently selected tools */
				   _("Can't recognize disc in drive. Try changing tools family in Configuration -> Tools or use different disc type."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

		return false;
	}

	if (current_disc->type_writable != CDW_TRUE) {
		/* CD-AUDIO, DVD-ROM etc. */
		/* 2TRANS: this is the title of the dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is the message in the dialog window - disc is not writable */
				   _("Disc in drive is read-only."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

		return false;
	}


	if (current_disc->state_writable == CDW_UNKNOWN) {
		/* perhaps device path is invalid? */

		/* 2TRANS: this is the title of the dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is the message in the dialog window: for some
				      reason cdw cannot read disc metainfo; this may be a
				      problem with the hardware configuration or with the family
				      of tools (cdrtools / dvd+rw-tools) used for the task */
				   _("Cannot get full information about disc. Please check configuration of hardware or tools. (Are you using wodim?)."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return false;
	} else if (current_disc->state_writable == CDW_FALSE) {

		/* 2TRANS: this is the title of the dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is the message in the dialog window - user tries to write to a closed disc */
				   _("Cannot write to this disc anymore, disc is closed."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return false;
	} else {
		/* current_disc->state_appendable == CDW_DISC_APPENDABLE_YES */
		if (current_disc->type == CDW_DVD_RP_DL
		    && global_config.support_dvd_rp_dl) {

			/* 2TRANS: this is the title of the dialog window */
			cdw_rv_t crv2 = cdw_buttons_dialog(_("Information"),
							   /* 2TRANS: this is the message in the dialog window */
							  _("You are attempting to burn to a DVD+R DL disc. You REALLY need to use high quality discs and have a bit of luck to correctly burn data to DVD+R DL. Additionally cdw offers very limited support for DVD+RD DL discs. Continue?"),
							  CDW_BUTTONS_OK_CANCEL, CDW_COLORS_DIALOG);
			if (crv2 == CDW_OK) {
				return true;
			} else {
				return false;
			}
		}
		return true;
	}
}



