#ifndef H_CDW_DISC
#define H_CDW_DISC

#include <stdbool.h>
#include <stddef.h> /* size_t */

#include <cdio/iso9660.h>

#include "main.h"
#include "cdw_cdio.h"



/* this value is used as size of table used to hold speed dropdown labels AND
   size of table used to store values of CD / DVD writing speeds;
   values from the second table are used to initialize the first table */
#define CDW_DISC_N_SPEEDS_MAX 52



/** \brief length of label describing disc type, visible in "Disc info" view */
#define CDW_DISC_TYPE_LABEL_LEN 15

typedef enum {
	CDW_DISC_TYPE_UNKNOWN = 0,

	CDW_DISC_NONE,  /* no disc in drive */

	CDW_CD_AUDIO,

	CDW_CD_R,       /* CD-R */
	CDW_CD_RW,      /* CD-RW */
	CDW_CD_ROM,     /* CD-ROM */

	CDW_DVD_R,      /* generic DVD-R */
	CDW_DVD_R_SEQ,  /* DVD-R in Incremental Sequential mode */
	CDW_DVD_R_RES,  /* DVD-R in Restricted Overwrite (I think that I got overzealous when creating this value) */
	CDW_DVD_RP,     /* DVD+R (R Plus) */

	CDW_DVD_RW,     /* generic DVD-RW */
	CDW_DVD_RW_SEQ, /* DVD-RW in Incremental Sequential mode */
	CDW_DVD_RW_RES, /* DVD-RW in Restricted Overwrite */

	CDW_DVD_RWP,    /* DVD+RW (RW Plus) */
	CDW_DVD_ROM,    /* DVD-ROM, e.g. video DVD */

	CDW_DVD_RP_DL   /* dual layer DVD+R (R Plus Dual Layer) */
} cdw_disc_type_t;




typedef struct {
	/* two parameters describing current state of disc, reported by
	   cdrecord and used when writing with cdrecord */
	long int last_sess_start;
	long int next_sess_start;

	/* did cdrecord manage to read toc? by default "true",
	   may be set to "false" by pipe regexp code (-toc call) */
	bool has_toc;
} cdw_cdrecord_info_t;


/* applicable to dvd (and maybe blu-ray) discs only */
typedef struct {
	/* table of starting sectors of tracks on DVD disc */
	// long int track_address[100];
	// long int n_tracks;

	ssize_t read_capacity; /* number of 2048-bytes blocks, can be -1 as initial value */
	char disc_status; /* first letter of "blank", "appendable", "complete", "other" */
} cdw_dvd_rw_mediainfo_info_t;


typedef struct {
	/* write speeds accepted by disc and drive,
	   values reported by cdrecord or dvd+rw-mediainfo */
	int speeds[CDW_DISC_N_SPEEDS_MAX];
	/* number of write speeds available for given disc and drive */
	int n_speeds;
	int drive_default_speed;
	int drive_max_speed;
} cdw_disc_speeds_info_t;




/* variable storing information about disc currently in drive */
typedef struct {
	/* simple type of disc currently in drive:
	   CD, DVD or UNKNOWN - sometimes knowing this is enough */
	cdw_disc_simple_type_t simple_type;
	/* human-readable representation of disc simple type */
	char simple_type_label[4 + 1];

	/* precise type of disc currently in drive */
	cdw_disc_type_t type;
	/* human-readable representation of disc type */
	char type_label[CDW_DISC_TYPE_LABEL_LEN + 1];

	discmode_t cdio_disc_mode;

	int n_tracks;

	int state_empty;
	int state_writable;
	int type_writable;
	int type_erasable;

	cdw_optical_fs_t fs;

	cdw_disc_speeds_info_t write_speeds;

	int media_info_source; /* CDW_TOOL_NONE, CDW_TOOL_CDRECORD, CDW_TOOL_DVD_RW_MEDIAINFO */

	cdw_cdrecord_info_t cdrecord_info;
	cdw_dvd_rw_mediainfo_info_t dvd_rw_mediainfo_info;
} cdw_disc_t;


/* this function only serves to initialize some labels;
   the labels don't have to be free()d, so there is no cdw_disc_clean() */
void cdw_disc_init(void);

cdw_disc_t *cdw_disc_get(void);
cdw_rv_t cdw_disc_resolve(cdw_disc_t *disc);
/* reset all values in given disc */
void cdw_disc_reset(cdw_disc_t *disc);
void cdw_disc_debug_print_states_and_types(cdw_disc_t *disc);
const char *cdw_disc_check_if_is_readable(cdw_disc_t *disc);

/* one and only function to read metadata from optical disc to 'cdw_disc_t' data structure */
cdw_rv_t cdw_disc_get_meta_info(cdw_disc_t *disc);


/* use when you need to get writing speed index based on config.speed_range */
int cdw_disc_get_initial_write_speed_index(cdw_disc_t *disc);




#endif /* H_CDW_DISC */

