/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <math.h> /* floor() */

#include <libintl.h>
#include <locale.h>
#include <sys/types.h>
#include <ncursesw/ncurses.h>

#include "cdw_ui.h"
#include "options.h"
#include "gettext.h"
#include "color.h"
#include "log.h"
#include "config.h"
#include "main.h"
#include "cdw_widgets.h"
#include "processwin.h"




/* ******************************** */
/* some processwin layout constants */
/* ******************************** */
#define PROCESSWIN_COLS 40
#define PROCESSWIN_ROWS 12
/* first goes primary information line, telling user what happens now */
#define PROCESSWIN_INFO_TEXT_1_ROW 2
/* then some additional text info (e.g. "press any key" or "fixating") */
#define PROCESSWIN_INFO_TEXT_2_ROW 3
#define PROCESSWIN_PERMANENT_INFO_TEXT_ROW 4
/* some data that might be most important to the user */
#define PROCESSWIN_ETA_ROW 5
#define PROCESSWIN_AMOUNT_ROW 6
/* some visualisation - less important than numeric data presented above,
 * because user can get data presented on progress bar from percentage
 * information - both are calculated using the same data, but progress bar
 * is less accurate; "CENTRAL_ROW" means that there is subwindow border
 * around this one-line progress bar*/
#define PROCESSWIN_PROGRESSBAR_CENTRAL_ROW 8
/* boring stuff at the bottom - common user might not care */
#define PROCESSWIN_FIFO_ROW 10






/* this string is used to erease parts of processwin - it has to be
 * filled with '\0' first */
char processwin_empty_string[PROCESSWIN_COLS - 2];

/* This variable keeps track of changes of percentage of task done. Progressbar
   and percentage value is updated only when the prececentage changes,
   avoiding useless refreshes. It must be set to zero every time new
   processwin is created */
float static_percent;

WINDOW *processwin = NULL;
extern WINDOW *mainwin;


/**
 * Create UI window in which process progress of external process is shown
 *
 * \param bool is_process
 * \param char *window_title - tilte of processwin window
 * \param char *window_label - text displayed somewhere in window
 */
void processwin_create(char *window_title, char *window_label, bool is_process)
{
	memset(processwin_empty_string, ' ', PROCESSWIN_COLS - 2);
	*(processwin_empty_string + PROCESSWIN_COLS - 2) = '\0';

	static_percent = 0;

	processwin = newwin(PROCESSWIN_ROWS, PROCESSWIN_COLS,
			    (LINES - PROCESSWIN_ROWS) / 2, (COLS - PROCESSWIN_COLS) / 2);
	wbkgd(processwin, COLOR_PAIR(2));
	werase(processwin);
	keypad(processwin, true);
	nodelay(processwin, false);
	// notimeout(processwin, true); /* now wgetch() _really_ waits for input in processwin, without timeouts */

	/* third argument is displayed on bottom of window, so this is not a window label */
	nice_box(processwin, window_title, "");

	if (is_process) {
		/* place for progress bar */
		/* WINDOW *derwin(WINDOW *orig, int nlines, int ncols, int begin_y, int begin_x); */
		/* 27 = 25 + 2: 2 chars for borders and 25 chars for
		progressbar itself, that gives 4% per char */
		box(derwin(processwin, 3, 27,
		    PROCESSWIN_PROGRESSBAR_CENTRAL_ROW - 1,
		    floor((PROCESSWIN_COLS - 27) / 2)),
	            0, 0);
	}

	/* display only primary label, secondary is empty */
	processwin_display_text_info(window_label, "");
	wrefresh(processwin);

	return;
}





/**
 * Destroy UI window created by processwin_create()
 *
 * This is just wrapper for simple destroying of curses window
 */
void processwin_destroy(char *status_string, bool wait)
{
	if (wait) {
		/* 2TRANS: message displayed in progress window: user must press
		any key to close the window after task is completed */
		processwin_display_text_info(status_string, _("Press any key"));
		wrefresh(processwin); /* refresh so user sees above message */
		wgetch(processwin);
	}

	// wborder(processwin, ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ');
	// wrefresh(processwin);
	delwin(processwin);

	main_win_refresh_part(PROCESSWIN_ROWS, PROCESSWIN_COLS,
			      (LINES - PROCESSWIN_ROWS) / 2, (COLS - PROCESSWIN_COLS) / 2);
	return;
}




void processwin_display_eta(char *eta_string)
{
	if (!strcmp(eta_string, "")) { /* empty string - empty info area */
		mvwprintw(processwin, PROCESSWIN_ETA_ROW, 1, "%s", processwin_empty_string);
	} else {
		mvwprintw(processwin, PROCESSWIN_ETA_ROW, 1, "%s", processwin_empty_string);
		mvwprintw(processwin, PROCESSWIN_ETA_ROW, (PROCESSWIN_COLS/2) - (strlen(eta_string) / 2), "%s", eta_string);
	}

	wrefresh(processwin);

	return;
}




/**
 * Draw progress bar in (global variable) processwin
 *
 * \param int value - length of progress bar to draw
 * \param int color1
 * \param int color2
 * \param char c - character to be used to draw bar
 */
void processwin_draw_progress_bar(int value, int color1, int color2, char c)
{
	int i;

	/* (PROCESSWIN_COLS - 27) / 2 is a start column of progress bar box;
	27 = 2 + 25: 2 chars for borders and 25 for bar (4% per char) */
	int start_col = floor((PROCESSWIN_COLS - 27) / 2 ) + 1;

	for (i = 0; i <= value; i++) {
		wattrset(processwin, COLOR_PAIR(color1));
		mvwaddch(processwin,
			 PROCESSWIN_PROGRESSBAR_CENTRAL_ROW, start_col + i, c);
		wattrset(processwin, COLOR_PAIR(color2));
	}

	return;
}




/**
 * Show (in processwin window) information about progress of process. Do this only if necessary.
 *
 * Check if value describing current status (in terms of percentage of task
 * done) has changed. If yes, then update processwin with new information
 * (which requires call of ncurses wrefresh()). If not then do nothing.
 *
 * \param current - current status of task
 * \param total - total size of task
 * \param char *current_value_string - string containing data/information
 *                                   prepared by caller (can be NULL or empty)
 */
void conditional_processwin_display_progress(long current, long total, char *current_value_string)
{
	float new_static_percent = (float) current/total;

	if (new_static_percent != static_percent) {
		static_percent = new_static_percent;
		processwin_display_progress(current, total, current_value_string);
	}

	return;
}




/**
 * Show (in processwin window) information about progress of process.
 *
 * Show to user updated information about progress of current process. This
 * can be done in two ways: by displaying prepared by caller text line and by
 * showing percentage information. Currenty there is no  method for disabling
 * any of this method, so you have to provide valid values for all three
 * arguments (although first can be NULL or empty).
 *
 * First argument can have any character, even descriptive, but since this
 * function is called processwin_display_progress(), it would be nice if this
 * argumend hold some sort of numerical information, perhaps with units of
 * measure.
 *
 * Second and third argument shoud be provided without any preprocessing,
 * because (in theory) this function will adjust and normailze the values
 * so displayed information will make sense.
 *
 * Percentage information is shown both in form of "x% done" text and in form
 * of progress bar.
 *
 * \param int current amount - value representing current state (stage) of process
 * \param int total_amount - value representing final, expected state (stage) of process
 * \param char *preformatted_status_line - string containing data/information prepared by caller (can be NULL or empty)
 */
void processwin_display_progress(int current_amount, int total_amount, char *current_value_string)
{
	bool cvs = false; /* is preformatted value string non-empty? */
	char amount_string[PROCESSWIN_COLS - 2];

	/* first check for NULLness, so strlen arg will be safe */
	if ( (current_value_string != NULL) && (strlen(current_value_string) != 0)) {
		cvs = true;
	}

	if (total_amount != 0) { /* total_amount != 0, so we can calculate percentage and display progress bar */
		int percent_done = ((current_amount * 100) / total_amount); /* FIXME - if the values are proeprocessed then multiplying will not be necessary */

		/* display percentage information and additional, preformatted string with numeric data */
		if (cvs) {
			/* 2TRANS: status message displayed in progress window: %d%% is percent amount of
			task finished, %s is previously prepared string displaying amount of task finished
			in megabytes */
			sprintf(amount_string, _("%d%% done (%s)"), percent_done, current_value_string);
		} else {
			/* 2TRANS: status message displayed in progress window: %d%% is percent amount of
			task finished */
			sprintf(amount_string, _("%d%% done"), percent_done);
		}
		mvwprintw(processwin, PROCESSWIN_AMOUNT_ROW, (PROCESSWIN_COLS/2) - (strlen(amount_string) / 2), "%s", amount_string);
		processwin_draw_progress_bar((percent_done / 4) - 1, 2, 2, ACS_BLOCK);
	} else { /* total_amount == 0, so there is no way we can show percentage information */
		if (cvs) {
			mvwprintw(processwin, PROCESSWIN_AMOUNT_ROW, (PROCESSWIN_COLS/2) - (strlen(current_value_string) / 2), "%s", current_value_string);
		}
	}

	wrefresh(processwin);

	return;
}




/**
 * Show (in processwin window) text information about status of process
 *
 * Display one or two strings with information, or erase (blank) them.
 * The two strings are independent and don't influence each other.
 * They are displayed in PROCESSWIN_INFO_TEXT_1_ROW and in
 * PROCESSWIN_INFO_TEXT_2_ROW respectively. Set an argument to NULL if you
 * don't want to modify or update given string in processwin, set it
 * to empty string if you want to erase it, or set it to pointer to a
 * valid string if you want to display the string.
 *
 * \param string1 - string displayed in PROCESSWIN_INFO_TEXT_1_ROW row
 * \param string2 - string displayed in PROCESSWIN_INFO_TEXT_2_ROW row
 */
void processwin_display_text_info(char *string1, char *string2)
{
	if (string1 == NULL) { /* no string - nothing to display/update */
		;
	} else if (!strcmp(string1, "")) { /* empty string - empty info area */
		mvwprintw(processwin, PROCESSWIN_INFO_TEXT_1_ROW, 1, "%s", processwin_empty_string);
	} else {
		mvwprintw(processwin, PROCESSWIN_INFO_TEXT_1_ROW, 1, "%s", processwin_empty_string);
		mvwprintw(processwin, PROCESSWIN_INFO_TEXT_1_ROW, (PROCESSWIN_COLS/2) - (strlen(string1) / 2), "%s", string1);
	}


	if (string2 == NULL) { /* no string - nothing to display/update */
		;
	} else if (!strcmp(string2, "")) { /* empty string - empty info area */
		mvwprintw(processwin, PROCESSWIN_INFO_TEXT_2_ROW, 1, "%s", processwin_empty_string);
	} else {
		mvwprintw(processwin, PROCESSWIN_INFO_TEXT_2_ROW, 1, "%s", processwin_empty_string);
		mvwprintw(processwin, PROCESSWIN_INFO_TEXT_2_ROW, (PROCESSWIN_COLS/2) - (strlen(string2) / 2), "%s", string2);
	}

	wrefresh(processwin);

	return;
}






/**
 * Display fifo buffer information
 *
 * This function displays test string in form of "Fifo: %d%%" in
 * process window. It is useful when writing data to a disc.
 * If fifo value is lower than 25, the text background is highlighted.
 *
 * \param fifo_level - FIFO buffer usage value (0 - 100)
 */
void processwin_display_fifo(int fifo_level)
{
	char fifo_string[25];
	/* 2TRANS: this is message displayed in progress window, describing
	FIFO buffer usage (in percents)	during some resources-consuming tasks;
	please keep size of string, pad with blanks if necessary */
	sprintf(fifo_string, _("  Fifo: %3.d%%  "), fifo_level);

	if (fifo_level <= 25) {
		wattrset(processwin, COLOR_PAIR(7));
	}
	mvwprintw(processwin, PROCESSWIN_FIFO_ROW, (PROCESSWIN_COLS/2) - (strlen(fifo_string) / 2), "%s", fifo_string);
	wattrset(processwin, COLOR_PAIR(2));

	wrefresh(processwin);

	return;
}



void processwin_refresh(void)
{
	touchwin(processwin);
	wrefresh(processwin);

	return;
}

