/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * A few small modifications (C) 2007 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * Functions stdout_regexp_*() and_stderr_regexp_*() defined in this file
 * process information that was sent via UNIX pipes from child processes that
 * do something interesting and send information about their activities to
 * stdout and stderr (which is captured in threading code). See thread.c for
 * more information about the pipes.
 *
 * Information read from pipes is then compared against set of regexps,
 * and depending on results of this comparisons one of handle_* function is
 * called.
 *
 * One of argumants of such function is matchesX - table of offset
 * pairs, each pair pointing to beginning and to end of consecutive
 * (I guess: non-white) substrings in captured buffer. Each substring (each
 * 'word' or lexem) in process output line contains some useful information,
 * for example output of wodim writing files to disc looks like this:
 *
 * "Track 02:    7 of   52 MB written (fifo 100%) [buf 100%]   4.1x.".
 *
 * Pairs of offsets allow us to grab useful information from such string,
 * e.g. using offsets pair number 3 we can extract substring '7', which is
 * information about amount of data already written to disc - quite a useful
 * information. That's why in handle_* functions such conditions appear:
 * "if (i == 3)" - here we check for offset pair number.
 *
 * Each handle_* function knows format of only one kind of data that can be
 * obtained from socket. If we want to capture and analyse this data, we have
 * to write regexp, compare our line from socket against this regexp, and
 * (if data matches regexp) call some specific handle_* function, which will
 * analyse this one line.
 *
 * If you are not sure which one of handle_* functions you need to call, use
 * current_command variable to find out what process is currently performed.
 */



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <ncursesw/ncurses.h>
#include <regex.h>
#include <nl_types.h>
#include <time.h>
#include <libintl.h>

#include "processwin.h"
#include "gettext.h"
#include "commands.h"
#include "log.h"
#include "thread.h" /* PIPE_BUFFER_SIZE */


extern char stdout_pipe_buffer[PIPE_BUFFER_SIZE + 1];
extern char stderr_pipe_buffer[PIPE_BUFFER_SIZE + 1];

/* time captured at the beginning of process */
extern time_t time0;


extern FILE *logfi;

extern WINDOW *processwin;

extern enum media_erasable_t media_erasable;
extern enum current_command_t current_command;

int prev_percent = 0;

/* small helper functions */
void eta_calculations(char *eta_string, int eta_value);
void regcomp_error_handler(int rv, int id);

/* functions used for output handling in print_stdout */
void handle_image_writing(regex_t *regex, regmatch_t *matches);
void handle_direct_writing(regex_t *regex, regmatch_t *matches);
void handle_fixating(regex_t *regex, regmatch_t *matches);
void handle_is_erasable(regex_t *regex, regmatch_t *matches);
void handle_blanking_not_supported(regex_t *regex, regmatch_t *matches);

/* functions used for output handling in print_stderr */
void handle_audio_grab(char *c5);
void handle_image_creation(regex_t *regex, regmatch_t *matches);
void handle_image_creation_100percent(regex_t *regex, regmatch_t *matches);
void handle_device_or_resource_busy(regex_t *regex, regmatch_t *matches);


regex_t *regex = NULL, *regex2 = NULL, *regex3 = NULL, *regex4 = NULL, *regex5 = NULL, *regex6 = NULL;
regmatch_t *matches = NULL, *matches2 = NULL, *matches3 = NULL, *matches4 = NULL, *matches5 = NULL, *matches6 = NULL;
regex_t *eregex = NULL, *eregex2 = NULL, *eregex3 = NULL, *eregex4 = NULL, *eregex5 = NULL;
regmatch_t *ematches = NULL, *ematches2 = NULL, *ematches3 = NULL, *ematches4 = NULL, *ematches5 = NULL;

/* regular expressions setup code */
void stdout_regexp_prepare(void)
{
	int rv;

	/* Device or resource busy. */
	regex6 = (regex_t *) calloc(1, sizeof(regex_t));
	rv = regcomp(regex6, "wodim: Device or resource busy", REG_EXTENDED);
	matches6 = (regmatch_t *) calloc(1, (regex6->re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		regcomp_error_handler(rv, 10);
	}

	/* Direct writing... */
	regex = (regex_t *) calloc(1, sizeof(regex_t));
	rv = regcomp(regex, "Track ([0-9]+): ([ ]*)([0-9]+) of ([ ]*)([0-9]+) MB written ([(])fifo([ ]*)([0-9]+)", REG_EXTENDED);
	matches = (regmatch_t *) calloc(1, (regex->re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		regcomp_error_handler(rv, 0);
	}

	/* Image writing... */
	regex3 = (regex_t *) calloc(1, sizeof(regex_t));
	rv = regcomp(regex3, "Track ([0-9]+): ([ ]*)([0-9]+) MB written ([(])fifo([ ]*)([0-9]+)", REG_EXTENDED);
	matches3 = (regmatch_t *) calloc(1, (regex3->re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		regcomp_error_handler(rv, 1);
	}

	/* Fixating... */
	regex2 = (regex_t *) calloc(1, sizeof(regex_t));
	rv = regcomp(regex2, "Fixating...", REG_EXTENDED);
	matches2 = (regmatch_t *) calloc(1, (regex2->re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		regcomp_error_handler(rv, 2);
	}

	/* Is erasable... */
	regex4 = (regex_t *) calloc(1, sizeof(regex_t));
	rv = regcomp(regex4, "Is erasable", REG_EXTENDED);
	matches4 = (regmatch_t *) calloc(1, (regex4->re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		regcomp_error_handler(rv, 3);
	}

	/* this media does not support blanking */
	regex5 = (regex_t *) calloc(1, sizeof(regex_t));
	rv = regcomp(regex5, "this media does not support blanking", REG_EXTENDED);
	matches5 = (regmatch_t *) calloc(1, (regex5->re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		regcomp_error_handler(rv, 4);
	}

	return;
}



void stdout_regexp_execute(void)
{
	int rv1, rv2, rv3, rv4, rv5, rv6;

	/* Case 5: Device or resource busy. */
	rv6 = regexec(regex6, stdout_pipe_buffer, regex6->re_nsub + 1, matches6, 0);
	if (rv6 == 0) {
		/* Device or resource busy */
		handle_device_or_resource_busy(regex6, matches6);
	}
	/* case 1: Direct writing */
	rv1 = regexec(regex, stdout_pipe_buffer, regex->re_nsub + 1, matches, 0);
	if (rv1 == 0) {
		/* direct writing */
		handle_direct_writing(regex, matches);
	}

	/* Case 3: Image writing... */
	rv3 = regexec(regex3, stdout_pipe_buffer, regex3->re_nsub + 1, matches3, 0);
	if (rv3 == 0) {
		/* image writing */
		handle_image_writing(regex3, matches3);

	}

	/* Case 2: Fixating */
	rv2 = regexec(regex2, stdout_pipe_buffer, regex2->re_nsub, matches2, 0);
	if ((rv2 == 0) && (rv1 != 0)) {
		/* fixating */
		handle_fixating(regex2, matches2);;
	}

	/* Case 4: Is erasable */
	rv4 = regexec(regex4, stdout_pipe_buffer, regex4->re_nsub + 1, matches4, 0);
	if (rv4 == 0) {
		/* media is erasable */
		handle_is_erasable(regex4, matches4);
	}

	/* Case 5: this media does not support blanking */
	rv5 = regexec(regex5, stdout_pipe_buffer, regex5->re_nsub + 1, matches5, 0);
	if (rv5 == 0) {
		/* media is not erasable */
		handle_blanking_not_supported(regex5, matches5);
	}

	return;
}



void stdout_regexp_destroy(void)
{
	regfree(regex);
	regfree(regex2);
	regfree(regex3);
	regfree(regex4);
	regfree(regex5);

	free(regex);
	free(regex2);
	free(regex3);
	free(regex4);
	free(regex5);

	free(matches);
	free(matches2);
	free(matches3);
	free(matches4);
	free(matches5);

	return;
}










void stderr_regexp_prepare(void)
{
	int rv = 0;
	char c1[25], c2[25], c3[25], c4[25], c5[25], tmp[25];

	/* Device or resource busy. */
	eregex5 = (regex_t *) calloc(1, sizeof(regex_t));
	rv = regcomp(eregex5, "wodim: Device or resource busy.", REG_NOSUB);
	ematches5 = (regmatch_t *) calloc(1, (eregex5->re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		regcomp_error_handler(rv, 10);
	}

	/* Cdda2wav process */
	/* ??/??/??/???????   0% */
	if (sscanf(stderr_pipe_buffer, "%s %s %s %s %s %s", c1, c2, c3, c4, c5, tmp) == 5) {
		handle_audio_grab(c5);
	}

	/* Mkisofs process during image cration (0 - 99,9%) */
	eregex = (regex_t *) calloc(1, sizeof(regex_t));
	rv = regcomp(eregex, "([ ]*)([0-9]+).([0-9]+)% done,", REG_EXTENDED);
	ematches = (regmatch_t *) calloc(1, (eregex->re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		regcomp_error_handler(rv, 6);
	}

	/* Mkisofs 100% (image created) */
	eregex2 = (regex_t *) calloc(1, sizeof(regex_t));
	rv = regcomp(eregex2, "Total translation table size", REG_EXTENDED);
	ematches2 = (regmatch_t *) calloc(1, (eregex2->re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		regcomp_error_handler(rv, 7);
	}
	/* cannot write medium - incompatible format */
	eregex3 = (regex_t *) calloc(1, sizeof(regex_t));
	rv = regcomp(eregex3, "cannot write medium - incompatible format", REG_EXTENDED);
	ematches3 = (regmatch_t *) calloc(1, (eregex3->re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		regcomp_error_handler(rv, 8);
	}
	/* Cannot blank disk */
	eregex4 = (regex_t *) calloc(1, sizeof(regex_t));
	rv = regcomp(eregex4, "Cannot blank disk", REG_EXTENDED);
	ematches4 = (regmatch_t *) calloc(1, (eregex4->re_nsub + 1) * sizeof(regmatch_t));
	if (rv) {
		regcomp_error_handler(rv, 9);
	}

	return;
}


void stderr_regexp_execute(void)
{
	int rv1, rv2, rv3, rv4, rv5;
	/* Case 5: Device or resource busy. */
	rv5 = regexec(eregex5, stderr_pipe_buffer, eregex5->re_nsub + 1, ematches5, 0);
	if (rv5 == 0) {
		/* media is not erasable */
		handle_device_or_resource_busy(eregex5, ematches5);
	}

	rv1 = regexec(eregex, stderr_pipe_buffer, eregex->re_nsub + 1, ematches, 0);
	if ((rv1 == 0) && (current_command == COMMAND_CREATE_IMAGE)) {
		/* additional check was to avoid mistake with direct writing */
		handle_image_creation(eregex, ematches);
	}

	rv2 = regexec(eregex2, stderr_pipe_buffer, eregex2->re_nsub + 1, ematches2, 0);
	if ((rv2 == 0) && (current_command == COMMAND_CREATE_IMAGE)) {
		/* additional check was to avoid mistake with direct writing */
		handle_image_creation_100percent(eregex2, ematches2);
	}
	/* Case 3: cannot write medium - incompatible format */
	rv3 = regexec(eregex3, stderr_pipe_buffer, eregex3->re_nsub, ematches3, 0);
	if (rv3 == 0) {
		/* media is not erasable */
		handle_blanking_not_supported(eregex3, ematches3);
	}
	/* Case 4: Cannot blank disk */
	rv4 = regexec(eregex4, stderr_pipe_buffer, eregex4->re_nsub, ematches4, 0);
	if (rv4 == 0) {
		/* media is not erasable */
		handle_blanking_not_supported(eregex4, ematches4);
	}

	return;
}






void stderr_regexp_destroy(void)
{
	regfree(eregex);
	regfree(eregex2);
	regfree(eregex3);
	regfree(eregex4);
	regfree(eregex5);

	free(eregex);
	free(eregex2);
	free(eregex3);
	free(eregex4);
	free(eregex5);

	free(ematches);
	free(ematches2);
	free(ematches3);
	free(ematches4);
	free(ematches5);

	return;
}





/**
 * Calculate estimated time of accomplishment for a task, convert this time to string
 *
 * \param int eta_value - input eta value
 * \param char *eta_string - outpus string with ETA in hh:mm:ss format
 */
void eta_calculations(char *eta_string, int eta)
{
	char seta_hour[3];
	char seta_min[3];
	char seta_sec[3];

	int eta_hour = (eta / 60) / 60;
	int eta_min = (eta - ((eta_hour * 60) * 60)) / 60;
	int eta_sec = eta - (((eta_hour * 60) * 60) + (eta_min * 60));

	if (eta_hour < 10)
		sprintf(seta_hour, "0%d", eta_hour);
	else
		sprintf(seta_hour, "%d", eta_hour);

	if (eta_min < 10)
		sprintf(seta_min, "0%d", eta_min);
	else
		sprintf(seta_min, "%d", eta_min);

	if (eta_sec < 10)
		sprintf(seta_sec, "0%d", eta_sec);
	else
		sprintf(seta_sec, "%d", eta_sec);

	/* 2TRANS: ETA is Expected Time of Arrival, time to finish a task;
	three %s values are number of hours, number of minutes and number of secs left.
	The string will be displayedin progress dialog window */
	sprintf(eta_string, _("ETA: %s:%s:%s"), seta_hour, seta_min, seta_sec);

	return;
}





void handle_image_writing(regex_t *regex, regmatch_t *matches)
{
	/* data to collect */
	int fifo = 0;
	int total_size = 0;
	int current_done = 0;
	int tracknum = 0;

	/* data to calculate */
	char text_info2_string[PROCESSWIN_MAX_TEXT_LEN];
	char current_value_string[PROCESSWIN_MAX_TEXT_LEN];
	char eta_string[PROCESSWIN_MAX_TEXT_LEN];

	/* collect data */
	int i;
	for (i = 0; i < (regex->re_nsub + 1); ++i) {
		int len = matches[i].rm_eo - matches[i].rm_so;
		char submatch[PIPE_BUFFER_SIZE];
		strncpy(submatch, (char *) &stdout_pipe_buffer[matches[i].rm_so], len);
		submatch[len] = '\0';
		if (i == 8) {
			fifo = atoi((char *) &submatch);
		} else if (i == 5) {
			total_size = atoi((char *) &submatch);
		} else if (i == 3) {
			current_done = atoi((char *) &submatch);
		} else if (i == 1) {
			tracknum = atoi((char *) &submatch);
		}
	}

/*
	if (tracknum != lasttrack) {
		mvwprintw(processwin, 5, 5, "");
		processwin_draw_progress_bar(19, 2, 2, ' ');
	}

	lasttrack = tracknum;

	if (total_size < new_total) {
		total_size = new_total;
	}
*/

	/* compute data */
	float per_done = ((float) current_done) / ((float) total_size);
	if ((per_done >= 0.0) && (per_done <= 1.0)) {
		if (prev_percent != per_done) {
			int eta = (difftime(time(NULL), time0) / current_done) * (total_size - current_done);
			eta_calculations(eta_string, eta);
			prev_percent = per_done;
		}
	}

	/* 2TRANS: this is label displayed in progress window, Track is cd data
	   track, %d is its number */
	sprintf(text_info2_string, _("Track: %d"), tracknum);
	/* 2TRANS: this is label displayed in progress window, first %d is amount
	   of data already written to CD, second %d is total amount of data to write */
	sprintf(current_value_string, _("%d/%d MB"), current_done, total_size);


	/* display data */
	conditional_processwin_display_progress(current_done, total_size, current_value_string);
	processwin_display_text_info(NULL, text_info2_string);
	processwin_display_fifo(fifo);
	wrefresh(processwin);

	return;
}





void handle_direct_writing(regex_t *regex, regmatch_t *matches)
{
	/* data to collect */
	int fifo = 0;
	int total_size = 0;
	int current_done = 0;
	int tracknum = 0;

	/* data to calculate */
	char text_info2_string[PROCESSWIN_MAX_TEXT_LEN];
	char eta_string[PROCESSWIN_MAX_TEXT_LEN];
	char current_value_string[PROCESSWIN_MAX_TEXT_LEN];

	/* collect data */
	int i;
	for (i = 0; i < (regex->re_nsub + 1); ++i) {
		int len = matches[i].rm_eo - matches[i].rm_so;
		char submatch[PIPE_BUFFER_SIZE];
		strncpy(submatch, (char *) &stdout_pipe_buffer[matches[i].rm_so], len);
		submatch[len] = '\0';
		if (i == 8) {
			fifo = atoi((char *) &submatch);
		} else if (i == 5) {
			total_size = atoi((char *) &submatch);
		} else if (i == 3) {
			current_done = atoi((char *) &submatch);
		} else if (i == 1) {
			tracknum = atoi((char *) &submatch);
		}
	}

/*
	if (tracknum != lasttrack) {
		mvwprintw(processwin, 5, 5, "");
		processwin_draw_progress_bar(19, 2, 2, ' ');
	}
	lasttrack = tracknum;

	if (total_size < new_total) {
		total_size = new_total;
	}
*/

	/* compute data */
	float per_done = ((float) current_done) / ((float) total_size);
	if ((per_done >= 0.0) && (per_done <= 1.0)) {
		if (prev_percent != per_done) {
			int eta = (difftime(time(NULL), time0) / current_done) * (total_size - current_done);
			eta_calculations(eta_string, eta);
			prev_percent = per_done;
		}
	}

	/* 2TRANS: this is label displayed in progress window, first %d is amount
	   of data already written to CD, second %d is total amount of data to write */
	sprintf(current_value_string, _("%d/%d MB"), current_done, total_size);
	/* 2TRANS: this is label displayed in progress window, Track is cd data
	   track, %d is its number */
	sprintf(text_info2_string, _("Track: %d"), tracknum);

	/* display data */
	conditional_processwin_display_progress(current_done, total_size, current_value_string);
	processwin_display_text_info(NULL, text_info2_string);
	processwin_display_fifo(fifo);
	wrefresh(processwin);

	return;
}





void handle_fixating(regex_t *regex, regmatch_t *matches)
{
	/* 2TRANS: this is message displayed in process window, meaning that
	closing session is in progress */
	processwin_display_text_info(_("Fixating..."), NULL);

	/* this is unnecessary - writing process always (?) displays 100%,
	   so it will be displayed in processwin as well */
	/* processwin_display_progress(100, 100, ""); */

	wrefresh(processwin);

	return;
}





void handle_audio_grab(char *c5)
{
	int perc = atoi(c5);
	char percent[30];

	if ((current_command == COMMAND_RIP_AUDIO_CD) && (perc <= 100)) {
		if (prev_percent != perc) {
			char eta_string[25];
			int eta = (difftime(time(NULL), time0) / perc) * (100 - perc);
			eta_calculations(eta_string, eta);
			prev_percent = perc;
		}
		mvwprintw(processwin, 5, 5, "");
		processwin_draw_progress_bar( (perc / 5), 5, 2, ACS_BLOCK);

		/* 2TRANS: this string is currently not used; %d%% is amount of audio
		   data already read from cd (in percents) */
		sprintf(percent, _("  %d%% done  "), perc);
		mvwprintw(processwin, 3, 15 - (strlen(percent) / 2), "%s", percent);
		/* mvwprintw(processwin, 5, 5, "");
		   mvwprintw(processwin, 9, 29, "");
		   mvwprintw(processwin, 7, 5, "T: %d P: %d L: %d   ", track, perc, lasttrack); */
		wrefresh(processwin);
		/* fprintf(log, "\nT: %d P: %d L: %d\n", track, perc, lasttrack); */
		fflush(logfi);
		stderr_pipe_buffer[0] = '\0';
		/* track=perc;
		   sleep(1); */
	}

	return;
}





/**
 * Handle normal mkisofs output during image creation
 *
 * Normal mkisofs output durnig image creation is follownig:
 * '58.65% done, estimate finish Sat Mar  3 21:20:06 2007'.
 * mkisofs doesn't write 100% value therefore handle_image_creation_100percent()
 * is needed.
 */
void handle_image_creation(regex_t *regex, regmatch_t *matches)
{
	int i;

	/* Image creation output looks like this:
	* "([ ]*)([0-9]+).([0-9]+)% done," + something. */

	/* Data to collect: */
	int current_done = 0; /* amount of data written to image */

	/* Data to compute: */
	int eta = 0;
	char eta_string[25];
	memset(eta_string, '\0', 25);

	/* collect data */
	for (i = 0; i < (regex->re_nsub + 1); ++i) {
		if (i == 2) {
			int len = matches[i].rm_eo - matches[i].rm_so;
			char submatch[PIPE_BUFFER_SIZE];
			strncpy(submatch, (char *) &stderr_pipe_buffer[matches[i].rm_so], len);
			submatch[len] = '\0';

			current_done = atoi((char *) &submatch);
		}
	}

	/* calculate data */
	eta = (difftime(time(NULL), time0) / current_done) * (100 - current_done);
	eta_calculations(eta_string, eta);

	/* display data */
	processwin_display_eta(eta_string);
	conditional_processwin_display_progress(current_done, 100, "");
	wrefresh(processwin);

	return;
}





/**
 * Handle normal mkisofs output after image creation
 *
 * Normal mkisofs output durnig image creation is follownig:
 * '58.65% done, estimate finish Sat Mar  3 21:20:06 2007'.
 * mkisofs doesn't write 100% value, but it prints following after completion:
 * '
 * Total translation table size: 0
 * Total rockridge attributes bytes: 0
 * Total directory bytes: 0
 * Path table size(bytes): 10
 * Max brk space used 0
 * 127880 extents written (249 MB)
 * '
 *
 * This function is called after capturing
 * 'Total translation table size: 0'
 */
void handle_image_creation_100percent(regex_t *regex, regmatch_t *matches)
{
	/* fake some data to show that creating image has finished */
	char eta_string[25];
	memset(eta_string, '\0', 25);
	eta_calculations(eta_string, 0); /* ETA == 0 */

	/* display data */
	processwin_display_eta(eta_string);
	processwin_display_progress(100, 100, "");
	wrefresh(processwin);

	return;
}





void handle_is_erasable(regex_t *regex, regmatch_t *matches)
{
	media_erasable = MEDIA_ERASABLE_YES;

	return;
}





void handle_blanking_not_supported(regex_t *regex, regmatch_t *matches)
{

	media_erasable = MEDIA_ERASABLE_NO;

	return;
}



void handle_device_or_resource_busy(regex_t *regex, regmatch_t *matches)
{
	/* 2TRANS: this is message in progress window, meaning that
	   subprocess cannot access CD drive, because the drive is busy */
	processwin_display_text_info(_("Device or resource busy"),
				     /* 2TRANS: this is message in progress
				        window, possible explanation for
				        previous message */
				     _("Make sure that device is unmounted."));
	wrefresh(processwin);

	return;
}




void regcomp_error_handler(int rv, int id)
{
	/* buffer for explanation of regcomp failure; FIXME: arbitrary buffer size! */
	char errbuf[100];

	regerror(rv, NULL, errbuf, sizeof(errbuf));
	/* 2TRANS: this is debug message displayed in console, regcomp is a
	   function compiling regular expressions, used in cdw code; 'return
	   value' is return value of regcomp; %d, %d and %s are debug values:
	   function's return value, regcomp expression id, and error message
	   for regcomp error that has occurred */
	fprintf(stderr, _("ERROR regcomp (return value = %d, id = %d): %s\n"), rv, id, errbuf);
	exit(1);
}
