/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


/*
 * Small cdw functions that could be used to implement widgets in other
 * ncurses-based projects with small modifications or no modificatins at all.
 */


#include <string.h>
#include <ctype.h>
#include <ncursesw/ncurses.h>
#include <ncursesw/form.h>

#include <libintl.h>
#include "config.h"
#include "color.h"
#include "gettext.h"
#include "main.h"
#include "cdw_widgets.h"


/* main application window */
extern WINDOW *mainwin;




/*
 * Small, one-line editable text area
 *
 * This function/widget simulates behaviour of FORM field and probably
 * should be replaced by it. One advantage of this function is that
 * it does not require building/initializing FORM and its fields.
 * Second advaitage (I think) is that it can be used to enter passwords
 * (visible input is hashed).
 *
 * Entered text can be longer than area width - part of
 * text will be shifted left.
 *
 * \param WINDOW *window - initialized ncurses window to be used as widget area
 * \param char *init - initial value to be displayed in text area
 * \param bool hash - show astersks in input field instead of real user input
 *
 * \returns 0 if user pressed ESCAPE key, 1 if ENTER pressed
 *
 */
int inbox(WINDOW *window, char *init, bool hash)
{
	int ch = 0, i;
	int pos; /* position of cursor in input window */

	char input[256]; /* string entered in text field */
	int in_x; /* current length of string entered in text filed */

	keypad(window, TRUE);
	werase (window);

	// put initial text in text area
	if (strlen(init) == 0) {
		input[0] = '\0';
		pos = 0;
	} else { /* put initial text into buffer */
		strcpy(input, init);

		/* now print buffer content in the window */
		if (strlen(init) < getmaxx(window) - 1) { /* print whole string */
			mvwprintw(window, 0, 0, init);
			pos = strlen(input);
		} else { /* print end of string, no longer than half of window width */
			i = strlen(init) - (getmaxx(window) / 2) - 1;
			for (; i < strlen(init); i++) {
				if (hash) {
					waddch(window, '*');
				} else {
					waddch(window, input[i]);
				}
			}
			pos = (getmaxx(window) / 2) + 1; /* cursor at the end of string */
		}
	}

	in_x = strlen(input);

	/* now user can enter/append text */
	while ( (ch != CDW_ENTER) && (ch != CDW_ESCAPE) ) {
		ch = wgetch(window);
		if (ch == CDW_ESCAPE) {
			strcpy(input, init);
		} else if ((ch == KEY_BACKSPACE || ch == KEY_DC) && in_x > 0) {
			input[in_x - 1] = '\0';
			in_x--;
			pos--;
			mvwaddch(window, 0, pos, ' ');
			wmove(window, 0, pos);
			if ((pos == (getmaxx(window) / 2) - 1) &&
						  (in_x > (getmaxx(window) / 2) + 1)) {
				werase(window);
				if (in_x - (getmaxx(window) - 1) >= 0) {
					for (i = in_x - (getmaxx(window) - 1); i < in_x; i++) {
						if (hash) {
							waddch(window, '*');
						} else {
							waddch(window, input[i]);
						}
					}
					pos = getmaxx(window) - 1;
				} else {
					for (i = 0; i < in_x; i++) {
						if (hash) {
							waddch(window, '*');
						} else {
							waddch(window, input[i]);
						}
					}
					pos = in_x;
				}
			}
		} else if (isprint(ch)) {
			if (pos < getmaxx(window) - 1) {
				if (hash) {
					waddch(window, '*');
				} else {
					waddch(window, ch);
				}
				input[in_x] = ch;
				input[in_x + 1] = '\0';
				in_x++;
				pos++;
			}
			if (pos == getmaxx(window) - 1) {
				werase(window);
				// input[in_x] = ch;
				// input[in_x + 1] = '\0';
				for (i = in_x - (getmaxx(window) / 2); i < in_x; i++) {
					if (hash) {
						waddch(window, '*');
					} else {
						waddch(window, input[i]);
					}
				}
				// in_x++;
				pos = (getmaxx(window) / 2);
			}
		}
		wrefresh(window);
	}

	strcpy(init, input);

	if (ch == CDW_ESCAPE) {
		return 0;
	} else {
		return 1;
	}
}





#if 0 // replaced by int inbox(WINDOW *window, char *init, bool hash)
int inpass(WINDOW *input, char *init)
{
	int in_x, ch=0, i, pos;
	char instr[256], instr2[256];

	keypad(input, TRUE);
	werase(input);
	strcpy(instr, init);
	if ( strlen(init)==0 )
		instr[0]='\0';
	else
		strcpy(instr, init);
	if (strlen(init)<getmaxx(input)-1) {
		mvwprintw(input, 0, 0, init);
		pos=strlen(instr);
} else {
		for (i=strlen(init)-(getmaxx(input)/2)-1; i<strlen(init); i++) {
			waddch(input, '*');
}
		pos=(getmaxx(input)/2)+1;
}
	in_x=strlen(instr);
	while ( (ch!=10) && (ch!=27) ) {
		ch=wgetch(input);
		if (( ch==27 )) {
			strcpy(instr, init);
}
		if ( ch==KEY_BACKSPACE && in_x>0) {
			instr[in_x-1]='\0';
			in_x--;
			pos--;
			mvwaddch(input, 0, pos, ' ');
			wmove(input, 0, pos);
			if ((pos==(getmaxx(input)/2)-1) && (in_x>(getmaxx(input)/2)+1)) {
				werase(input);
				if (in_x-(getmaxx(input)-1)>=0) {
					for (i=in_x-(getmaxx(input)-1); i<in_x; i++) {
						waddch(input, '*');
}
					pos=getmaxx(input)-1;
} else {
					for (i=0; i<in_x; i++) {
						waddch(input, '*');
}
					pos=in_x;
}
}
}
		if ( isprint(ch) ) {
			if (pos<getmaxx(input)-1) {
				waddch(input, '*');
				instr[in_x]=ch;
				instr[in_x+1]='\0';
				in_x++;
				pos++;
}
			if (pos==getmaxx(input)-1) {
				werase(input);
                //	instr[in_x]=ch;
                //	instr[in_x+1]='\0';
				for (i=in_x-(getmaxx(input)/2); i<in_x; i++) {
					waddch(input, '*');
}
                //	in_x++;
				pos=(getmaxx(input)/2);
}
}
		wrefresh(input);
}

	strcpy(init, instr);
	if (ch==27)
		return 0;
	else
		return 1;
}


#endif // if 0


int input_passwd(char *title, char *label, char *prop)
{
	WINDOW *volwin, *edit;
	int ret;

	volwin=newwin(10,30,(LINES-10)/2,(COLS-30)/2);
	wbkgd(volwin, COLOR_PAIR(2));
	werase(volwin);
	box(volwin,0,0);
	mvwaddch(volwin, 0, 2, ACS_RTEE);
	mvwprintw(volwin, 0, 3, " %s ", _(title));
	waddch(volwin, ACS_LTEE);
	mvwprintw(volwin, 3, 3, _(label));
	wrefresh(volwin);
	edit=derwin(volwin, 1, 24, 5, 3);
	wbkgd(edit, COLOR_PAIR(6));
	werase(edit);
	wrefresh(edit);
	ret=inbox(edit, prop, true);

	return ret;
}




/*
 * Draw nice box around window, with label on top and
 * (optional) label at the bottom of window
 *
 * \param WINDOW *window - initialized ncurses window to be boxed
 * \param char *top_string - string that will appear on top of window
 * \param char *bottom_string - string that will appear on bottom of window
 */
void nice_box(WINDOW *window, char *top_string, char *bottom_string)
{
	/* FIXME: check if strings aren't longer than horizontal window size */
	/* draw standartd ncurses box, and then
	 * overwrite parts of border with string(s) */

	box(window, 0, 0);

	if (strcmp(bottom_string, "")) {
		/* additinal window information ath the bottom */
		mvwprintw(window, getmaxy(window) - 1, 1, "( %s )", bottom_string);
	}

	mvwaddch(window, 0, 2, ACS_RTEE);
	/* add window title */
	mvwprintw(window, 0, 3, " %s ", top_string);
	waddch(window, ACS_LTEE);

	return;
}




/*
 * Show dialog window with title, message and buttons
 *
 * Show dialog window with title and message. Depending on type of dialog
 * user is presented with one, two or three buttons: value of
 * parameter 'type' describes them. Function returns value of selected button.
 * Background color depends on 'type' value: it is red for 'OK/Cancel' and
 * 'Yes/No/Cancel' dialogs and gray for 'Ok' dialog.
 *
 * FIXME: 'Enter' behaviour is not described
 *
 * \param char *message - message displayed in dialog window
 * \param char *title - dialog window title
 * \param int type - type of dialog - values defined in header
 * \returns value of pressed button (valid values defined in header)
 */
int dialogbox(char *title, char *message, int type)
{
	WINDOW *dialog;
	int line = 2; /* line number in dialog window, in which we write message; this is starting value, can be incremented for multi-line messages */
	int n, i, c, bg_color;

	// initial dialog WINDOW size: maxwidth and maxline
	int maxline=0, maxwidth=30;

	char temp[50];

	int button; /* currently selected button */
	/* set initial button to OK/YES */
	if (type == DIALOG_YES_NO_CANCEL) {
		button = BUTTON_YES;
	} else { // DIALOG_OK_CANCEL or DIALOG_OK, BUTTON_OK wil be default in both cases
		button = BUTTON_OK;
	}

	/* recalculate window size based on message length */
	temp[0] = '\0';
	n = 0;
	for (i = 0; i < strlen(message); i++){
		if (message[i] != '\n')
			temp[n++] = message[i];
		if (( message[i] == '\n') || (i == strlen(message) - 1) || ((n > 30) && message[i] == ' ')){
			temp[n] = '\0';
			if (maxwidth < strlen(temp) + 8) {
				maxwidth = strlen(temp) + 8;
			}
			temp[0] = '\0';
			n = 0;
			if (message[i] == '\n') {
				maxline++;
			}
		}
	}

	switch (type){
		case DIALOG_OK:
			bg_color = 2;
			break;
		case DIALOG_OK_CANCEL:
		case DIALOG_YES_NO_CANCEL:
			bg_color = 10;
			break;
	}

	dialog = newwin(9 + maxline, maxwidth, (LINES - (9 + maxline)) / 2, (COLS - maxwidth) / 2);
	wbkgd(dialog, COLOR_PAIR(bg_color));
	werase(dialog);
	nice_box(dialog, title, "");
	keypad(dialog, TRUE);

	/* wrap message text so that it fits into dialog window;
	 * print each wrapped line into dialog window */
	temp[0] = '\0';
	n = 0;
	for (i = 0; i < strlen(message); i++){
		if (message[i] != '\n')
			temp[n++] = message[i];
		if ((message[i] == '\n') || (i == strlen(message) - 1) || ((n > 30) && message[i] == ' ')){
			temp[n] = '\0';
			mvwprintw(dialog, line++, (maxwidth - strlen(temp)) / 2, temp);
			temp[0] = '\0';
			n = 0;
		}
	}

	// buttons positions
	// buttons are in one row
	int buttons_row = maxline + 6;
	// 1-button dialog
	int ok_col_1 = (getmaxx(dialog) - strlen(_("Ok"))) / 2;
	// 2-button dialog
	int ok_col_2 = 8;
	int cancel_col_2 = (maxwidth - 7) - strlen(_("Cancel"));
	// 3-button dialog
	int yes_col_3 = 4;
	int no_col_3 = 14;
	int cancel_col_3 = (maxwidth - 7) - strlen(_("Cancel"));

	/* draw initial dialog box */
	if (type == DIALOG_OK) {
		wattrset(dialog, COLOR_PAIR(6));
		mvwprintw(dialog, buttons_row, ok_col_1	, "[ %s ]", _("Ok"));
	}
	if (type == DIALOG_OK_CANCEL) {
		wattrset(dialog, COLOR_PAIR(6));
		mvwprintw(dialog, buttons_row, ok_col_2, "[ %s ]", _("Ok"));

		wattrset(dialog, COLOR_PAIR(bg_color));
		mvwprintw(dialog, buttons_row, cancel_col_2, "[ %s ]", _("Cancel"));
	}
	if (type == DIALOG_YES_NO_CANCEL) {
		wattrset(dialog, COLOR_PAIR(6));
		mvwprintw(dialog, buttons_row, yes_col_3, "[ %s ]", _("Yes"));

		wattrset(dialog, COLOR_PAIR(bg_color));
		mvwprintw(dialog, buttons_row, no_col_3, "[ %s ]", _("No"));

		wattrset(dialog, COLOR_PAIR(bg_color));
		mvwprintw(dialog, buttons_row, cancel_col_3, "[ %s ]", _("Cancel"));
	}
	wrefresh(dialog);

	/* handle cursor keys - highlight selected button */
	while (((c = mvwgetch(dialog, 8 + maxline, maxwidth - 1)) != CDW_ESCAPE) && (c != CDW_ENTER)) {
		/* select action according to dialog type,
		* key pressed and currently selected button */
		if (type == DIALOG_OK_CANCEL) {
			switch(c){
				case KEY_RIGHT:
					if (button == BUTTON_OK){
						wattrset(dialog, COLOR_PAIR(bg_color));
						mvwprintw(dialog, buttons_row, ok_col_2, "[ %s ]", _("Ok"));
						wattrset(dialog, COLOR_PAIR(6));
						mvwprintw(dialog, buttons_row, cancel_col_2, "[ %s ]", _("Cancel"));
						button = BUTTON_CANCEL;
					}
					break;
				case KEY_LEFT:
					if (button == BUTTON_CANCEL){
						wattrset(dialog, COLOR_PAIR(6));
						mvwprintw(dialog, buttons_row, ok_col_2, "[ %s ]", _("Ok"));
						wattrset(dialog, COLOR_PAIR(bg_color));
						mvwprintw(dialog, buttons_row, cancel_col_2, "[ %s ]", _("Cancel"));
						button = BUTTON_OK;
					}
					break;
			}
		} else 	if (type == DIALOG_YES_NO_CANCEL) {
			switch(c){
				case KEY_RIGHT:
					if (button == BUTTON_YES){
						wattrset(dialog, COLOR_PAIR(bg_color));
						mvwprintw(dialog, buttons_row, yes_col_3, "[ %s ]", _("Yes"));
						wattrset(dialog, COLOR_PAIR(6));
						mvwprintw(dialog, buttons_row, no_col_3, "[ %s ]", _("No"));
						button = BUTTON_NO;
					} else 	if (button == BUTTON_NO){
						wattrset(dialog, COLOR_PAIR(bg_color));
						mvwprintw(dialog, buttons_row, no_col_3, "[ %s ]", _("No"));
						wattrset(dialog, COLOR_PAIR(6));
						mvwprintw(dialog, buttons_row, cancel_col_3, "[ %s ]", _("Cancel"));
						button = BUTTON_CANCEL;
					} else 	if (button == BUTTON_CANCEL){
						wattrset(dialog, COLOR_PAIR(bg_color));
						mvwprintw(dialog, buttons_row, cancel_col_3, "[ %s ]", _("Cancel"));
						wattrset(dialog, COLOR_PAIR(6));
						mvwprintw(dialog, buttons_row, yes_col_3, "[ %s ]", _("Yes"));
						button = BUTTON_YES;
					}
					break;
				case KEY_LEFT:
					if (button == BUTTON_YES){ // yes -> cancel
						wattrset(dialog, COLOR_PAIR(bg_color));
						mvwprintw(dialog, buttons_row, yes_col_3, "[ %s ]", _("Yes"));
						wattrset(dialog, COLOR_PAIR(6));
						mvwprintw(dialog, buttons_row, cancel_col_3, "[ %s ]", _("Cancel"));
						button = BUTTON_CANCEL;
					} else if (button == BUTTON_NO){ // no -> yes
						wattrset(dialog, COLOR_PAIR(bg_color));
						mvwprintw(dialog, buttons_row, no_col_3, "[ %s ]", _("No"));
						wattrset(dialog, COLOR_PAIR(6));
						mvwprintw(dialog, buttons_row, yes_col_3, "[ %s ]", _("Yes"));
						button = BUTTON_YES;
					} else 	if (button == BUTTON_CANCEL){ // cancel -> no
						wattrset(dialog, COLOR_PAIR(bg_color));
						mvwprintw(dialog, buttons_row, cancel_col_3, "[ %s ]", _("Cancel"));
						wattrset(dialog, COLOR_PAIR(6));
						mvwprintw(dialog, buttons_row, no_col_3, "[ %s ]", _("No"));
						button = BUTTON_NO;
					}
					break;
			} /* switch */
		} /* if */
		wrefresh(dialog);
	} /* while */

	if (c == CDW_ESCAPE) {
		if (type == DIALOG_OK_CANCEL || type == DIALOG_YES_NO_CANCEL) {
			button = BUTTON_CANCEL;
		} else if (type == DIALOG_OK) {
			button = BUTTON_OK;
		} else {
			;
		}
	}

	delwin(dialog);

	if (mainwin != NULL) { /* cdw ui already constructed */
		wrefresh(derwin(mainwin, LINES - 1, COLS - 1, 0, 0));
		select_window(FALSE);
	} else {
		; /* looks like dialogbox is called before constructing
		   * app UI - emergency situation? */
	}

	return button;
}




/**
 * Display widget allowing to enter simple string value
 *
 * Display widget with title and label, allowing user to enter simple
 * value, like label, path or number. Parameter "mode" puts restrictions
 * on entered value.
 *
 * \param char *title - window title
 * \param char *label - label put into window
 * \param char *prop - string containing entered value - it has to be initialized/allocated by caller
 * \param int mode - widget mode, puts some restrictions on entered value
 *
 * \returns - CDW_ENTER or CDW_ESCAPE
 */
int input_box(char *title, char *label, char *prop, int mode)
{
	WINDOW *volwin;
	FIELD *field[1];
	FORM *form1;
	int ret;
	int ch;

	/* dialog window */
	volwin = newwin(10, 50, (LINES - 10) / 2, (COLS - 50) / 2);
	keypad(volwin, TRUE);
	wbkgd(volwin, COLOR_PAIR(2));
	werase(volwin);
	mvwprintw(volwin, 3, 3, _(label)); /* print descripion first, then border, because description can erase barts of border */
	nice_box(volwin, _(title), "");


	/* input field
	 * new_field params: height, width, starty, startx, number of
	 * offscreen rows and number of additional working buffers */
	field[0] = new_field(1, 44, 0, 0, 0, 0);
	set_field_buffer(field[0], 0, prop);
	set_field_back(field[0], COLOR_PAIR(6));
	set_field_fore(field[0], COLOR_PAIR(6));
	field_opts_off(field[0], O_STATIC); /* stretch to fit entered data */
	set_max_field(field[0], 200); /* limit stretching of the field; FIXME - arbitrary magic number */
	field[1] = NULL;

	/* form containing field. placed in window */
	form1 = new_form(field);
	set_form_win(form1, volwin);
	set_form_sub(form1, derwin(volwin, 1, 44, 5, 2));
	post_form(form1);

	set_current_field(form1, field[0]);
	form_driver(form1, REQ_END_LINE);
	wrefresh(volwin);

	ch = 0;
	while ((ch != CDW_ESCAPE) && (ch != CDW_ENTER)) {
		ch = wgetch(volwin);
		switch (ch) {
			case KEY_LEFT:
				form_driver(form1, REQ_PREV_CHAR);
				break;
			case KEY_RIGHT:
				form_driver(form1, REQ_NEXT_CHAR);
				break;
			case KEY_BACKSPACE:
				form_driver(form1, REQ_DEL_PREV);
				break;
			case KEY_DC:
				form_driver(form1, REQ_DEL_CHAR);
				break;
			case KEY_F(10):
			case 10:
				/* return main loop */
				form_driver(form1, REQ_VALIDATION);
				sprintf(prop, (char *)rtrim(field_buffer(field[0], 0)));
				// write_conf();
				/* return ch; */
				break;
			default:
				/* push char ch to form */
				form_driver(form1, ch);
				break;
				wrefresh(volwin);
		}
	}

/*    edit=derwin(volwin, 1, 24, 5, 3);
	wbkgd(edit, COLOR_PAIR(6));
	werase(edit);
	wrefresh(edit);
	ret=inbox(edit, prop, false);*/

	/* clean up */
	unpost_form(form1);
	free_form(form1);
	free_field(field[0]);
	delwin(volwin);
	wrefresh(volwin);

	return ch;
}
