/*
  This file is part of CDO. CDO is a collection of Operators to manipulate and analyse Climate model Data.

  Author: Uwe Schulzweida

*/

// #define WITH_XYZCOORDS 1

#include "remap_grid_cell_search.h"
#include "cdo_output.h"
#include "compare.h"
#include "varray.h"
#include "cdo_options.h"
#include "cdo_omp.h"
#include <mpim_grid.h>

extern "C"
{
#include "lib/yac/sphere_part.h"
}

CellSearchMethod cellSearchMethod(CellSearchMethod::spherepart);

void
set_cell_search_method(const std::string &methodString)
{
  // clang-format off
  if      (methodString == "spherepart") cellSearchMethod = CellSearchMethod::spherepart;
  else if (methodString == "latbins")    cellSearchMethod = CellSearchMethod::latbins;
  else cdo_abort("Grid cell search method %s not available!", methodString);
  // clang-format on
}

static void
grid_boundbox_reg2d(size_t nx, size_t ny, const Varray<double> &reg2d_corner_lon, const Varray<double> &reg2d_corner_lat,
                    double *grid_bound_box)
{
  grid_bound_box[0] = reg2d_corner_lat[0];
  grid_bound_box[1] = reg2d_corner_lat[ny];
  if (grid_bound_box[0] > grid_bound_box[1])
    {
      grid_bound_box[0] = reg2d_corner_lat[ny];
      grid_bound_box[1] = reg2d_corner_lat[0];
    }
  grid_bound_box[2] = reg2d_corner_lon[0];
  grid_bound_box[3] = reg2d_corner_lon[nx];
}

static void
grid_cell_search_create_reg2d(GridCellSearch &gcs, const RemapGrid &remapGrid)
{
  const auto &reg2d_corner_lon = remapGrid.reg2d_corner_lon;
  const auto &reg2d_corner_lat = remapGrid.reg2d_corner_lat;

  gcs.isReg2d = true;
  gcs.dims[0] = remapGrid.dims[0];
  gcs.dims[1] = remapGrid.dims[1];
  auto nx = remapGrid.dims[0];
  auto ny = remapGrid.dims[1];
  auto nxp1 = nx + 1;
  auto nyp1 = ny + 1;

  gcs.reg2d_corner_lon.resize(nxp1);
  gcs.reg2d_corner_lat.resize(nyp1);

  varray_copy(nxp1, reg2d_corner_lon, gcs.reg2d_corner_lon);
  varray_copy(nyp1, reg2d_corner_lat, gcs.reg2d_corner_lat);

  grid_boundbox_reg2d(nx, ny, gcs.reg2d_corner_lon, gcs.reg2d_corner_lat, gcs.gridBoundboxReg2d);

  gcs.inUse = true;
}

static void
grid_cell_search_create_unstruct(GridCellSearch &gcs, const RemapGrid &remapGrid)
{
  auto numCells = remapGrid.size;
  auto numCorners = remapGrid.numCorners;
  const auto &cellCornerLon = remapGrid.cell_corner_lon;
  const auto &cellCornerLat = remapGrid.cell_corner_lat;

  gcs.method = cellSearchMethod;

  Varray<enum yac_edge_type> edgeTypes(numCorners, YAC_GREAT_CIRCLE_EDGE);
  Varray<yac_grid_cell> cells(Threading::ompNumThreads);
  for (int i = 0; i < Threading::ompNumThreads; ++i)
    {
      cells[i].coordinates_xyz = new double[numCorners][3];
      cells[i].edge_type = edgeTypes.data();
      cells[i].num_corners = numCorners;
      cells[i].array_size = numCorners;
    }

  auto bndCircles = new bounding_circle[numCells];

#ifdef WITH_XYZCOORDS
  gcs.xyzCoords = new double[numCells * numCorners][3];
#endif

#ifdef _OPENMP
#pragma omp parallel for default(shared)
#endif
  for (size_t i = 0; i < numCells; ++i)
    {
      auto ompthID = cdo_omp_get_thread_num();
      auto &cell = cells[ompthID];
      auto xyz = cell.coordinates_xyz;

      for (size_t k = 0; k < numCorners; ++k)
        gcLLtoXYZ(cellCornerLon[i * numCorners + k], cellCornerLat[i * numCorners + k], xyz[k]);

      if (numCorners == 3)
        yac_get_cell_bounding_circle_unstruct_triangle(xyz[0], xyz[1], xyz[2], &bndCircles[i]);
      else
        yac_get_cell_bounding_circle(cell, &bndCircles[i]);

#ifdef WITH_XYZCOORDS
      auto offset = i * numCorners;
      for (size_t k = 0; k < numCorners; ++k)
        for (size_t l = 0; l < 3; ++l) gcs.xyzCoords[offset + k][l] = xyz[k][l];
#endif
    }

  gcs.yacSearch = yac_bnd_sphere_part_search_new(bndCircles, numCells);
  gcs.yacBndCircles = bndCircles;

  for (int i = 0; i < Threading::ompNumThreads; ++i) delete[] cells[i].coordinates_xyz;

  gcs.inUse = true;
}

void
grid_cell_search_create(GridCellSearch &gcs, const RemapGrid &remapGrid)
{
  if (remapGrid.type == RemapGridType::Reg2D)
    grid_cell_search_create_reg2d(gcs, remapGrid);
  else
    grid_cell_search_create_unstruct(gcs, remapGrid);
}

void
grid_cell_search_delete(GridCellSearch &gcs)
{
  if (gcs.inUse)
    {
#ifdef WITH_XYZCOORDS
      if (gcs.xyzCoords) delete[] gcs.xyzCoords;
#endif
      varray_free(gcs.reg2d_corner_lon);
      varray_free(gcs.reg2d_corner_lat);

      if (gcs.yacSearch) yac_bnd_sphere_part_search_delete((bnd_sphere_part_search *) gcs.yacSearch);
      if (gcs.yacBndCircles) delete[] (bounding_circle *) gcs.yacBndCircles;

      gcs.inUse = false;
    }
}

static size_t
doGridCellSearchYac(bnd_sphere_part_search *yacSearch, bounding_circle *yacBndCircles, bool isReg2dCell,
                    const yac_grid_cell &yacGridCell, Varray<size_t> &srchAddr)
{
  size_t numCorners = yacGridCell.num_corners;
  bounding_circle bndCircle;
  auto xyz = yacGridCell.coordinates_xyz;

  if (numCorners == 4 && isReg2dCell)
    yac_get_cell_bounding_circle_reg_quad(xyz[0], xyz[1], xyz[2], &bndCircle);
  else if (numCorners == 3)
    yac_get_cell_bounding_circle_unstruct_triangle(xyz[0], xyz[1], xyz[2], &bndCircle);
  else
    yac_get_cell_bounding_circle(yacGridCell, &bndCircle);

  size_t numSearchCells;
  size_t *currNeighs;
  yac_bnd_sphere_part_search_do_bnd_circle_search(yacSearch, &bndCircle, 1, &currNeighs, &numSearchCells);

  if (srchAddr.size() < numSearchCells) srchAddr.resize(numSearchCells);

  size_t k = 0;
  // for (size_t i = 0; i < numSearchCells; ++i) srchAddr[i] = currNeighs[i];
  for (size_t i = 0; i < numSearchCells; ++i)
    {
      if (yac_extents_overlap(&bndCircle, &yacBndCircles[currNeighs[i]])) srchAddr[k++] = currNeighs[i];
    }
  numSearchCells = k;
  free(currNeighs);

  return numSearchCells;
}

size_t
do_grid_cell_search(GridCellSearch &gcs, bool isReg2dCell, GridCell &gridCell, Varray<size_t> &srchAddr)
{
  if (gcs.inUse)
    {
      // clang-format off
      if (gcs.method == CellSearchMethod::spherepart)
        return doGridCellSearchYac((bnd_sphere_part_search*)gcs.yacSearch, (bounding_circle *)gcs.yacBndCircles,
                                   isReg2dCell, gridCell.yacGridCell, srchAddr);
      else
        cdo_abort("%s::method undefined!", __func__);
      // clang-format on
    }

  return 0;
}
