/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2021 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include <cdi.h>

#include "process_int.h"

template <typename T>
static void
vfarcmulcplx(size_t len, size_t nmiss, Varray<T> &v, const double missval, const double rconst[2])
{
  // z1 x z2 = (x1x2 - y1y2) + i(x1y2 + x2y1)
  const auto missval1 = missval;
  const auto missval2 = missval;
  for (size_t i = 0; i < len; i++)
    {
      const double v1r = v[2 * i];
      const double v1i = v[2 * i + 1];
      v[2 * i] = SUBMN(MULMN(v1r, rconst[0]), MULMN(v1i, rconst[1]));
      v[2 * i + 1] = ADDMN(MULMN(v1r, rconst[1]), MULMN(v1i, rconst[0]));
    }
}

static void
vfarcmulcplx(Field &field, const double rconst[2])
{
  if (field.memType == MemType::Float)
    vfarcmulcplx(field.size, field.nmiss, field.vec_f, field.missval, rconst);
  else
    vfarcmulcplx(field.size, field.nmiss, field.vec_d, field.missval, rconst);
}

template <typename T>
static void
vfarcdivcplx(size_t len, size_t nmiss, Varray<T> &v, const double missval, const double rconst[2])
{
  // z1 / z2 = (x1x2 + y1y2) / (x2x2 + y2y2) + i (y1x2 - x1y2) / (x2x2 + y2y2)
  const auto missval1 = missval;
  const auto missval2 = missval;
  for (size_t i = 0; i < len; i++)
    {
      const double v1r = v[2 * i];
      const double v1i = v[2 * i + 1];
      const double denominator = ADDMN(MULMN(rconst[0], rconst[0]), MULMN(rconst[1], rconst[1]));
      v[2 * i] = DIVMN(ADDMN(MULMN(v1r, rconst[0]), MULMN(v1i, rconst[1])), denominator);
      v[2 * i + 1] = DIVMN(SUBMN(MULMN(v1i, rconst[0]), MULMN(v1r, rconst[1])), denominator);
    }
}

static void
vfarcdivcplx(Field &field, const double rconst[2])
{
  if (field.memType == MemType::Float)
    vfarcdivcplx(field.size, field.nmiss, field.vec_f, field.missval, rconst);
  else
    vfarcdivcplx(field.size, field.nmiss, field.vec_d, field.missval, rconst);
}

template <typename T>
static void
vfarcaddcplx(size_t len, size_t nmiss, Varray<T> &v, const double missval, const double rconst[2])
{
  const auto missval1 = missval;
  const auto missval2 = missval;
  for (size_t i = 0; i < len; i++)
    {
      v[2 * i] = ADDMN(v[2 * i], rconst[0]);
      v[2 * i + 1] = ADDMN(v[2 * i + 1], rconst[1]);
    }
}

static void
vfarcaddcplx(Field &field, const double rconst[2])
{
  if (field.memType == MemType::Float)
    vfarcaddcplx(field.size, field.nmiss, field.vec_f, field.missval, rconst);
  else
    vfarcaddcplx(field.size, field.nmiss, field.vec_d, field.missval, rconst);
}

template <typename T>
static void
vfarcsubcplx(size_t len, size_t nmiss, Varray<T> &v, const double missval, const double rconst[2])
{
  const auto missval1 = missval;
  const auto missval2 = missval;
  for (size_t i = 0; i < len; i++)
    {
      v[2 * i] = SUBMN(v[2 * i], rconst[0]);
      v[2 * i + 1] = SUBMN(v[2 * i + 1], rconst[1]);
    }
}

static void
vfarcsubcplx(Field &field, const double rconst[2])
{
  if (field.memType == MemType::Float)
    vfarcsubcplx(field.size, field.nmiss, field.vec_f, field.missval, rconst);
  else
    vfarcsubcplx(field.size, field.nmiss, field.vec_d, field.missval, rconst);
}

void
vfarcfuncplx(Field &field, const double rconst[2], int function)
{
  switch (function)
    {
    case FieldFunc_Add: vfarcaddcplx(field, rconst); break;
    case FieldFunc_Sub: vfarcsubcplx(field, rconst); break;
    case FieldFunc_Mul: vfarcmulcplx(field, rconst); break;
    case FieldFunc_Div: vfarcdivcplx(field, rconst); break;
    default: cdo_abort("%s: function %d not implemented!", __func__, function);
    }
}
