/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2021 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include "cdo_output.h"
#include "field.h"

template <typename T>
static void
vfarcmul(size_t len, size_t nmiss, Varray<T> &v, const T missval, const double rconst)
{
  const auto missval1 = missval;
  const auto missval2 = missval;

  if (nmiss)
    {
      for (size_t i = 0; i < len; i++) v[i] = MULMN(v[i], rconst);
    }
  else
    {
      for (size_t i = 0; i < len; i++) v[i] *= rconst;
    }
}

void
vfarcmul(Field &field, const double rconst)
{
  if (field.memType == MemType::Float)
    vfarcmul(field.size, field.nmiss, field.vec_f, (float) field.missval, rconst);
  else
    vfarcmul(field.size, field.nmiss, field.vec_d, field.missval, rconst);
}

template <typename T>
static size_t
vfarcdiv(size_t len, size_t nmiss, Varray<T> &v, const T missval, const double rconst)
{
  const auto missval1 = missval;
  const auto missval2 = missval;

  if (nmiss || IS_EQUAL(rconst, 0))
    {
      for (size_t i = 0; i < len; i++) v[i] = DIVMN(v[i], rconst);

      if (IS_EQUAL(rconst, 0)) nmiss = len;
    }
  else
    {
      for (size_t i = 0; i < len; i++) v[i] /= rconst;
    }

  return nmiss;
}

void
vfarcdiv(Field &field, const double rconst)
{
  if (field.memType == MemType::Float)
    field.nmiss = vfarcdiv(field.size, field.nmiss, field.vec_f, (float) field.missval, rconst);
  else
    field.nmiss = vfarcdiv(field.size, field.nmiss, field.vec_d, field.missval, rconst);
}

template <typename T>
static void
vfarcadd(size_t len, size_t nmiss, Varray<T> &v, const T missval, const double rconst)
{
  const auto missval1 = missval;
  const auto missval2 = missval;

  if (nmiss)
    {
      for (size_t i = 0; i < len; i++) v[i] = ADDMN(v[i], rconst);
    }
  else
    {
      for (size_t i = 0; i < len; i++) v[i] += rconst;
    }
}

void
vfarcadd(Field &field, const double rconst)
{
  if (field.memType == MemType::Float)
    vfarcadd(field.size, field.nmiss, field.vec_f, (float) field.missval, rconst);
  else
    vfarcadd(field.size, field.nmiss, field.vec_d, field.missval, rconst);
}

void
vfarcsub(Field &field, const double rconst)
{
  vfarcadd(field, -rconst);
}

template <typename T>
static void
vfarcmin(size_t len, size_t nmiss, Varray<T> &v, const T missval, const T rconst)
{
  const auto missval1 = missval;

  if (nmiss)
    {
      for (size_t i = 0; i < len; i++)
        if (DBL_IS_EQUAL(v[i], missval1) || v[i] > rconst) v[i] = rconst;
    }
  else
    {
      for (size_t i = 0; i < len; i++)
        if (v[i] > rconst) v[i] = rconst;
    }
}

void
vfarcmin(Field &field, const double rconst)
{
  if (field.memType == MemType::Float)
    vfarcmin(field.size, field.nmiss, field.vec_f, (float) field.missval, (float) rconst);
  else
    vfarcmin(field.size, field.nmiss, field.vec_d, field.missval, rconst);
}

template <typename T>
static void
vfarcmax(size_t len, size_t nmiss, Varray<T> &v, const T missval, const T rconst)
{
  const auto missval1 = missval;

  if (nmiss)
    {
      for (size_t i = 0; i < len; i++)
        if (DBL_IS_EQUAL(v[i], missval1) || v[i] < rconst) v[i] = rconst;
    }
  else
    {
      for (size_t i = 0; i < len; i++)
        if (v[i] < rconst) v[i] = rconst;
    }
}

void
vfarcmax(Field &field, const double rconst)
{
  if (field.memType == MemType::Float)
    vfarcmax(field.size, field.nmiss, field.vec_f, (float) field.missval, (float) rconst);
  else
    vfarcmax(field.size, field.nmiss, field.vec_d, field.missval, rconst);
}

template <typename T>
static void
vfarmod(size_t len, size_t nmiss, Varray<T> &v, const T missval, const double divisor)
{
  const auto missval1 = missval;

  for (size_t i = 0; i < len; i++)
    {
      v[i] = DBL_IS_EQUAL(v[i], missval1) ? missval1 : std::fmod((double) v[i], divisor);
    }
}

void
vfarmod(Field &field, const double divisor)
{
  if (field.memType == MemType::Float)
    vfarmod(field.size, field.nmiss, field.vec_f, (float) field.missval, divisor);
  else
    vfarmod(field.size, field.nmiss, field.vec_d, field.missval, divisor);
}

void
vfarcfun(Field &field, const double rconst, const int function)
{
  switch (function)
    {
    case FieldFunc_Add: vfarcadd(field, rconst); break;
    case FieldFunc_Sub: vfarcsub(field, rconst); break;
    case FieldFunc_Mul: vfarcmul(field, rconst); break;
    case FieldFunc_Div: vfarcdiv(field, rconst); break;
    case FieldFunc_Min: vfarcmin(field, rconst); break;
    case FieldFunc_Max: vfarcmax(field, rconst); break;
    case FieldFunc_Mod: vfarmod(field, rconst); break;
    default: cdo_abort("%s: function %d not implemented!", __func__, function);
    }
}
