/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2021 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Splittime  splithour       Split hours
      Splittime  splitday        Split days
      Splittime  splitmon        Split months
      Splittime  splitseas       Split seasons
*/

#include <time.h>
#include <cdi.h>

#include "cdo_options.h"
#include "process_int.h"
#include "cdo_season.h"
#include "util_files.h"

constexpr int MaxStreams = 32;

struct tm
datetime_to_tm(int64_t date, int time)
{
  int year, month, day, hour, minute, second;
  cdiDecodeDate(date, &year, &month, &day);
  cdiDecodeTime(time, &hour, &minute, &second);

  struct tm stime;
  memset(&stime, 0, sizeof(struct tm));

  stime.tm_sec = second;
  stime.tm_min = minute;
  stime.tm_hour = hour;
  stime.tm_mday = day;
  stime.tm_mon = month - 1;
  stime.tm_year = year - 1900;

  return stime;
}

void *
Splittime(void *process)
{
  CdoStreamID streamID2;
  int varID, levelID;
  CdoStreamID streamIDs[MaxStreams];
  int tsIDs[MaxStreams];
  int index = 0;
  size_t nmiss;

  cdo_initialize(process);

  if (process_self().m_ID != 0) cdo_abort("This operator can't be combined with other operators!");

  const auto lcopy = unchanged_record();

  enum
  {
    func_time,
    func_date
  };

  // clang-format off
  const auto SPLITHOUR = cdo_operator_add("splithour", func_time, 10000, nullptr);
  const auto SPLITDAY  = cdo_operator_add("splitday",  func_date,     1, nullptr);
  const auto SPLITMON  = cdo_operator_add("splitmon",  func_date,   100, nullptr);
  const auto SPLITSEAS = cdo_operator_add("splitseas", func_date,   100, nullptr);
  // clang-format on

  (void) (SPLITDAY);   // CDO_UNUSED
  (void) (SPLITHOUR);  // CDO_UNUSED

  const auto operatorID = cdo_operator_id();
  const auto operfunc = cdo_operator_f1(operatorID);
  const auto operintval = cdo_operator_f2(operatorID);

  const char *format = nullptr;
  if (operatorID == SPLITMON && cdo_operator_argc() == 1)
    format = cdo_operator_argv(0).c_str();
  else
    operator_check_argc(0);

  const char *seas_name[4];
  get_season_name(seas_name);

  for (int i = 0; i < MaxStreams; i++) streamIDs[i] = CDO_STREAM_UNDEF;
  for (int i = 0; i < MaxStreams; i++) tsIDs[i] = 0;

  const auto streamID1 = cdo_open_read(0);

  const auto vlistID1 = cdo_stream_inq_vlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  char filename[8192];
  strcpy(filename, cdo_get_obase());
  const int nchars = strlen(filename);

  auto refname = cdo_get_stream_name(0);
  char filesuffix[32] = { 0 };
  FileUtils::gen_suffix(filesuffix, sizeof(filesuffix), cdo_inq_filetype(streamID1), vlistID1, refname);

  Varray<double> array;
  //  if ( ! lcopy )
  {
    auto gridsizemax = vlistGridsizeMax(vlistID1);
    if (vlistNumber(vlistID1) != CDI_REAL) gridsizemax *= 2;
    array.resize(gridsizemax);
  }

  const auto nvars = vlistNvars(vlistID1);
  int nconst = 0;
  for (varID = 0; varID < nvars; varID++)
    if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT) nconst++;

  FieldVector2D vars;
  if (nconst)
    {
      vars.resize(nvars);

      for (varID = 0; varID < nvars; varID++)
        {
          if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT)
            {
              const auto gridID = vlistInqVarGrid(vlistID1, varID);
              const auto nlevel = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));
              const auto gridsize = gridInqSize(gridID);

              vars[varID].resize(nlevel);

              for (levelID = 0; levelID < nlevel; levelID++)
                {
                  vars[varID][levelID].grid = gridID;
                  vars[varID][levelID].resize(gridsize);
                }
            }
        }
    }

  int tsID = 0;
  while (true)
    {
      const auto nrecs = cdo_stream_inq_timestep(streamID1, tsID);
      if (nrecs == 0) break;

      taxisCopyTimestep(taxisID2, taxisID1);
      const auto vdate = taxisInqVdate(taxisID1);
      const auto vtime = taxisInqVtime(taxisID1);

      if (operfunc == func_date)
        {
          index = (vdate / operintval) % 100;
          if (index < 0) index = -index;

          if (operatorID == SPLITSEAS) index = month_to_season(index);
        }
      else if (operfunc == func_time)
        {
          index = (vtime / operintval) % 100;
        }

      if (index < 0 || index >= MaxStreams) cdo_abort("Index out of range!");

      streamID2 = streamIDs[index];
      if (streamID2 == CDO_STREAM_UNDEF)
        {
          if (operatorID == SPLITSEAS)
            {
              sprintf(filename + nchars, "%3s", seas_name[index]);
              if (filesuffix[0]) sprintf(filename + nchars + 3, "%s", filesuffix);
            }
          else
            {
              size_t slen;
              char oformat[32];
              strcpy(oformat, "%02d");

              if (operatorID == SPLITMON && format)
                {
                  char sbuf[32];
                  auto stime = datetime_to_tm(vdate, vtime);
                  slen = strftime(sbuf, 32, format, &stime);

                  if (slen) strcpy(oformat, sbuf);
                }

              slen = sprintf(filename + nchars, oformat, index);
              if (filesuffix[0]) sprintf(filename + nchars + slen, "%s", filesuffix);
            }

          if (Options::cdoVerbose) cdo_print("create file %s", filename);

          streamID2 = cdo_open_write(filename);
          cdo_def_vlist(streamID2, vlistID2);
          streamIDs[index] = streamID2;
        }

      cdo_def_timestep(streamID2, tsIDs[index]);

      if (tsID > 0 && tsIDs[index] == 0 && nconst)
        {
          for (varID = 0; varID < nvars; varID++)
            {
              if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT)
                {
                  const auto nlevels = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));
                  for (levelID = 0; levelID < nlevels; levelID++)
                    {
                      cdo_def_record(streamID2, varID, levelID);
                      nmiss = vars[varID][levelID].nmiss;
                      cdo_write_record(streamID2, vars[varID][levelID].vec_d.data(), nmiss);
                    }
                }
            }
        }

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdo_inq_record(streamID1, &varID, &levelID);
          cdo_def_record(streamID2, varID, levelID);

          if (lcopy && !(tsID == 0 && nconst))
            {
              cdo_copy_record(streamID2, streamID1);
            }
          else
            {
              cdo_read_record(streamID1, array.data(), &nmiss);
              cdo_write_record(streamID2, array.data(), nmiss);

              if (tsID == 0 && nconst)
                {
                  if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT)
                    {
                      const auto gridID = vlistInqVarGrid(vlistID1, varID);
                      const auto gridsize = gridInqSize(gridID);
                      varray_copy(gridsize, array, vars[varID][levelID].vec_d);
                      vars[varID][levelID].nmiss = nmiss;
                    }
                }
            }
        }

      tsIDs[index]++;
      tsID++;
    }

  cdo_stream_close(streamID1);

  for (index = 0; index < MaxStreams; index++)
    {
      if (streamIDs[index] != CDO_STREAM_UNDEF) cdo_stream_close(streamIDs[index]);
    }

  vlistDestroy(vlistID2);

  cdo_finish();

  return nullptr;
}
