/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2021 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Setgatt    setgatt         Set global attribute
      Setgatt    setgatts        Set global attributes
*/

#include <cdi.h>

#include "process_int.h"
#include "readline.h"
#include <string.h>

void *
Setgatt(void *process)
{
  cdo_initialize(process);

  const auto SETGATT = cdo_operator_add("setgatt", 0, 0, "attribute name and string");
  cdo_operator_add("setgatts", 0, 0, nullptr);

  const auto operatorID = cdo_operator_id();

  char *attname = nullptr, *attstring = nullptr, *attfile = nullptr;  // TODO to std::string(all three)
  if (operatorID == SETGATT)
    {
      operator_input_arg(cdo_operator_enter(operatorID));
      attname = strdup(cdo_operator_argv(0).c_str());    // TODO replace attname through std::string
      attstring = strdup(cdo_operator_argv(1).c_str());  // TODO replace attstring through std::string
    }
  else
    {
      attfile = strdup(cdo_operator_argv(0).c_str());  // TODO replace attfile through std::string
    }

  const auto streamID1 = cdo_open_read(0);

  const auto vlistID1 = cdo_stream_inq_vlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdo_open_write(1);

  if (operatorID == SETGATT)
    {
      cdiDefAttTxt(vlistID2, CDI_GLOBAL, attname, (int) strlen(attstring), attstring);
    }
  else
    {
      char line[1024];
      int attlen = 0;

      auto fp = fopen(attfile, "r");
      if (fp == nullptr) cdo_abort("Open failed on %s", attfile);

      while (cdo::readline(fp, line, 1024))
        {
          attlen = 0;
          if (line[0] == '#') continue;
          if (line[0] == '\0') continue;
          attname = line;
          while (isspace((int) *attname)) attname++;
          if (attname[0] == '\0') continue;
          attstring = attname;
          while (*attstring != ' ' && *attstring != '\0' && *attstring != '=' && *attstring != '"') attstring++;
          if (*attstring == '\0')
            attstring = nullptr;
          else
            {
              *attstring = '\0';
              attstring++;
              while (isspace((int) *attstring) || (int) *attstring == '=' || (int) *attstring == '"' || (int) *attstring == '\'')
                attstring++;
              attlen = strlen(attstring);
              if (attstring[attlen - 1] == '"' || attstring[attlen - 1] == '\'') attstring[--attlen] = 0;
            }

          if (attstring && attlen) cdiDefAttTxt(vlistID2, CDI_GLOBAL, attname, attlen, attstring);
        }

      fclose(fp);
    }

  cdo_def_vlist(streamID2, vlistID2);

  auto gridsizemax = vlistGridsizeMax(vlistID1);
  if (vlistNumber(vlistID1) != CDI_REAL) gridsizemax *= 2;
  Varray<double> array(gridsizemax);

  int tsID = 0;
  while (true)
    {
      const auto nrecs = cdo_stream_inq_timestep(streamID1, tsID);
      if (nrecs == 0) break;

      taxisCopyTimestep(taxisID2, taxisID1);
      cdo_def_timestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          int varID, levelID;
          cdo_inq_record(streamID1, &varID, &levelID);
          cdo_def_record(streamID2, varID, levelID);

          size_t nmiss;
          cdo_read_record(streamID1, array.data(), &nmiss);
          cdo_write_record(streamID2, array.data(), nmiss);
        }

      tsID++;
    }

  if (attname != nullptr) free(attname);
  if (attstring != nullptr) free(attstring);
  if (attfile != nullptr) free(attfile);

  cdo_stream_close(streamID1);
  cdo_stream_close(streamID2);

  cdo_finish();

  return nullptr;
}
