/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2021 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include <cdi.h>

#include "cdo_options.h"
#include "process_int.h"
#include "param_conversion.h"
#include "parse_literals.h"
#include "pmlist.h"
#include "util_wildcards.h"

static void
set_attributes(const KVList &kvlist, int vlistID)
{
  constexpr int Undefined = -99;
  constexpr int delim = '@';
  const auto nvars = vlistNvars(vlistID);
  const auto ngrids = vlistNgrids(vlistID);
  const auto nzaxis = vlistNzaxis(vlistID);
  const auto maxvars = nvars + ngrids * 2 + nzaxis;
  std::vector<int> varIDs(maxvars);

  const int kvn = kvlist.size();
  std::vector<char *> wname(kvn, nullptr);
  for (int i = 0; i < kvn; ++i) wname[i] = nullptr;

  char name[CDI_MAX_NAME];
  char buffer[CDI_MAX_NAME];
  for (const auto &kv : kvlist)
    {
      int dtype = -1;
      char *varname = nullptr, *attname = nullptr;
      strcpy(buffer, kv.key.c_str());
      auto slen = strlen(buffer);
      if (slen >= 3)
        {
          if (buffer[slen - 2] == ':')
            {
              if (slen >= 4 && buffer[slen - 3] == '\\')
                {
                  for (int i = 2; i >= 0; i--) buffer[slen - i - 1] = buffer[slen - i];
                }
              else
                {
                  const auto type = buffer[slen - 1];
                  // clang-format off
                  if      (type == 's') dtype = 999;
                  else if (type == 'd') dtype = CDI_DATATYPE_FLT64;
                  else if (type == 'i') dtype = CDI_DATATYPE_INT32;
                  else cdo_abort("Attribute type '%c' not supported!", type);
                  // clang-format on
                  buffer[slen - 2] = 0;
                }
            }
        }
      char *const result = strrchr(buffer, delim);
      if (result == nullptr)
        {
          attname = buffer;
        }
      else
        {
          attname = result + 1;
          *result = 0;
          varname = buffer;
        }

      if (*attname == 0) cdo_abort("Attribute name missing in >%s<!", kv.key.c_str());

      int nv = 0;
      int cdiID = Undefined;
      if (varname && *varname)
        {
          for (int idx = 0; idx < nvars; idx++)
            {
              vlistInqVarName(vlistID, idx, name);
              if (wildcardmatch(varname, name) == 0)
                {
                  cdiID = vlistID;
                  varIDs[nv++] = idx;
                }
            }

          if (cdiID == Undefined)
            {
              int length = CDI_MAX_NAME;
              /*
              for ( int idx = 0; idx < ngrids; idx++ )
                {
                  int gridID = vlistGrid(vlistID, idx);
                  length = CDI_MAX_NAME;
                  cdiInqKeyString(gridID, CDI_XAXIS, CDI_KEY_NAME, name, &length);
                  if (wildcardmatch(varname, name) == 0)
                    {
                      cdiID = gridID;
                      varIDs[nv++] = CDI_GLOBAL;
                    }
                  length = CDI_MAX_NAME;
                  cdiInqKeyString(gridID, CDI_YAXIS, CDI_KEY_NAME, name, &length);
                  if (wildcardmatch(varname, name) == 0)
                    {
                      cdiID = gridID;
                      varIDs[nv++] = CDI_GLOBAL;
                    }
                }
              */
              for (int idx = 0; idx < nzaxis; idx++)
                {
                  const auto zaxisID = vlistZaxis(vlistID, idx);
                  length = CDI_MAX_NAME;
                  cdiInqKeyString(zaxisID, CDI_GLOBAL, CDI_KEY_NAME, name, &length);
                  if (wildcardmatch(varname, name) == 0)
                    {
                      cdiID = zaxisID;
                      varIDs[nv++] = CDI_GLOBAL;
                    }
                }
            }

          if (cdiID == Undefined)
            {
              bool lwarn = true;
              for (int i = 0; i < kvn; ++i)
                {
                  if (wname[i] == nullptr)
                    {
                      wname[i] = strdup(varname);
                      break;
                    }
                  if (cdo_cmpstr(wname[i], varname))
                    {
                      lwarn = false;
                      break;
                    }
                }
              if (lwarn) cdo_warning("Variable >%s< not found!", varname);
            }
        }
      else
        {
          cdiID = vlistID;
          varIDs[nv++] = CDI_GLOBAL;
        }

      if (cdiID != Undefined && nv > 0)
        {
          const auto &values = kv.values;
          const auto &value = kv.values[0];
          int nvalues = kv.nvalues;
          if (nvalues == 1 && value.empty()) nvalues = 0;

          if (dtype == -1) dtype = literals_find_datatype(nvalues, values);

          for (int idx = 0; idx < nv; ++idx)
            {
              const auto varID = varIDs[idx];
              // if (Options::cdoVerbose) printf("varID, cdiID, attname %d %d %s %d\n", varID, cdiID, attname,
              // (int)strlen(attname));
              if (nvalues == 0)
                {
                  const auto status = cdiDelAtt(cdiID, varID, attname);
                  if (status != CDI_NOERR)  // check CDI keys
                    {
                      // clang-format off
                      if      (cdo_cmpstr(attname, "long_name")) cdiDeleteKey(cdiID, varID, CDI_KEY_LONGNAME);
                      else if (cdo_cmpstr(attname, "units"))     cdiDeleteKey(cdiID, varID, CDI_KEY_UNITS);
                      // clang-format on
                    }
                }
              else
                {
                  if (dtype == CDI_DATATYPE_INT8 || dtype == CDI_DATATYPE_INT16 || dtype == CDI_DATATYPE_INT32)
                    {
                      std::vector<int> ivals(nvalues);
                      for (int i = 0; i < nvalues; ++i) ivals[i] = literal_to_int(values[i]);
                      cdiDefAttInt(cdiID, varID, attname, dtype, nvalues, ivals.data());
                    }
                  else if (dtype == CDI_DATATYPE_FLT32 || dtype == CDI_DATATYPE_FLT64)
                    {
                      Varray<double> dvals(nvalues);
                      for (int i = 0; i < nvalues; ++i) dvals[i] = literal_to_double(values[i]);
                      cdiDefAttFlt(cdiID, varID, attname, dtype, nvalues, dvals.data());
                    }
                  else
                    {
                      if (nvalues > 1)
                        cdo_abort("Multidimensional string attributes not supported! %s=\"%s\"", attname, values[1].c_str());
                      const auto len = (int) value.size();
                      int outlen = 0;
                      std::vector<char> outvalue(len);
                      for (int i = 0; i < len; ++i)
                        {
                          if (i > 0 && value[i - 1] == '\\' && value[i] == 'n')
                            outvalue[outlen - 1] = '\n';
                          else
                            outvalue[outlen++] = value[i];
                        }
                      cdiDefAttTxt(cdiID, varID, attname, outlen, outvalue.data());
                    }
                }
            }
        }
    }

  for (int i = 0; i < kvn; ++i)
    if (wname[i]) free(wname[i]);
}

void *
Setattribute(void *process)
{
  cdo_initialize(process);

  cdo_operator_add("setattribute", 0, 0, "attributes");

  const auto lcopy = unchanged_record();

  const auto operatorID = cdo_operator_id();

  operator_input_arg(cdo_operator_enter(operatorID));

  const auto natts = cdo_operator_argc();
  if (natts == 0) cdo_abort("Parameter missing!");

  PMList pmlist;
  KVList kvlist;
  kvlist.name = "SETATTRIBUTE";
  if (kvlist.parse_arguments(natts, cdo_get_oper_argv()) != 0) cdo_abort("Parse error!");
  if (Options::cdoVerbose) kvlist.print();

  auto pkvlist = &kvlist;
  if (natts == 1)
    {
      KeyValues &kv = kvlist.front();
      if (kv.key == "FILE")
        {
          if (Options::cdoVerbose) cdo_print("Reading attributes from: %s", kv.values[0].c_str());
          auto filename = parameter_to_word(kv.values[0].c_str());
          auto fp = fopen(filename, "r");
          if (fp == nullptr) cdo_abort("Open failed on: %s\n", filename);
          pmlist.read_namelist(fp, filename);
          pkvlist = &pmlist.front();
          if (pkvlist == nullptr) cdo_abort("Parse error!");
          fclose(fp);
          if (Options::cdoVerbose) pkvlist->print();
        }
    }

  const auto streamID1 = cdo_open_read(0);

  const auto vlistID1 = cdo_stream_inq_vlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  set_attributes(*pkvlist, vlistID2);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdo_open_write(1);
  cdo_def_vlist(streamID2, vlistID2);

  Field field;

  VarList varList1;
  varListInit(varList1, vlistID1);

  int tsID = 0;
  while (true)
    {
      const auto nrecs = cdo_stream_inq_timestep(streamID1, tsID);
      if (nrecs == 0) break;

      taxisCopyTimestep(taxisID2, taxisID1);
      cdo_def_timestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          int varID, levelID;
          cdo_inq_record(streamID1, &varID, &levelID);
          cdo_def_record(streamID2, varID, levelID);

          if (lcopy)
            {
              cdo_copy_record(streamID2, streamID1);
            }
          else
            {
              field.init(varList1[varID]);
              cdo_read_record(streamID1, field);
              cdo_write_record(streamID2, field);
            }
        }

      tsID++;
    }

  cdo_stream_close(streamID1);
  cdo_stream_close(streamID2);

  cdo_finish();

  return nullptr;
}
