/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include <cdi.h>

#include <vector>
#include <algorithm>

#include "functs.h"
#include "process_int.h"
#include "percentiles.h"

template <typename T>
static void
varrayCopyZon(size_t offset, size_t nx, const Varray<T> &v1, Varray<double> &v2)
{
  std::copy(v1.begin() + offset, v1.begin() + offset + nx, v2.begin());
}

static size_t
fillReducedPoints(const int gridID, const size_t ny, std::vector<int> &reducedPoints, std::vector<int> &cumreducedPoints)
{
  reducedPoints.resize(ny);
  cumreducedPoints.resize(ny);
  gridInqReducedPoints(gridID, reducedPoints.data());
  cumreducedPoints[0] = 0;
  for (size_t j = 1; j < ny; j++) cumreducedPoints[j] = cumreducedPoints[j - 1] + reducedPoints[j - 1];
  size_t nx = reducedPoints[0];
  for (size_t j = 1; j < ny; j++)
    if (reducedPoints[j] > (int) nx) nx = reducedPoints[j];
  return nx;
}

using funcType = double (size_t, const Varray<double> &);
using funcTypeMV = double (size_t, const Varray<double> &, double);
using funcTypeNmissMV = double (size_t, const Varray<double> &, size_t, double);

static void
fieldZonKernel1(const Field &field1, Field &field2, funcTypeNmissMV funcNmissMV)
{
  size_t rnmiss = 0;
  const auto nmiss = field1.nmiss;
  const auto missval = field1.missval;

  const auto ny = gridInqYsize(field1.grid);
  const auto lreduced = (gridInqType(field1.grid) == GRID_GAUSSIAN_REDUCED);
  std::vector<int> reducedPoints, cumreducedPoints;
  auto nx = lreduced ? fillReducedPoints(field1.grid, ny, reducedPoints, cumreducedPoints) : gridInqXsize(field1.grid);

  Varray<double> v(nx);

  for (size_t j = 0; j < ny; ++j)
    {
      if (lreduced) nx = reducedPoints[j];
      const size_t offset = lreduced ? cumreducedPoints[j] : j * nx;
      if (field1.memType == MemType::Float)
        varrayCopyZon(offset, nx, field1.vec_f, v);
      else
        varrayCopyZon(offset, nx, field1.vec_d, v);

      const auto result = funcNmissMV(nx, v, nmiss, missval);
      if (DBL_IS_EQUAL(result, missval)) rnmiss++;
      field2.vec_d[j] = result;
    }

  field2.nmiss = rnmiss;
}

static void
fieldZonKernel2(const Field &field1, Field &field2, funcType func, funcTypeMV funcMV)
{
  size_t rnmiss = 0;
  const auto nmiss = field1.nmiss;
  const auto missval = field1.missval;

  const auto ny = gridInqYsize(field1.grid);
  const auto lreduced = (gridInqType(field1.grid) == GRID_GAUSSIAN_REDUCED);
  std::vector<int> reducedPoints, cumreducedPoints;
  auto nx = lreduced ? fillReducedPoints(field1.grid, ny, reducedPoints, cumreducedPoints) : gridInqXsize(field1.grid);

  Varray<double> v(nx);

  for (size_t j = 0; j < ny; ++j)
    {
      if (lreduced) nx = reducedPoints[j];
      const size_t offset = lreduced ? cumreducedPoints[j] : j * nx;
      if (field1.memType == MemType::Float)
        varrayCopyZon(offset, nx, field1.vec_f, v);
      else
        varrayCopyZon(offset, nx, field1.vec_d, v);

      const auto result = nmiss ? funcMV(nx, v, missval) : func(nx, v);
      if (DBL_IS_EQUAL(result, missval)) rnmiss++;
      field2.vec_d[j] = result;
    }

  field2.nmiss = rnmiss;
}

void
field_zon_min(const Field &field1, Field &field2)
{
  fieldZonKernel2(field1, field2, varray_min, varray_min_mv);
}

void
field_zon_max(const Field &field1, Field &field2)
{
  fieldZonKernel2(field1, field2, varray_max, varray_max_mv);
}

void
field_zon_range(const Field &field1, Field &field2)
{
  fieldZonKernel2(field1, field2, varray_range, varray_range_mv);
}

void
field_zon_sum(const Field &field1, Field &field2)
{
  fieldZonKernel2(field1, field2, varray_sum, varray_sum_mv);
}

void
field_zon_mean(const Field &field1, Field &field2)
{
  fieldZonKernel2(field1, field2, varray_mean, varray_mean_mv);
}

void
field_zon_avg(const Field &field1, Field &field2)
{
  fieldZonKernel2(field1, field2, varray_mean, varray_avg_mv);
}

void
field_zon_var(const Field &field1, Field &field2)
{
  fieldZonKernel1(field1, field2, varray_var);
}

void
field_zon_var1(const Field &field1, Field &field2)
{
  fieldZonKernel1(field1, field2, varray_var_1);
}

void
field_zon_std(const Field &field1, Field &field2)
{
  size_t rnmiss = 0;
  const auto missval = field1.missval;
  const auto ny = gridInqYsize(field1.grid);

  field_zon_var(field1, field2);

  for (size_t j = 0; j < ny; j++)
    {
      const auto rstd = varToStd(field2.vec_d[j], missval);
      if (DBL_IS_EQUAL(rstd, missval)) rnmiss++;
      field2.vec_d[j] = rstd;
    }

  field2.nmiss = rnmiss;
}

void
field_zon_std1(const Field &field1, Field &field2)
{
  size_t rnmiss = 0;
  const auto missval = field1.missval;
  const auto ny = gridInqYsize(field1.grid);

  field_zon_var1(field1, field2);

  for (size_t j = 0; j < ny; j++)
    {
      const auto rstd = varToStd(field2.vec_d[j], missval);
      if (DBL_IS_EQUAL(rstd, missval)) rnmiss++;
      field2.vec_d[j] = rstd;
    }

  field2.nmiss = rnmiss;
}

void
field_zon_kurt(const Field &field1, Field &field2)
{
  fieldZonKernel1(field1, field2, varray_kurt);
}

void
field_zon_skew(const Field &field1, Field &field2)
{
  fieldZonKernel1(field1, field2, varray_skew);
}

void
field_zon_pctl(const Field &field1, Field &field2, const int p)
{
  size_t rnmiss = 0;
  const auto missval = field1.missval;

  const auto ny = gridInqYsize(field1.grid);
  const auto lreduced = (gridInqType(field1.grid) == GRID_GAUSSIAN_REDUCED);
  std::vector<int> reducedPoints, cumreducedPoints;
  auto nx = lreduced ? fillReducedPoints(field1.grid, ny, reducedPoints, cumreducedPoints) : gridInqXsize(field1.grid);

  Varray<double> v(nx);

  if (field1.nmiss)
    {
      for (size_t j = 0; j < ny; j++)
        {
          if (lreduced) nx = reducedPoints[j];
          const size_t offset = lreduced ? cumreducedPoints[j] : j * nx;
          if (field1.memType == MemType::Float)
            varrayCopyZon(offset, nx, field1.vec_f, v);
          else
            varrayCopyZon(offset, nx, field1.vec_d, v);

          size_t k = 0;
          for (size_t i = 0; i < nx; i++)
            if (!DBL_IS_EQUAL(v[i], missval)) v[k++] = v[i];

          if (k > 0)
            {
              field2.vec_d[j] = percentile(v.data(), k, p);
            }
          else
            {
              field2.vec_d[j] = missval;
              rnmiss++;
            }
        }
    }
  else
    {
      for (size_t j = 0; j < ny; j++)
        {
          if (lreduced) nx = reducedPoints[j];
          const size_t offset = lreduced ? cumreducedPoints[j] : j * nx;
          if (field1.memType == MemType::Float)
            varrayCopyZon(offset, nx, field1.vec_f, v);
          else
            varrayCopyZon(offset, nx, field1.vec_d, v);

          if (nx > 0)
            {
              field2.vec_d[j] = percentile(v.data(), nx, p);
            }
          else
            {
              field2.vec_d[j] = missval;
              rnmiss++;
            }
        }
    }

  field2.nmiss = rnmiss;
}

void
field_zon_function(const Field &field1, Field &field2, int function)
{
  // clang-format off
  switch (function)
    {
    case func_min:    return field_zon_min(field1, field2);
    case func_max:    return field_zon_max(field1, field2);
    case func_range:  return field_zon_range(field1, field2);
    case func_sum:    return field_zon_sum(field1, field2);
    case func_mean:   return field_zon_mean(field1, field2);
    case func_avg:    return field_zon_avg(field1, field2);
    case func_std:    return field_zon_std(field1, field2);
    case func_std1:   return field_zon_std1(field1, field2);
    case func_var:    return field_zon_var(field1, field2);
    case func_var1:   return field_zon_var1(field1, field2);
    case func_kurt:   return field_zon_kurt(field1, field2);
    case func_skew:   return field_zon_skew(field1, field2);
    default: cdo_abort("%s: function %d not implemented!",  __func__, function);
    }
  // clang-format on
}
