/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/
#ifndef CDO_OUTPUT_H
#define CDO_OUTPUT_H

#include "mpmo.h"

// Debug Switches
extern int cdoDebug;
extern int cdoDebugExt;  //  Debug level for the KNMI extensions
                         // Subsystem Debug Switches
extern int PROCESS;
extern int PIPE;
extern int PIPE_STREAM;
extern int FILE_STREAM;
extern int PTHREAD;
extern int PROCESS_MANAGER;
extern int PIPE;

void print_debug_options();
void query_user_exit(const char *argument);

namespace cdo
{
void set_debug(int p_debug_level);
extern void (*exitProgram)(void);
extern const char *(*getContext)(void);
void set_exit_function(void (*func)(void));
void set_context_function(const char *(*func)(void) );
}  // namespace cdo

void cdi_open_error(int cdiErrno, const std::string &format, const char *path);

template <typename... Args>
void
cdo_abort(const std::string &format, Args const &... args) noexcept
{
  fflush(stdout);
  MpMO::PrintCerr(Red("\n%s (Abort): ") + format, cdo::getContext(), args...);
  if (MpMO::exitOnError) cdo::exitProgram();
}

template <typename... Args>
void
cdo_warning(const std::string &format, Args const &... args) noexcept
{
  if (MpMO::warningsEnabled)
    {
      if (MpMO::pedantic)
        {
          MpMO::PrintCerr(Red("%s (Warning): ") + format, cdo::getContext(), args...);
          if (MpMO::exitOnError) cdo::exitProgram();
        }
      else
        {
          MpMO::PrintCerr(Yellow("%s (Warning): ") + format, cdo::getContext(), args...);
        }
    }
}

template <typename... Args>
void
cdo_print(const std::string &format, Args const &... args) noexcept
{
  if (!MpMO::silentMode) MpMO::PrintCerr(Green("%s: ") + format, cdo::getContext(), args...);
}

#ifdef WITH_CALLER_NAME
#define cdo_sys_error(...) MpMO::SysError_(__func__, __VA_ARGS__)
#else
#define cdo_sys_error(...) MpMO::SysError_("", __VA_ARGS__)
#endif

#endif
