/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

   Yhourstat     yhourmin        Multi-year hourly minimum
   Yhourstat     yhourmax        Multi-year hourly maximum
   Yhourstat     yhourrange      Multi-year hourly range
   Yhourstat     yhoursum        Multi-year hourly sum
   Yhourstat     yhourmean       Multi-year hourly mean
   Yhourstat     yhouravg        Multi-year hourly average
   Yhourstat     yhourstd        Multi-year hourly standard deviation
   Yhourstat     yhourstd1       Multi-year hourly standard deviation (n-1)
   Yhourstat     yhourvar        Multi-year hourly variance
   Yhourstat     yhourvar1       Multi-year hourly variance (n-1)

   Dhourstat     dhourmin        Multi-day hourly minimum
   Dhourstat     dhourmax        Multi-day hourly maximum
   Dhourstat     dhourrange      Multi-day hourly range
   Dhourstat     dhoursum        Multi-day hourly sum
   Dhourstat     dhourmean       Multi-day hourly mean
   Dhourstat     dhouravg        Multi-day hourly average
   Dhourstat     dhourstd        Multi-day hourly standard deviation
   Dhourstat     dhourstd1       Multi-day hourly standard deviation (n-1)
   Dhourstat     dhourvar        Multi-day hourly variance
   Dhourstat     dhourvar1       Multi-day hourly variance (n-1)
*/

#include <cdi.h>

#include "cdo_options.h"
#include "datetime.h"
#include "functs.h"
#include "process_int.h"
#include "printinfo.h"

static void
addOperators(void)
{
  // clang-format off
  cdo_operator_add("yhourrange", func_range, 0, nullptr);
  cdo_operator_add("yhourmin",   func_min,   0, nullptr);
  cdo_operator_add("yhourmax",   func_max,   0, nullptr);
  cdo_operator_add("yhoursum",   func_sum,   0, nullptr);
  cdo_operator_add("yhourmean",  func_mean,  0, nullptr);
  cdo_operator_add("yhouravg",   func_avg,   0, nullptr);
  cdo_operator_add("yhourvar",   func_var,   0, nullptr);
  cdo_operator_add("yhourvar1",  func_var1,  0, nullptr);
  cdo_operator_add("yhourstd",   func_std,   0, nullptr);
  cdo_operator_add("yhourstd1",  func_std1,  0, nullptr);

  cdo_operator_add("dhourrange", func_range, 0, nullptr);
  cdo_operator_add("dhourmin",   func_min,   1, nullptr);
  cdo_operator_add("dhourmax",   func_max,   1, nullptr);
  cdo_operator_add("dhoursum",   func_sum,   1, nullptr);
  cdo_operator_add("dhourmean",  func_mean,  1, nullptr);
  cdo_operator_add("dhouravg",   func_avg,   1, nullptr);
  cdo_operator_add("dhourvar",   func_var,   1, nullptr);
  cdo_operator_add("dhourvar1",  func_var1,  1, nullptr);
  cdo_operator_add("dhourstd",   func_std,   1, nullptr);
  cdo_operator_add("dhourstd1",  func_std1,  1, nullptr);
  // clang-format on
}

void *
Yhourstat(void *process)
{
  TimeStat timestat_date = TimeStat::LAST;

  cdo_initialize(process);

  addOperators();

  const auto operatorID = cdo_operator_id();
  const auto operfunc = cdo_operator_f1(operatorID);

  const auto ldaily = (cdo_operator_f2(operatorID) == 1);

  const auto lrange = (operfunc == func_range);
  const auto lmean = (operfunc == func_mean || operfunc == func_avg);
  const auto lstd = (operfunc == func_std || operfunc == func_std1);
  const auto lvarstd = (lstd || operfunc == func_var || operfunc == func_var1);
  const auto lvars2 = (lvarstd || lrange);
  const int divisor = (operfunc == func_std1 || operfunc == func_var1);

  auto vfarstdvar_func = lstd ? vfarstd : vfarvar;
  auto vfarcstdvar_func = lstd ? vfarcstd : vfarcvar;

  operator_check_argc(0);

  const auto streamID1 = cdo_open_read(0);

  const auto vlistID1 = cdo_stream_inq_vlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  taxisWithBounds(taxisID2);
  if (taxisInqType(taxisID2) == TAXIS_FORECAST) taxisDefType(taxisID2, TAXIS_RELATIVE);

  const int MaxHours = ldaily ? 25 : 9301;     // year: 31*12*25 + 1
  std::vector<int> hourot_nsets(MaxHours, 0);  // hour of time
  FieldVector3D vars1(MaxHours), vars2(MaxHours), samp1(MaxHours);

  std::vector<DateTimeList> dtlist(MaxHours);
  for (int hourot = 0; hourot < MaxHours; ++hourot)
    {
      dtlist[hourot].set_stat(timestat_date);
      dtlist[hourot].set_calendar(taxisInqCalendar(taxisID1));
    }

  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdo_open_write(1);
  cdo_def_vlist(streamID2, vlistID2);

  const auto maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  VarList varList;
  varListInit(varList, vlistID1);

  int VARS_MEMTYPE = 0;
  if ((operfunc == func_min) || (operfunc == func_max)) VARS_MEMTYPE = FIELD_NAT;

  Field field;

  int tsID = 0;
  int otsID = 0;
  while (true)
    {
      const auto nrecs = cdo_stream_inq_timestep(streamID1, tsID);
      if (nrecs == 0) break;

      const auto vdate = taxisInqVdate(taxisID1);
      const auto vtime = taxisInqVtime(taxisID1);

      if (Options::cdoVerbose) cdo_print("process timestep: %d %d %d", tsID + 1, vdate, vtime);

      const auto hourot = ldaily ? decode_hour_of_day(vdate, vtime) : decode_hour_of_year(vdate, vtime);
      if (hourot < 0 || hourot >= MaxHours)
        cdo_abort("Hour of year %d out of range (%s %s)!", hourot, date_to_string(vdate).c_str(), time_to_string(vtime).c_str());

      dtlist[hourot].taxis_inq_timestep(taxisID1, hourot_nsets[hourot]);

      if (!vars1[hourot].size())
        {
          fields_from_vlist(vlistID1, samp1[hourot]);
          fields_from_vlist(vlistID1, vars1[hourot], FIELD_VEC | VARS_MEMTYPE);
          if (lvars2) fields_from_vlist(vlistID1, vars2[hourot], FIELD_VEC);
        }

      for (int recID = 0; recID < nrecs; recID++)
        {
          int varID, levelID;
          cdo_inq_record(streamID1, &varID, &levelID);

          if (tsID == 0)
            {
              recList[recID].varID = varID;
              recList[recID].levelID = levelID;
              recList[recID].lconst = (varList[varID].timetype == TIME_CONSTANT);
            }

          auto &rsamp1 = samp1[hourot][varID][levelID];
          auto &rvars1 = vars1[hourot][varID][levelID];

          const auto nsets = hourot_nsets[hourot];

          if (nsets == 0)
            {
              cdo_read_record(streamID1, rvars1);
              if (lrange)
                {
                  vars2[hourot][varID][levelID].nmiss = rvars1.nmiss;
                  vars2[hourot][varID][levelID].vec_d = rvars1.vec_d;
                }

              if (rvars1.nmiss || !rsamp1.empty())
                {
                  if (rsamp1.empty()) rsamp1.resize(rvars1.size);
                  vfarvinit(rsamp1, rvars1);
                }
            }
          else
            {
              field.init(varList[varID]);
              cdo_read_record(streamID1, field);

              if (field.nmiss || !rsamp1.empty())
                {
                  if (rsamp1.empty()) rsamp1.resize(rvars1.size, nsets);
                  vfarvincr(rsamp1, field);
               }

              // clang-format off
              if      (lvarstd) vfarsumsumq(rvars1, vars2[hourot][varID][levelID], field);
              else if (lrange)  vfarmaxmin(rvars1, vars2[hourot][varID][levelID], field);
              else              vfarfun(rvars1, field, operfunc);
              // clang-format on
            }
        }

      if (hourot_nsets[hourot] == 0 && lvarstd)
        for (int recID = 0; recID < maxrecs; recID++)
          {
            if (recList[recID].lconst) continue;

            const auto varID = recList[recID].varID;
            const auto levelID = recList[recID].levelID;
            vfarmoq(vars2[hourot][varID][levelID], vars1[hourot][varID][levelID]);
          }

      hourot_nsets[hourot]++;
      tsID++;
    }

  for (int hourot = 0; hourot < MaxHours; ++hourot)
    if (hourot_nsets[hourot])
      {
        const auto nsets = hourot_nsets[hourot];
        for (int recID = 0; recID < maxrecs; recID++)
          {
            if (recList[recID].lconst) continue;

            const auto varID = recList[recID].varID;
            const auto levelID = recList[recID].levelID;
            const auto &rsamp1 = samp1[hourot][varID][levelID];
            auto &rvars1 = vars1[hourot][varID][levelID];

            if (lmean)
              {
                if (!rsamp1.empty())
                  vfardiv(rvars1, rsamp1);
                else
                  vfarcdiv(rvars1, (double) nsets);
              }
            else if (lvarstd)
              {
                if (!rsamp1.empty())
                  vfarstdvar_func(rvars1, vars2[hourot][varID][levelID], rsamp1, divisor);
                else
                  vfarcstdvar_func(rvars1, vars2[hourot][varID][levelID], nsets, divisor);
              }
            else if (lrange)
              {
                vfarsub(rvars1, vars2[hourot][varID][levelID]);
              }
          }

        dtlist[hourot].stat_taxis_def_timestep(taxisID2, hourot_nsets[hourot]);
        cdo_def_timestep(streamID2, otsID);

        for (int recID = 0; recID < maxrecs; recID++)
          {
            if (otsID && recList[recID].lconst) continue;

            const auto varID = recList[recID].varID;
            const auto levelID = recList[recID].levelID;
            auto &rvars1 = vars1[hourot][varID][levelID];

            cdo_def_record(streamID2, varID, levelID);
            cdo_write_record(streamID2, rvars1);
          }

        otsID++;
      }

  cdo_stream_close(streamID2);
  cdo_stream_close(streamID1);

  cdo_finish();

  return nullptr;
}
