/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Yearmonstat   yearmonmean        Yearly mean from monthly data
      Yearmonstat   yearmonavg         Yearly average from monthly data
*/

#include <cdi.h>

#include "cdo_options.h"
#include "cdo_vlist.h"
#include "functs.h"
#include "process_int.h"
#include "calendar.h"
#include "datetime.h"
#include "printinfo.h"


// init valid values
template <typename T1, typename T2>
static void
vfarvinit(const size_t len, Varray<T1> &v1, const Varray<T2> &v2, const T2 missval, int dpm)
{
  for (size_t i = 0; i < len; i++)
    v1[i] = (DBL_IS_EQUAL(v2[i], missval)) ? 0 : dpm;
}

static void
vfarvinit(Field &field1, const Field &field2, int dpm)
{
  if (field1.size != field2.size) cdo_abort("Fields have different size (%s)", __func__);

  if (field1.memType == MemType::Float && field2.memType == MemType::Float)
    vfarvinit(field1.size, field1.vec_f, field2.vec_f, (float)field1.missval, dpm);
  else if (field1.memType == MemType::Double && field2.memType == MemType::Float)
    vfarvinit(field1.size, field1.vec_d, field2.vec_f, (float)field1.missval, dpm);
  else
    vfarvinit(field1.size, field1.vec_d, field2.vec_d, field1.missval, dpm);

  field1.nmiss = field2.nmiss;
}

// increment valid values
template <typename T1, typename T2>
static void
vfarvincr(const size_t len, Varray<T1> &v1, const Varray<T2> &v2, const T2 missval, int dpm)
{
  for (size_t i = 0; i < len; i++)
    if (!DBL_IS_EQUAL(v2[i], missval)) v1[i] += dpm;
}

static void
vfarvincr(Field &field1, const Field &field2, int dpm)
{
  if (field1.size != field2.size) cdo_abort("Fields have different size (%s)", __func__);

  if (field1.memType == MemType::Float && field2.memType == MemType::Float)
    vfarvincr(field1.size, field1.vec_f, field2.vec_f, (float)field1.missval, dpm);
  else if (field1.memType == MemType::Double && field2.memType == MemType::Float)
    vfarvincr(field1.size, field1.vec_d, field2.vec_f, (float)field1.missval, dpm);
  else
    vfarvincr(field1.size, field1.vec_d, field2.vec_d, field1.missval, dpm);

  field1.nmiss = field2.nmiss;
}

void *
Yearmonstat(void *process)
{
  TimeStat timestat_date = TimeStat::MEAN;
  int64_t vdate0 = 0;
  int vtime0 = 0;
  int year0 = 0, month0 = 0;
  int year, month, day;

  cdo_initialize(process);

  // clang-format off
  cdo_operator_add("yearmonmean",  func_mean, 0, nullptr);
  cdo_operator_add("yearmonavg",   func_avg,  0, nullptr);
  // clang-format on

  const auto operatorID = cdo_operator_id();
  const auto operfunc = cdo_operator_f1(operatorID);

  operator_check_argc(0);

  const auto streamID1 = cdo_open_read(0);

  const auto vlistID1 = cdo_stream_inq_vlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  taxisWithBounds(taxisID2);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdo_open_write(1);
  cdo_def_vlist(streamID2, vlistID2);

  const auto nvars = vlistNvars(vlistID1);

  const auto maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  const auto calendar = taxisInqCalendar(taxisID1);
  DateTimeList dtlist;
  dtlist.set_stat(timestat_date);
  dtlist.set_calendar(calendar);

  VarList varList;
  varListInit(varList, vlistID1);

  Field field;

  FieldVector2D samp1, vars1;
  fields_from_vlist(vlistID1, samp1);
  fields_from_vlist(vlistID1, vars1, FIELD_VEC);

  int tsID = 0;
  int otsID = 0;
  while (true)
    {
      int nrecs = 0;
      long nsets = 0;
      double dsets = 0.0;
      while (true)
        {
          nrecs = cdo_stream_inq_timestep(streamID1, tsID);
          if (nrecs == 0) break;

          dtlist.taxis_inq_timestep(taxisID1, nsets);
          const auto vdate = dtlist.get_vdate(nsets);
          const auto vtime = dtlist.get_vtime(nsets);
          cdiDecodeDate(vdate, &year, &month, &day);

          if (nsets == 0) year0 = year;

          if (year != year0) break;

          if (nsets > 0 && month == month0)
            {
              cdo_warning("   last timestep: %s %s", date_to_string(vdate0).c_str(), time_to_string(vtime0).c_str());
              cdo_warning("current timestep: %s %s", date_to_string(vdate).c_str(), time_to_string(vtime).c_str());
              cdo_abort("Month does not change!");
            }

          const auto dpm = days_per_month(calendar, year, month);

          for (int recID = 0; recID < nrecs; recID++)
            {
              int varID, levelID;
              cdo_inq_record(streamID1, &varID, &levelID);

              if (tsID == 0)
                {
                  recList[recID].varID = varID;
                  recList[recID].levelID = levelID;
                  recList[recID].lconst = (varList[varID].timetype == TIME_CONSTANT);
                }

              auto &rsamp1 = samp1[varID][levelID];
              auto &rvars1 = vars1[varID][levelID];

              if (nsets == 0)
                {
                  cdo_read_record(streamID1, rvars1);

                  vfarcmul(rvars1, dpm);

                  if (rvars1.nmiss || !rsamp1.empty())
                    {
                      if (rsamp1.empty()) rsamp1.resize(rvars1.size);
                      vfarvinit(rsamp1, rvars1, dpm);
                    }
                }
              else
                {
                  field.init(varList[varID]);
                  cdo_read_record(streamID1, field);

                  vfarcmul(field, dpm);

                  if (field.nmiss || !rsamp1.empty())
                    {
                      if (rsamp1.empty()) rsamp1.resize(rvars1.size, dsets);
                      vfarvincr(rsamp1, field, dpm);
                    }

                  vfarfun(rvars1, field, operfunc);
                }
            }

          month0 = month;
          vdate0 = vdate;
          vtime0 = vtime;
          nsets++;
          dsets += dpm;
          tsID++;
        }

      if (nrecs == 0 && nsets == 0) break;

      for (int varID = 0; varID < nvars; varID++)
        {
          if (varList[varID].timetype == TIME_CONSTANT) continue;
          const auto nlevels = varList[varID].nlevels;
          for (int levelID = 0; levelID < nlevels; levelID++)
            {
              const auto &rsamp1 = samp1[varID][levelID];
              auto &rvars1 = vars1[varID][levelID];
              if (!rsamp1.empty())
                vfardiv(rvars1, rsamp1);
              else
                vfarcdiv(rvars1, dsets);
            }
        }

      if (Options::cdoVerbose)
        cdo_print("%s %s  nsets = %ld", date_to_string(vdate0).c_str(), time_to_string(vtime0).c_str(), nsets);

      dtlist.stat_taxis_def_timestep(taxisID2, nsets);
      cdo_def_timestep(streamID2, otsID);

      for (int recID = 0; recID < maxrecs; recID++)
        {
          if (otsID && recList[recID].lconst) continue;

          const auto varID = recList[recID].varID;
          const auto levelID = recList[recID].levelID;
          auto &rvars1 = vars1[varID][levelID];
          cdo_def_record(streamID2, varID, levelID);
          cdo_write_record(streamID2, rvars1);
        }

      if (nrecs == 0) break;
      otsID++;
    }

  cdo_stream_close(streamID2);
  cdo_stream_close(streamID1);

  cdo_finish();

  return nullptr;
}
