/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Pressure    pressure_fl          Pressure on full hybrid levels
      Pressure    pressure_hl          Pressure on half hybrid levels
      Pressure    deltap               Difference of two half hybrid levels
*/

#include <cdi.h>

#include "cdo_options.h"
#include "process_int.h"
#include "cdo_vlist.h"
#include "vertical_interp.h"
#include "stdnametable.h"
#include "util_string.h"
#include "const.h"

void *
Pressure(void *process)
{
  ModelMode mode(ModelMode::UNDEF);
  int k;
  int zaxisIDp, zaxisIDh = -1;
  int nhlevf = 0, nhlevh = 0, nlevel = 0;
  int nvct = 0;
  int psID = -1, lnpsID = -1;
  char paramstr[32];
  char varname[CDI_MAX_NAME];
  double *pout = nullptr;
  gribcode_t gribcodes;

  cdo_initialize(process);

  // clang-format off
  const auto PRESSURE_FL = cdo_operator_add("pressure_fl", 0, 0, nullptr);
  const auto PRESSURE_HL = cdo_operator_add("pressure_hl", 0, 0, nullptr);
  const auto DELTAP      = cdo_operator_add("deltap",      0, 0, nullptr);
  // clang-format on

  const auto operatorID = cdo_operator_id();

  operator_check_argc(0);

  const auto streamID1 = cdo_open_read(0);
  const auto vlistID1 = cdo_stream_inq_vlist(streamID1);

  const auto gridsize = vlist_check_gridsize(vlistID1);

  int nhlev;
  Varray<double> vct;
  vlist_read_vct(vlistID1, &zaxisIDh, &nvct, &nhlev, &nhlevf, &nhlevh, vct);

  bool l3Dvars = (zaxisIDh != -1 && gridsize > 0);
  if (!l3Dvars) cdo_abort("No 3D variable with hybrid sigma pressure coordinate found!");

  Varray<double> psProg(gridsize);
  Varray<double> deltap(gridsize * nhlevf), fullPress(gridsize * nhlevf), halfPress(gridsize * nhlevh);

  if (operatorID == PRESSURE_FL || operatorID == DELTAP)
    {
      if (Options::cdoVerbose) cdo_print("Creating ZAXIS_HYBRID .. (nhlevf=%d)", nhlevf);
      zaxisIDp = zaxisCreate(ZAXIS_HYBRID, nhlevf);
    }
  else
    {
      if (Options::cdoVerbose) cdo_print("Creating ZAXIS_HYBRID_HALF .. (nhlevh=%d)", nhlevh);
      zaxisIDp = zaxisCreate(ZAXIS_HYBRID_HALF, nhlevh);
    }

  Varray<double> level(nhlevh);
  for (int l = 0; l < nhlevh; l++) level[l] = l + 1;
  zaxisDefLevels(zaxisIDp, level.data());

  zaxisDefVct(zaxisIDp, 2 * nhlevh, vct.data());

  const auto nvars = vlistNvars(vlistID1);

  auto useTable = false;
  for (int varID = 0; varID < nvars; varID++)
    {
      const auto tableNum = tableInqNum(vlistInqVarTable(vlistID1, varID));
      if (tableNum > 0 && tableNum != 255)
        {
          useTable = true;
          break;
        }
    }

  if (Options::cdoVerbose && useTable) cdo_print("Use code tables!");

  for (int varID = 0; varID < nvars; varID++)
    {
      const auto zaxisID = vlistInqVarZaxis(vlistID1, varID);
      const auto nlevels = zaxisInqSize(zaxisID);
      const auto instNum = institutInqCenter(vlistInqVarInstitut(vlistID1, varID));
      const auto tableNum = tableInqNum(vlistInqVarTable(vlistID1, varID));

      auto code = vlistInqVarCode(vlistID1, varID);
      auto param = vlistInqVarParam(vlistID1, varID);

      cdiParamToString(param, paramstr, sizeof(paramstr));

      if (useTable)
        {
          if (tableNum == 2)
            {
              mode = ModelMode::WMO;
              wmo_gribcodes(&gribcodes);
            }
          else if (tableNum == 128)
            {
              mode = ModelMode::ECHAM;
              echam_gribcodes(&gribcodes);
            }
          //  KNMI: HIRLAM model version 7.2 uses tableNum=1    (LAMH_D11*)
          //  KNMI: HARMONIE model version 36 uses tableNum=1   (grib*)
          //  (opreational NWP version) KNMI: HARMONIE model version 38 uses tableNum=253 (grib,grib_md)
          //   and tableNum=1 (grib_sfx) (research version)
          else if (tableNum == 1 || tableNum == 253)
            {
              mode = ModelMode::HIRLAM;
              hirlam_harmonie_gribcodes(&gribcodes);
            }
        }
      else
        {
          mode = ModelMode::ECHAM;
          echam_gribcodes(&gribcodes);
        }

      if (Options::cdoVerbose)
        {
          vlistInqVarName(vlistID1, varID, varname);
          cdo_print("Mode = %d  Center = %d TableNum =%d Code = %d Param = %s Varname = %s varID = %d", (int) mode, instNum,
                   tableNum, code, paramstr, varname, varID);
        }

      if (code <= 0)
        {
          vlistInqVarName(vlistID1, varID, varname);

          cstr_to_lower_case(varname);

          //                       ECHAM                         ECMWF
          // clang-format off
          if      (cdo_cmpstr(varname, "geosp") || cdo_cmpstr(varname, "z"))  code = 129;
          else if (cdo_cmpstr(varname, "st") || cdo_cmpstr(varname, "t"))     code = 130;
          else if (cdo_cmpstr(varname, "aps") || cdo_cmpstr(varname, "sp"))   code = 134;
          else if (cdo_cmpstr(varname, "ps"))                                 code = 134;
          else if (cdo_cmpstr(varname, "lsp") || cdo_cmpstr(varname, "lnsp")) code = 152;
          // else if (cdo_cmpstr(varname, "geopoth"))                            code = 156;
          // clang-format on
       }

      if (mode == ModelMode::ECHAM)
        {
          if (code == gribcodes.ps && nlevels == 1)
            psID = varID;
          else if (code == gribcodes.lsp && nlevels == 1)
            lnpsID = varID;
        }
      else if (mode == ModelMode::WMO || mode == ModelMode::HIRLAM)
        {
          if (code == gribcodes.ps && nlevels == 1) psID = varID;
        }
    }

  auto pvarID = lnpsID;
  if (zaxisIDh != -1 && lnpsID != -1)
    {
      const auto gridID = vlistInqVarGrid(vlistID1, lnpsID);
      if (gridInqType(gridID) == GRID_SPECTRAL)
        {
          lnpsID = -1;
          cdo_warning("Spectral LOG(%s) not supported - using %s!", var_stdname(surface_air_pressure),
                     var_stdname(surface_air_pressure));
        }
    }

  if (zaxisIDh != -1 && lnpsID == -1)
    {
      pvarID = psID;
      if (psID == -1) cdo_abort("%s not found!", var_stdname(surface_air_pressure));
    }

  const auto gridID = vlistInqVarGrid(vlistID1, pvarID);
  if (gridInqType(gridID) == GRID_SPECTRAL)
    cdo_abort("%s on spectral representation not supported!", var_stdname(surface_air_pressure));

  Varray<double> array(gridsize);

  const auto vlistID2 = vlistCreate();
  const auto ovarID = vlistDefVar(vlistID2, gridID, zaxisIDp, TIME_VARYING);
  vlistDefVarParam(vlistID2, ovarID, cdiEncodeParam(1, 255, 255));
  cdiDefKeyString(vlistID2, ovarID, CDI_KEY_NAME, "pressure");
  cdiDefKeyString(vlistID2, ovarID, CDI_KEY_STDNAME, "air_pressure");
  cdiDefKeyString(vlistID2, ovarID, CDI_KEY_UNITS, "Pa");

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdo_open_write(1);
  cdo_def_vlist(streamID2, vlistID2);

  int tsID = 0;
  while (true)
    {
      const auto nrecs = cdo_stream_inq_timestep(streamID1, tsID);
      if (nrecs == 0) break;

      taxisCopyTimestep(taxisID2, taxisID1);
      cdo_def_timestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          int varID, levelID;
          cdo_inq_record(streamID1, &varID, &levelID);

          if (varID == pvarID)
            {
              size_t nmiss;
              cdo_read_record(streamID1, array.data(), &nmiss);
              if (nmiss) cdo_abort("Missing valus unsupported!");
            }
        }

      if (zaxisIDh != -1)
        {
          if (lnpsID != -1)
            for (size_t i = 0; i < gridsize; i++) psProg[i] = std::exp(array[i]);
          else if (psID != -1)
            varray_copy(gridsize, array, psProg);

          // check range of psProg
          const auto mm = varray_min_max(psProg);
          if (mm.min < MIN_PS || mm.max > MAX_PS)
            cdo_warning("Surface pressure out of range (min=%g max=%g)!", mm.min, mm.max);

          vct_to_hybrid_pressure(fullPress.data(), halfPress.data(), vct.data(), psProg.data(), nhlevf, gridsize);
        }

      if (operatorID == PRESSURE_FL)
        {
          nlevel = nhlevf;
          pout = fullPress.data();
        }
      else if (operatorID == DELTAP)
        {
          nlevel = nhlevf;
          for (k = 0; k < nhlevf; ++k)
            for (size_t i = 0; i < gridsize; ++i)
              deltap[k * gridsize + i] = halfPress[(k + 1) * gridsize + i] - halfPress[k * gridsize + i];

          pout = deltap.data();
        }
      else if (operatorID == PRESSURE_HL)
        {
          nlevel = nhlevh;
          pout = halfPress.data();
        }

      int varID = 0;
      for (int levelID = 0; levelID < nlevel; levelID++)
        {
          cdo_def_record(streamID2, varID, levelID);
          const auto offset = levelID * gridsize;
          cdo_write_record(streamID2, pout + offset, 0);
        }

      tsID++;
    }

  cdo_stream_close(streamID2);
  cdo_stream_close(streamID1);

  cdo_finish();

  return nullptr;
}
