/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Copy       copy            Copy datasets
*/

#include <cdi.h>

#include "functs.h"
#include "process_int.h"
#include "cdo_vlist.h"
#include "progress.h"
#include "cdo_options.h"

void *
Copy(void *process)
{
  bool lconstvars = true;
  CdoStreamID streamID2 = CDO_STREAM_UNDEF;
  int vlistID2 = CDI_UNDEFID;
  int taxisID2 = CDI_UNDEFID;
  int varID, levelID;
  Field field;

  cdo_initialize(process);

  const auto lcopy = unchanged_record();

  // clang-format off
                       cdo_operator_add("copy",   0, 0, nullptr);
  const auto SELALL  = cdo_operator_add("selall", 0, 0, nullptr);
  const auto SZIP    = cdo_operator_add("szip",   0, 0, nullptr);
  // clang-format on

  const auto operatorID = cdo_operator_id();
  if (operatorID == SZIP)
    {
      Options::cdoCompType = CDI_COMPRESS_SZIP;
      Options::cdoCompLevel = 0;
    }

  operator_check_argc(0);

  const auto streamCnt = cdo_stream_cnt();
  const auto nfiles = streamCnt - 1;

  progress::init();

  int tsID2 = 0;
  for (int indf = 0; indf < nfiles; ++indf)
    {
      if (Options::cdoVerbose) cdo_print("Process file: %s", cdo_get_stream_name(indf));

      const auto streamID1 = cdo_open_read(indf);
      const auto vlistID1 = cdo_stream_inq_vlist(streamID1);
      const auto taxisID1 = vlistInqTaxis(vlistID1);

      VarList varList1;
      varListInit(varList1, vlistID1);

      if (indf == 0)
        {
          vlistID2 = vlistDuplicate(vlistID1);
          taxisID2 = taxisDuplicate(taxisID1);
          vlistDefTaxis(vlistID2, taxisID2);

          const auto nvars = vlistNvars(vlistID1);

          auto ntsteps = vlistNtsteps(vlistID1);
          if (ntsteps == 1)
            {
              for (varID = 0; varID < nvars; ++varID)
                if (varList1[varID].timetype != TIME_CONSTANT) break;
              if (varID == nvars) ntsteps = 0;
            }

          if (ntsteps == 0 && nfiles > 1)
            {
              lconstvars = false;
              for (varID = 0; varID < nvars; ++varID) vlistDefVarTimetype(vlistID2, varID, TIME_VARYING);
            }
        }
      else
        {
          vlist_compare(vlistID1, vlistID2, CMP_ALL);
        }

      if (streamID2 == CDO_STREAM_UNDEF)
        {
          streamID2 = cdo_open_write(nfiles);
          cdo_def_vlist(streamID2, vlistID2);
        }

      const auto ntsteps = vlistNtsteps(vlistID1);

      int tsID1 = 0;
      while (true)
        {
          const auto nrecs = cdo_stream_inq_timestep(streamID1, tsID1);
          if (nrecs == 0) break;

          const double fstatus = (ntsteps > 1) ? indf + (tsID1 + 1.) / ntsteps : indf + 1.;
          if (!Options::cdoVerbose) progress::update(0, 1, fstatus / nfiles);

          taxisCopyTimestep(taxisID2, taxisID1);
          cdo_def_timestep(streamID2, tsID2);

          for (int recID = 0; recID < nrecs; recID++)
            {
              cdo_inq_record(streamID1, &varID, &levelID);

              if (lconstvars && tsID2 > 0 && tsID1 == 0)
                if (varList1[varID].timetype == TIME_CONSTANT) continue;

              cdo_def_record(streamID2, varID, levelID);

              if (lcopy && (operatorID == SELALL || operatorID == SZIP))
                {
                  cdo_copy_record(streamID2, streamID1);
                }
              else
                {
                  field.init(varList1[varID]);
                  cdo_read_record(streamID1, field);
                  cdo_write_record(streamID2, field);
                }
            }

          tsID1++;
          tsID2++;
        }

      cdo_stream_close(streamID1);
    }

  cdo_stream_close(streamID2);

  if (vlistID2 != CDI_UNDEFID) vlistDestroy(vlistID2);

  cdo_finish();

  return nullptr;
}
