/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Remapeta     remapeta          Model to model level interpolation
*/

#include <cdi.h>

#include "process_int.h"
#include "cdo_vlist.h"
#include "readline.h"
#include "hetaeta.h"
#include "vertical_interp.h"
#include "stdnametable.h"
#include "util_string.h"
#include "timer.h"
#include "const.h"
#include "cdo_options.h"
#include "cdo_zaxis.h"
#include "cdi_lockedIO.h"


template <typename T>
static void
setmissval(long nvals, const Varray<int> &imiss, double missval, T *array)
{
  if (!imiss.empty())
    for (long i = 0; i < nvals; ++i)
      if (imiss[i]) array[i] = missval;
}

template <typename T>
static void
corr_hum(long gridsize, T *q, double q_min)
{
  for (long i = 0; i < gridsize; ++i)
    {
      if (q[i] < q_min) q[i] = q_min;
    }
}

static long
ncctop(double cptop, long nlev, long nlevp1, const double *vct_a, const double *vct_b)
{
  /*
    Description:
    Defines highest level *ncctop* where condensation is allowed.

    Author:

    E. Roeckner, MPI, October 2001
  */
  long nctop = 0;
  Varray<double> zph(nlevp1), zp(nlev);
  // double    cptop  =  1000.;   /* min. pressure level for cond. */

  // half level pressure values, assuming 101320. Pa surface pressure

  for (long jk = 0; jk < nlevp1; ++jk)
    {
      const auto za = vct_a[jk];
      const auto zb = vct_b[jk];
      zph[jk] = za + zb * 101320.;
    }

  // full level pressure

  for (long jk = 0; jk < nlev; ++jk) zp[jk] = (zph[jk] + zph[jk + 1]) * 0.5;

  // search for pressure level cptop (Pa)

  for (long jk = 0; jk < nlev; ++jk)
    {
      nctop = jk;
      if (zp[jk] >= cptop) break;
    }

  return nctop;
}

static void
vctFromFile(const char *filename, int *nvct, Varray<double> &vct2)
{
  char line[1024], *pline;
  int i = 0;
  constexpr int maxvct = 8192;

  auto fp = fopen(filename, "r");
  if (fp == nullptr)
    {
      perror(filename);
      exit(EXIT_FAILURE);
    }

  vct2.resize(maxvct);

  while (cdo::readline(fp, line, 1024))
    {
      if (line[0] == '#' || line[0] == '\0') continue;

      pline = line;
      auto num = (int) strtod(pline, &pline);
      if (pline == nullptr) cdoAbort("Format error in VCT file %s!", filename);
      if (num != i) cdoWarning("Inconsistent VCT file, entry %d is %d.", i, num);

      if (i + maxvct / 2 >= maxvct - 1) cdoAbort("Too many values in VCT file!");

      vct2[i] = strtod(pline, &pline);
      if (pline == nullptr) cdoAbort("Format error in VCT file %s!", filename);

      vct2[i + maxvct / 2] = strtod(pline, &pline);

      i++;
    }

  fclose(fp);

  const auto nvct2 = 2 * i;
  const auto nlevh2 = i - 1;

  for (i = 0; i < nlevh2 + 1; ++i) vct2[i + nvct2 / 2] = vct2[i + maxvct / 2];

  vct2.resize(nvct2);

  *nvct = nvct2;
}

template <typename T>
static void
vertSum(Varray<double> &sum, const Varray<T> &var3d, size_t gridsize, size_t nlevels)
{
  for (size_t i = 0; i < gridsize; ++i) sum[i] = 0;

  for (size_t k = 0; k < nlevels; ++k)
    for (size_t i = 0; i < gridsize; ++i)
      {
        sum[i] += var3d[k * gridsize + i];
      }
}

template <typename T>
static void
vertSumw(Varray<double> &sum, const Varray<T> &var3d, size_t gridsize, size_t nlevels, const Varray<double> &deltap)
{
  for (size_t i = 0; i < gridsize; ++i) sum[i] = 0;

  for (size_t k = 0; k < nlevels; ++k)
    for (size_t i = 0; i < gridsize; ++i)
      {
        sum[i] += var3d[k * gridsize + i] * deltap[k * gridsize + i];
      }
}

void
vlist_hybrid_vct(int vlistID, int &rzaxisIDh, int &rnvct, Varray<double> &vct, int &rnhlevf)
{
  int zaxisIDh = -1;
  int nhlevf = 0;
  int nvct = 0;

  auto lhavevct = false;
  const int nzaxis = vlistNzaxis(vlistID);
  for (int i = 0; i < nzaxis; i++)
    {
      const auto zaxisID = vlistZaxis(vlistID, i);
      const auto nlevels = zaxisInqSize(zaxisID);

      if (zaxisInqType(zaxisID) == ZAXIS_HYBRID && nlevels > 1)
        {
          nvct = zaxisInqVctSize(zaxisID);
          if (nlevels == (nvct / 2 - 1))
            {
              if (!lhavevct)
                {
                  lhavevct = true;
                  zaxisIDh = zaxisID;
                  nhlevf = nlevels;

                  vct.resize(nvct);
                  zaxisInqVct(zaxisID, vct.data());
                }
            }
          else
            {
              if (Options::cdoVerbose) cdoPrint("nlevels = (nvct1/2 - 1): nlevels = %d", nlevels);
              if (nlevels < (nvct / 2 - 1))
                cdoPrint("z-axis %d has only %d of %d hybrid sigma pressure levels!", i + 1, nlevels, (nvct / 2 - 1));
            }
        }
    }

  rzaxisIDh = zaxisIDh;
  rnvct = nvct;
  rnhlevf = nhlevf;
}

template <typename T>
void fieldCopyArray(size_t len, const Field &field, T *array)
{
  if (field.memType == MemType::Float)
    for (size_t i = 0; i < len; ++i) array[i] = field.vec_f[i];
  else
    for (size_t i = 0; i < len; ++i) array[i] = field.vec_d[i];
}

#define MAX_VARS3D 1024

template <typename T>
void
remapeta(MemType memType)
{
  constexpr double cconst = 1.0E-6;
  size_t nfis2gp = 0;
  int iv;
  int nvars3D = 0;
  int sgeopotID = -1, tempID = -1, sqID = -1, psID = -1, lnpsID = -1;
  char varname[CDI_MAX_NAME], stdname[CDI_MAX_NAME];
  Varray<double> fis2;
  size_t nmissout = 0;
  bool ltq = false;
  bool lfis2 = false;
  int varids[MAX_VARS3D];
  Varray<int> imiss;
  int timer_hetaeta = 0;
  double missval = 0;
  double cptop = 0.0; // min. pressure level for cond.

  if (Options::Timer) timer_hetaeta = timer_new("Remapeta_hetaeta");

  // clang-format off
  const auto REMAPETA  = cdoOperatorAdd("remapeta",   0, 0, "VCT file name");
  const auto REMAPETAS = cdoOperatorAdd("remapeta_s", 0, 0, "VCT file name");
  const auto REMAPETAZ = cdoOperatorAdd("remapeta_z", 0, 0, "VCT file name");
  // clang-format on

  const auto operatorID = cdoOperatorID();

  operatorInputArg(cdoOperatorEnter(operatorID));

  const auto envstr = getenv("REMAPETA_PTOP");
  if (envstr)
    {
      const auto fval = atof(envstr);
      if (fval > 0)
        {
          cptop = fval;
          //	  if ( Options::cdoVerbose )
          cdoPrint("Set REMAPETA_PTOP to %g", cptop);
        }
    }

  int nvct2 = 0;
  Varray<double> vct2;
  vctFromFile(cdoOperatorArgv(0).c_str(), &nvct2, vct2);
  const auto nhlevf2 = nvct2 / 2 - 1;

  const auto a2 = vct2.data();
  const auto b2 = vct2.data() + nvct2 / 2;

  if (Options::cdoVerbose)
    for (int i = 0; i < nhlevf2 + 1; ++i) cdoPrint("vct2: %5d %25.17f %25.17f", i, vct2[i], vct2[nvct2 / 2 + i]);

  const auto streamID1 = cdoOpenRead(0);

  if (operatorArgc() == 2)
    {
      lfis2 = true;

      const char *fname = cdoOperatorArgv(1).c_str();
      auto streamID = streamOpenReadLocked(fname);
      const auto vlistID1 = streamInqVlist(streamID);

      int varID, levelID;
      streamInqRecord(streamID, &varID, &levelID);
      const auto gridID = vlistInqVarGrid(vlistID1, varID);
      nfis2gp = gridInqSize(gridID);

      fis2.resize(nfis2gp);

      size_t nmiss;
      streamReadRecord(streamID, fis2.data(), &nmiss);

      if (nmiss)
        {
          missval = vlistInqVarMissval(vlistID1, varID);
          imiss.resize(nfis2gp);
          for (size_t i = 0; i < nfis2gp; ++i) imiss[i] = DBL_IS_EQUAL(fis2[i], missval);

          nmissout = nmiss;
        }

      // check range of surface_geopotential
      auto mm = arrayMinMaxMask(nfis2gp, fis2.data(), imiss);
      if (mm.min < MIN_FIS || mm.max > MAX_FIS)
        cdoWarning("%s out of range (min=%g max=%g)!", var_stdname(surface_geopotential), mm.min, mm.max);

      if (mm.min < -1.e10 || mm.max > 1.e10) cdoAbort("%s out of range!", var_stdname(surface_geopotential));

      streamClose(streamID);
    }

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  VarList varList1;
  varListInit(varList1, vlistID1);

  const auto gridID0 = vlistGrid(vlistID1, 0);
  if (gridInqType(gridID0) == GRID_SPECTRAL) cdoAbort("Spectral data unsupported!");

  const auto gridsize = vlist_check_gridsize(vlistID1);

  const auto zaxisID2 = zaxisCreate(ZAXIS_HYBRID, nhlevf2);

  {
    Varray<double> lev2(nhlevf2);
    for (int i = 0; i < nhlevf2; ++i) lev2[i] = i + 1;
    zaxisDefLevels(zaxisID2, lev2.data());
  }

  if (nvct2 == 0) cdoAbort("Internal problem, vct2 undefined!");
  zaxisDefVct(zaxisID2, nvct2, vct2.data());

  auto surfaceID = zaxisFromName("surface");

  int zaxisIDh = -1;
  int nvct1 = 0;
  int nhlevf1 = 0;
  Varray<double> vct1;
  vlist_hybrid_vct(vlistID1, zaxisIDh, nvct1, vct1, nhlevf1);

  vlist_change_hybrid_zaxis(vlistID1, vlistID2, zaxisIDh, zaxisID2);

  const auto nzaxis = vlistNzaxis(vlistID1);
  for (int i = 0; i < nzaxis; i++)
    {
      const auto zaxisID = vlistZaxis(vlistID1, i);
      const auto nlevels = zaxisInqSize(zaxisID);
      if (zaxisInqType(zaxisID) == ZAXIS_HYBRID && nlevels == 1) vlistChangeZaxisIndex(vlistID2, i, surfaceID);
    }

  const auto *a1 = vct1.data();
  const auto *b1 = vct1.data() + nvct1 / 2;
  if (Options::cdoVerbose)
    for (int i = 0; i < nvct1 / 2; ++i) cdoPrint("vct1: %5d %25.17f %25.17f", i, vct1[i], vct1[nvct1 / 2 + i]);

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  
  VarList varList2;
  varListInit(varList2, vlistID2);

  if (zaxisIDh == -1) cdoWarning("No 3D variable with hybrid sigma pressure coordinate found!");

  const auto nvars = vlistNvars(vlistID1);

  for (int varID = 0; varID < nvars; varID++)
    {
      const auto gridID = varList1[varID].gridID;
      const auto zaxisID = varList1[varID].zaxisID;
      const auto nlevels = varList1[varID].nlevels;

      auto code = varList1[varID].code;
      // code = -1;
      if (code <= 0 || code == 255)
        {
          vlistInqVarName(vlistID1, varID, varname);
          cstrToLowerCase(varname);

          int length = CDI_MAX_NAME;
          cdiInqKeyString(vlistID1, varID, CDI_KEY_STDNAME, stdname, &length);
          cstrToLowerCase(stdname);

          code = echamcode_from_stdname(stdname);
          if (code == -1)
            {
              //                                            ECHAM                         ECMWF
              // clang-format off
              if      (sgeopotID == -1 && (cdo_cmpstr(varname, "geosp") || cdo_cmpstr(varname, "z")))    code = 129;
              else if (tempID == -1    && (cdo_cmpstr(varname, "st")    || cdo_cmpstr(varname, "t")))    code = 130;
              else if (psID == -1      && (cdo_cmpstr(varname, "aps")   || cdo_cmpstr(varname, "ps")))   code = 134;
              else if (lnpsID == -1    && (cdo_cmpstr(varname, "lsp")   || cdo_cmpstr(varname, "lnsp"))) code = 152;
              else if (sqID == -1      && (cdo_cmpstr(varname, "q"))) code = 133;
              // clang-format on
            }
        }

      // clang-format off
      if      (code == 129 && nlevels == 1) sgeopotID = varID;
      else if (code == 134 && nlevels == 1) psID = varID;
      else if (code == 152 && nlevels == 1) lnpsID = varID;
      else if (code == 130 && nlevels == nhlevf1) tempID = varID;
      else if (code == 133 && nlevels == nhlevf1) sqID = varID;
      // clang-format on

      if (gridInqType(gridID) == GRID_SPECTRAL && zaxisInqType(zaxisID) == ZAXIS_HYBRID)
        cdoAbort("Spectral data on model level unsupported!");

      if (gridInqType(gridID) == GRID_SPECTRAL) cdoAbort("Spectral data unsupported!");

      if (zaxisInqType(zaxisID) == ZAXIS_HYBRID && zaxisIDh != -1 && nlevels == nhlevf1)
        {
          if (!(code == 130 || code == 133)) varids[nvars3D++] = varID;
        }
      else
        {
          if (code == 130) tempID = -1;
          if (code == 133) sqID = -1;
        }
    }

  if (Options::cdoVerbose)
    {
      cdoPrint("Found:");
      // clang-format off
      if (tempID    != -1) cdoPrint("  %s", var_stdname(air_temperature));
      if (psID      != -1) cdoPrint("  %s", var_stdname(surface_air_pressure));
      if (lnpsID    != -1) cdoPrint("  LOG(%s)", var_stdname(surface_air_pressure));
      if (sgeopotID != -1) cdoPrint("  %s", var_stdname(surface_geopotential));
      if (sqID      != -1) cdoPrint("  %s", var_stdname(specific_humidity));
      // clang-format on
    }

  if (tempID != -1 && sqID != -1)
    {
      ltq = true;
    }
  else
    {
      if (tempID != -1) cdoAbort("Temperature without humidity unsupported!");
      if (sqID != -1) cdoAbort("Humidity without temperature unsupported!");
    }
  /*
  if ( ltq == false )
    {
      cdoWarning("Temperature and Humidity not found!");
    }
  */
  if (operatorID == REMAPETA)
    {
    }

  Varray<double> sum1, sum2;
  if (operatorID == REMAPETAS || operatorID == REMAPETAZ)
    {
      sum1.resize(gridsize);
      sum2.resize(gridsize);
    }

  Varray<double> deltap1, deltap2;
  Varray<double> half_press1, half_press2;
  if (operatorID == REMAPETAZ)
    {
      deltap1.resize(gridsize * nhlevf1);
      deltap2.resize(gridsize * nhlevf2);
      half_press1.resize(gridsize * (nhlevf1 + 1));
      half_press2.resize(gridsize * (nhlevf2 + 1));
    }

  Field field;
  if (memType == MemType::Float)
    field.resizef(gridsize);
  else
    field.resize(gridsize);

  Varray<double> fis1(gridsize);
  Varray<double> ps1(gridsize);

  if (!lfis2) fis2.resize(gridsize);
  if (lfis2 && gridsize != nfis2gp) cdoAbort("Orographies have different grid size!");

  Varray<double> ps2(gridsize);

  Varray<T> t1, t2;
  Varray<T> q1, q2;
  Varray<double> tscor, pscor, secor;
  if (ltq)
    {
      tscor.resize(gridsize);
      pscor.resize(gridsize);
      secor.resize(gridsize);

      t1.resize(gridsize * nhlevf1);
      q1.resize(gridsize * nhlevf1);

      t2.resize(gridsize * nhlevf2);
      q2.resize(gridsize * nhlevf2);
    }

  Varray2D<T> vars1, vars2;
  if (nvars3D)
    {
      vars1.resize(nvars);
      vars2.resize(nvars);
      for (int varID = 0; varID < nvars3D; ++varID) vars1[varID].resize(gridsize * nhlevf1);
      for (int varID = 0; varID < nvars3D; ++varID) vars2[varID].resize(gridsize * nhlevf2);
    }

  if (zaxisIDh != -1 && sgeopotID == -1)
    {
      varrayFill(fis1, 0.0);
      if (ltq) cdoWarning("%s not found - set to zero!", var_stdname(surface_geopotential));
    }

  int presID = lnpsID;
  if (zaxisIDh != -1 && lnpsID == -1)
    {
      if (psID == -1)
        cdoAbort("%s not found!", var_stdname(surface_air_pressure));
      else
        presID = psID;
    }

  if (Options::cdoVerbose)
    {
      if (presID == lnpsID)
        cdoPrint("using LOG(%s)", var_stdname(surface_air_pressure));
      else
        cdoPrint("using %s", var_stdname(surface_air_pressure));
    }

  if (Options::cdoVerbose) cdoPrint("nvars3D = %d   ltq = %d", nvars3D, (int) ltq);

  int tsID = 0;
  while (true)
    {
      const auto nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs == 0) break;

      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          int varID, levelID;
          cdoInqRecord(streamID1, &varID, &levelID);
          cdoReadRecord(streamID1, field);

          if (zaxisIDh != -1)
            {
              const auto zaxisID = varList1[varID].zaxisID;
              const auto nlevels = varList1[varID].nlevels;
              const auto offset = gridsize * levelID;

              if (varID == sgeopotID)
                fieldCopyArray(gridsize, field, fis1.data());
              else if (varID == presID)
                {
                  if (lnpsID != -1)
                    {
                      if (field.memType == MemType::Float)
                        for (size_t i = 0; i < gridsize; ++i) ps1[i] = std::exp((double)field.vec_f[i]);
                      else
                        for (size_t i = 0; i < gridsize; ++i) ps1[i] = std::exp(field.vec_d[i]);
                    }
                  else if (psID != -1)
                    fieldCopyArray(gridsize, field, ps1.data());
                }
              else if (ltq && varID == tempID)
                fieldCopyArray(gridsize, field, &t1[offset]);
              else if (ltq && varID == sqID)
                fieldCopyArray(gridsize, field, &q1[offset]);
              // else if ( zaxisID == zaxisIDh )
              else if (zaxisInqType(zaxisID) == ZAXIS_HYBRID && nlevels == nhlevf1)
                {
                  int i;
                  for (i = 0; i < nvars3D; ++i)
                    if (varID == varids[i]) break;

                  if (i == nvars3D) cdoAbort("Internal error, 3D variable not found!");

                  fieldCopyArray(gridsize, field, &vars1[i][offset]);
                }
              else
                {
                  cdoDefRecord(streamID2, varID, levelID);
                  cdoWriteRecord(streamID2, field);
                }
            }
          else
            {
              cdoDefRecord(streamID2, varID, levelID);
              cdoWriteRecord(streamID2, field);
            }
        }

      if (zaxisIDh != -1)
        {
          // check range of ps_prog
          auto mm = arrayMinMaxMask(gridsize, ps1.data(), imiss);
          if (mm.min < MIN_PS || mm.max > MAX_PS) cdoWarning("Surface pressure out of range (min=%g max=%g)!", mm.min, mm.max);

          // check range of geop
          mm = arrayMinMaxMask(gridsize, fis1.data(), imiss);
          if (mm.min < MIN_FIS || mm.max > MAX_FIS) cdoWarning("Orography out of range (min=%g max=%g)!", mm.min, mm.max);
        }

      if (!lfis2)
        for (size_t i = 0; i < gridsize; i++) fis2[i] = fis1[i];

      if (ltq)
        {
          int varID = tempID;
          for (int levelID = 0; levelID < varList1[varID].nlevels; levelID++)
            {
              const auto offset = gridsize * levelID;
              auto mm = arrayMinMaxMask(gridsize, &t1[offset], imiss);
              if (mm.min < MIN_T || mm.max > MAX_T)
                cdoWarning("Input temperature at level %d out of range (min=%g max=%g)!", levelID + 1, mm.min, mm.max);
            }

          varID = sqID;
          for (int levelID = 0; levelID < varList1[varID].nlevels; levelID++)
            {
              const auto offset = gridsize * levelID;
              corr_hum(gridsize, &q1[offset], MIN_Q);

              auto mm = arrayMinMaxMask(gridsize, &q1[offset], imiss);
              if (mm.min < MIN_Q || mm.max > MAX_Q)
                cdoWarning("Input humidity at level %d out of range (min=%g max=%g)!", levelID + 1, mm.min, mm.max);
            }
        }

      if (nvars3D || ltq)
        {
          if (Options::Timer) timer_start(timer_hetaeta);
          hetaeta(ltq, gridsize, imiss.data(), nhlevf1, a1, b1, fis1, ps1, t1, q1, nhlevf2, a2, b2, fis2,
                  ps2, t2, q2, nvars3D, vars1, vars2, tscor, pscor, secor);
          if (Options::Timer) timer_stop(timer_hetaeta);
        }

      const long nctop = (cptop > 0) ? ncctop(cptop, (long) nhlevf2, (long) nhlevf2 + 1, a2, b2) : 0;

      if (zaxisIDh != -1 && sgeopotID != -1)
        {
          int varID = sgeopotID;
          int levelID = 0;
          setmissval(gridsize, imiss, missval, fis2.data());
          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, fis2.data(), nmissout);
        }

      if (zaxisIDh != -1 && lnpsID != -1)
        for (size_t i = 0; i < gridsize; ++i) ps2[i] = std::log(ps2[i]);

      if (zaxisIDh != -1 && presID != -1)
        {
          int varID = presID;
          int levelID = 0;
          setmissval(gridsize, imiss, missval, ps2.data());
          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, ps2.data(), nmissout);
        }

      if (ltq)
        {
          int varID = tempID;
          for (int levelID = 0; levelID < varList2[varID].nlevels; levelID++)
            {
              const auto offset = gridsize * levelID;
              const auto single2 = &t2[offset];

              auto mm = arrayMinMaxMask(gridsize, single2, imiss);
              if (mm.min < MIN_T || mm.max > MAX_T)
                cdoWarning("Output temperature at level %d out of range (min=%g max=%g)!", levelID + 1, mm.min, mm.max);

              setmissval(gridsize, imiss, missval, single2);
              cdoDefRecord(streamID2, varID, levelID);

              if (memType == MemType::Float)
                cdoWriteRecordF(streamID2, (float *) single2, nmissout);
              else
                cdoWriteRecord(streamID2, (double *) single2, nmissout);
            }

          varID = sqID;
          for (int levelID = 0; levelID < varList2[varID].nlevels; levelID++)
            {
              const auto offset = gridsize * levelID;
              const auto single2 = &q2[offset];

              corr_hum(gridsize, single2, MIN_Q);

              if (levelID < nctop)
                for (size_t i = 0; i < gridsize; ++i) single2[i] = cconst;

              auto mm = arrayMinMaxMask(gridsize, single2, imiss);
              if (mm.min < MIN_Q || mm.max > MAX_Q)
                cdoWarning("Output humidity at level %d out of range (min=%g max=%g)!", levelID + 1, mm.min, mm.max);

              setmissval(gridsize, imiss, missval, single2);
              cdoDefRecord(streamID2, varID, levelID);

              if (memType == MemType::Float)
                cdoWriteRecordF(streamID2, (float *) single2, nmissout);
              else
                cdoWriteRecord(streamID2, (double *) single2, nmissout);
            }
        }

      for (iv = 0; iv < nvars3D; ++iv)
        {
          int varID = varids[iv];

          const auto nlevels = varList2[varID].nlevels;

          if (operatorID == REMAPETAS)
            {
              vertSum(sum1, vars1[iv], gridsize, nhlevf1);
              vertSum(sum2, vars2[iv], gridsize, nhlevf2);
            }
          else if (operatorID == REMAPETAZ)
            {
              presh((double*)nullptr, half_press1.data(), vct1.data(), ps1.data(), nhlevf1, gridsize);
              for (int k = 0; k < nhlevf1; ++k)
                for (size_t i = 0; i < gridsize; ++i)
                  {
                    deltap1[k * gridsize + i] = half_press1[(k + 1) * gridsize + i] - half_press1[k * gridsize + i];
                    deltap1[k * gridsize + i] = std::log(deltap1[k * gridsize + i]);
                  }
              vertSumw(sum1, vars1[iv], gridsize, nhlevf1, deltap1);

              presh((double*)nullptr, half_press2.data(), vct2.data(), ps1.data(), nhlevf2, gridsize);
              for (int k = 0; k < nhlevf2; ++k)
                for (size_t i = 0; i < gridsize; ++i)
                  {
                    deltap2[k * gridsize + i] = half_press2[(k + 1) * gridsize + i] - half_press2[k * gridsize + i];
                    deltap2[k * gridsize + i] = std::log(deltap2[k * gridsize + i]);
                  }
              vertSumw(sum2, vars2[iv], gridsize, nhlevf2, deltap2);
            }

          for (int levelID = 0; levelID < nlevels; levelID++)
            {
              const auto offset = gridsize * levelID;
              const auto single2 = &vars2[iv][offset];

              if (operatorID == REMAPETAS || operatorID == REMAPETAZ)
                for (size_t i = 0; i < gridsize; ++i) single2[i] = single2[i] * sum1[i] / sum2[i];

              setmissval(gridsize, imiss, missval, single2);
              cdoDefRecord(streamID2, varID, levelID);

              if (memType == MemType::Float)
                cdoWriteRecordF(streamID2, (float *) single2, nmissout);
              else
                cdoWriteRecord(streamID2, (double *) single2, nmissout);
            }
        }

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);
}

void *
Remapeta(void *process)
{
  cdoInitialize(process);

  const auto memType = Options::CDO_Memtype;
  if (memType == MemType::Float)
    remapeta<float>(memType);
  else
    remapeta<double>(memType);

  cdoFinish();

  return nullptr;
}
