/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Merstat    merrange        Meridional range
      Merstat    mermin          Meridional minimum
      Merstat    mermax          Meridional maximum
      Merstat    mersum          Meridional sum
      Merstat    mermean         Meridional mean
      Merstat    meravg          Meridional average
      Merstat    merstd          Meridional standard deviation
      Merstat    merstd1         Meridional standard deviation [Normalize by (n-1)]
      Merstat    mervar          Meridional variance
      Merstat    mervar1         Meridional variance [Normalize by (n-1)]
      Merstat    merpctl         Meridional percentiles
*/

#include <cdi.h>

#include "functs.h"
#include "process_int.h"
#include "param_conversion.h"
#include <mpim_grid.h>

static void
addOperators(void)
{
  // clang-format off
  cdoOperatorAdd("merrange", func_range, 0, nullptr);
  cdoOperatorAdd("mermin",   func_min,   0, nullptr);
  cdoOperatorAdd("mermax",   func_max,   0, nullptr);
  cdoOperatorAdd("mersum",   func_sum,   0, nullptr);
  cdoOperatorAdd("mermean",  func_meanw, 1, nullptr);
  cdoOperatorAdd("meravg",   func_avgw,  1, nullptr);
  cdoOperatorAdd("mervar",   func_varw,  1, nullptr);
  cdoOperatorAdd("mervar1",  func_var1w, 1, nullptr);
  cdoOperatorAdd("merstd",   func_stdw,  1, nullptr);
  cdoOperatorAdd("merstd1",  func_std1w, 1, nullptr);
  cdoOperatorAdd("merskew",  func_skew,  0, nullptr);
  cdoOperatorAdd("merkurt",  func_kurt,  0, nullptr);
  cdoOperatorAdd("merpctl",  func_pctl,  0, nullptr);
  // clang-format on
}

void *
Merstat(void *process)
{
  int gridID1, gridID2 = -1, lastgrid = -1;
  int index;

  cdoInitialize(process);

  addOperators();

  const auto operatorID = cdoOperatorID();
  const auto operfunc = cdoOperatorF1(operatorID);
  const bool needWeights = cdoOperatorF2(operatorID) != 0;

  double pn = 0;
  if (operfunc == func_pctl)
    {
      operatorInputArg("percentile number");
      pn = parameter2double(cdoOperatorArgv(0));
    }
  else
    {
      operatorCheckArgc(0);
    }

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto ngrids = vlistNgrids(vlistID1);
  int ndiffgrids = 0;
  for (index = 1; index < ngrids; index++)
    if (vlistGrid(vlistID1, 0) != vlistGrid(vlistID1, index)) ndiffgrids++;

  if (ndiffgrids > 0) cdoAbort("Too many different grids!");

  index = 0;
  gridID1 = vlistGrid(vlistID1, index);

  if (gridInqType(gridID1) == GRID_LONLAT || gridInqType(gridID1) == GRID_GAUSSIAN || gridInqType(gridID1) == GRID_GENERIC)
    {
      gridID2 = gridToMeridional(gridID1);
    }
  else
    {
      cdoAbort("Unsupported gridtype: %s", gridNamePtr(gridInqType(gridID1)));
    }

  vlistChangeGridIndex(vlistID2, index, gridID2);

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  gridID1 = vlistInqVarGrid(vlistID1, 0);
  const int nlonmax = gridInqXsize(gridID1); // max nlon?
  const auto gridsizemax = vlistGridsizeMax(vlistID1);

  Field field1, field2;
  if (needWeights) field1.weightv.resize(gridsizemax);

  field2.resize(nlonmax);
  field2.grid = gridID2;
  field2.memType = MemType::Double;

  VarList varList1;
  varListInit(varList1, vlistID1);

  int tsID = 0;
  while (true)
    {
      const auto nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs == 0) break;

      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          int varID, levelID;
          cdoInqRecord(streamID1, &varID, &levelID);
          field1.init(varList1[varID]);
          cdoReadRecord(streamID1, field1);

          field2.missval = field1.missval;

          bool wstatus = false;
          if (needWeights && field1.grid != lastgrid)
            {
              lastgrid = field1.grid;
              wstatus = gridWeights(field1.grid, field1.weightv.data());
            }

          if (wstatus != 0 && tsID == 0 && levelID == 0)
            cdoWarning("Grid cell bounds not available, using constant grid cell area weights for variable %s!",
                       varList1[varID].name);

          (operfunc == func_pctl) ? fieldMerPctl(field1, field2, pn) : fieldMerFunction(field1, field2, operfunc);

          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, field2);
        }

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
