/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

     Sorttimestamp    sorttimestamp         Sort all timesteps
*/

#include <algorithm> // sort

#include <cdi.h>

#include "cdo_options.h"
#include "functs.h"
#include "process_int.h"
#include "cdo_vlist.h"
#include "printinfo.h"
#include "timebase.h"


bool getenvSkipSameTime();

struct TimeInfo
{
  int index;
  double datetime;
};

static bool
cmpdatetime(const TimeInfo &a, const TimeInfo &b)
{
  return a.datetime < b.datetime;
}

void *
Sorttimestamp(void *process)
{
  int nrecs;
  int varID, levelID;
  int lasttsID = -1;
  int nalloc = 0;
  int vlistID2 = -1, taxisID2 = -1;
  int nvars = 0;

  cdoInitialize(process);

  operatorCheckArgc(0);

  const auto skipSameTime = getenvSkipSameTime();

  FieldVector3D vars;
  std::vector<int64_t> vdate;
  std::vector<int> vtime;

  const auto nfiles = cdoStreamCnt() - 1;

  int xtsID = 0;
  for (int fileID = 0; fileID < nfiles; fileID++)
    {
      const auto streamID1 = cdoOpenRead(fileID);
      const auto vlistID1 = cdoStreamInqVlist(streamID1);
      const auto taxisID1 = vlistInqTaxis(vlistID1);

      VarList varList1;
      varListInit(varList1, vlistID1);

      if (fileID == 0)
        {
          vlistID2 = vlistDuplicate(vlistID1);
          taxisID2 = taxisDuplicate(taxisID1);
          if (taxisHasBounds(taxisID2))
            {
              cdoWarning("Time bounds unsupported by this operator, removed!");
              taxisDeleteBounds(taxisID2);
            }
        }
      else
        {
          vlistCompare(vlistID2, vlistID1, CMP_ALL);
        }

      nvars = vlistNvars(vlistID1);

      int tsID = 0;
      while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
        {
          if (xtsID >= nalloc)
            {
              constexpr int NALLOC_INC = 1024;
              nalloc += NALLOC_INC;
              vdate.resize(nalloc);
              vtime.resize(nalloc);
              vars.resize(nalloc);
            }

          vdate[xtsID] = taxisInqVdate(taxisID1);
          vtime[xtsID] = taxisInqVtime(taxisID1);

          fieldsFromVlist(vlistID1, vars[xtsID]);

          for (int recID = 0; recID < nrecs; recID++)
            {
              cdoInqRecord(streamID1, &varID, &levelID);
              auto &field = vars[xtsID][varID][levelID];
              field.init(varList1[varID]);
              cdoReadRecord(streamID1, field);
            }

          tsID++;
          xtsID++;
        }

      cdoStreamClose(streamID1);
    }

  int nts = xtsID;

  std::vector<TimeInfo> timeinfo(nts);
  const auto calendar = taxisInqCalendar(taxisID2);

  for (int tsID = 0; tsID < nts; tsID++)
    {
      const auto julday = date_to_julday(calendar, vdate[tsID]);
      const auto secofday = time_to_sec(vtime[tsID]);
      const double jdatetime = julday + secofday / 86400.;
      timeinfo[tsID].index = tsID;
      timeinfo[tsID].datetime = jdatetime;
    }

  std::sort(timeinfo.begin(), timeinfo.end(), cmpdatetime);

  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdoOpenWrite(nfiles);
  cdoDefVlist(streamID2, vlistID2);

  int tsID2 = 0;
  for (int tsID = 0; tsID < nts; tsID++)
    {
      xtsID = timeinfo[tsID].index;

      if (tsID > 0 && skipSameTime && IS_EQUAL(timeinfo[tsID].datetime, timeinfo[lasttsID].datetime))
        {
          if (Options::cdoVerbose)
            {
              cdoPrint("Timestep %4d %s %s already exists, skipped!", xtsID + 1, dateToString(vdate[xtsID]).c_str(),
                       timeToString(vtime[xtsID]).c_str());
            }
          continue;
        }

      lasttsID = tsID;

      taxisDefVdate(taxisID2, vdate[xtsID]);
      taxisDefVtime(taxisID2, vtime[xtsID]);
      cdoDefTimestep(streamID2, tsID2++);

      for (varID = 0; varID < nvars; varID++)
        {
          const auto nlevels = zaxisInqSize(vlistInqVarZaxis(vlistID2, varID));
          for (levelID = 0; levelID < nlevels; levelID++)
            {
              auto &field = vars[xtsID][varID][levelID];
              if (field.hasData())
                {
                  cdoDefRecord(streamID2, varID, levelID);
                  cdoWriteRecord(streamID2, field);
                }
            }
        }
    }

  cdoStreamClose(streamID2);

  cdoFinish();

  return nullptr;
}
