#include "param_conversion.h"

#include <inttypes.h>
#include <cctype>
#include <vector>
#include <cdi.h>

#include "cdo_output.h"
#include "cdo_options.h"
#include "compare.h"
#include "util_string.h"
#include "constants.h"
#include "const.h"
#include "assert.h"


const char *
parameter2word(const char *cstring)
{
  const size_t len = strlen(cstring);

  for (size_t i = 0; i < len; ++i)
    {
	  const int c = cstring[i];
      if (iscntrl(c) || isblank(c)) cdoAbort("Word parameter >%s< contains invalid character at position %zu!", cstring, i + 1);
    }

  if (len == 0) cdoAbort("Word parameter >%s< is empty!", cstring);

  return cstring;
}

static inline void
parameterError(const char *name, const char *cstring, const char *endptr)
{
  cdoAbort("%s parameter >%s< contains invalid character at position %d!", name, cstring, (int) (endptr - cstring + 1));
}

double
parameter2double(const char *const cstring)
{
  char *endptr = nullptr;
  const double fval = strtod(cstring, &endptr);
  if (*endptr) parameterError("Float", cstring, endptr);
  return fval;
}

int
parameter2int(const char *const cstring)
{
  char *endptr = nullptr;
  const int ival = (int) strtol(cstring, &endptr, 10);
  if (*endptr) parameterError("Integer", cstring, endptr);
  return ival;
}

long
parameter2long(const char *const cstring)
{
  char *endptr = nullptr;
  const long ival = strtol(cstring, &endptr, 10);
  if (*endptr) parameterError("Integer", cstring, endptr);
  return ival;
}

size_t
parameter2sizet(const char *const cstring)
{
  char *endptr = nullptr;
  const size_t ival = (size_t) strtoimax(cstring, &endptr, 10);
  if (*endptr) parameterError("Integer", cstring, endptr);
  return ival;
}

int
parameter2intlist(const char *const cstring)
{
  char *endptr = nullptr;
  const int ival = (int) strtol(cstring, &endptr, 10);
  if (*endptr && *endptr != '/' && (endptr - cstring) == 0) parameterError("Integer", cstring, endptr);
  return ival;
}

const std::string &
parameter2word(const std::string &string)
{
  const size_t len = string.size();

  for (size_t i = 0; i < len; ++i)
    {
	  const int c = string[i];
      if (iscntrl(c) || isblank(c))
        cdoAbort("Word parameter >%s< contains invalid character at position %zu!", string.c_str(), i + 1);
    }

  if (len == 0) cdoAbort("Word parameter >%s< is empty!", string.c_str());

  return string;
}

bool
parameter2bool(const std::string &string)
{
  const auto lstring = stringToLower(string);

  if (lstring == "1" || lstring == "t" || lstring == "true") return true;
  if (lstring == "0" || lstring == "f" || lstring == "false") return false;

  cdoAbort("Boolean parameter >%s< contains invalid characters!", string.c_str());

  return false;
}

double
parameter2double(const std::string &string)
{
  return parameter2double(string.c_str());
}

int
parameter2int(const std::string &string)
{
  return parameter2int(string.c_str());
}

long
parameter2long(const std::string &string)
{
  return parameter2long(string.c_str());
}

size_t
parameter2sizet(const std::string &string)
{
  return parameter2sizet(string.c_str());
}

int
parameter2intlist(const std::string &string)
{
  return parameter2intlist(string.c_str());
}

double
radius_str_to_deg(const char *string)
{
  char *endptr = nullptr;
  auto radius = strtod(string, &endptr);

  if (*endptr != 0)
    {
      if (strncmp(endptr, "km", 2) == 0)
        radius = 360 * ((radius * 1000) / (2 * PlanetRadius * M_PI));
      else if (strncmp(endptr, "m", 1) == 0)
        radius = 360 * ((radius) / (2 * PlanetRadius * M_PI));
      else if (strncmp(endptr, "deg", 3) == 0)
        ;
      else if (strncmp(endptr, "rad", 3) == 0)
        radius *= RAD2DEG;
      else
        cdoAbort("Float parameter >%s< contains invalid character at position %d!", string, (int) (endptr - string + 1));
    }

  if (radius > 180.) radius = 180.;

  return radius;
}

int
stringToParam(const char *const paramstr)
{
  int pnum = -1, pcat = 255, pdis = 255;
  sscanf(paramstr, "%d.%d.%d", &pnum, &pcat, &pdis);

  if (Options::cdoVerbose) cdoPrint("pnum, pcat, pdis: %d.%d.%d", pnum, pcat, pdis);

  return cdiEncodeParam(pnum, pcat, pdis);
}

int
stringToParam(const std::string &paramstr)
{
  return stringToParam(paramstr.c_str());
}

/* time/date/season converisons */
/* =================================================================================== */
void
season_to_months(const char *season, int *imonths)
{
  const char *const smons = "JFMAMJJASONDJFMAMJJASOND";
  const int imons[] = { 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12 };
  assert(strlen(smons) == (sizeof(imons) / sizeof(int)));

  const auto len = strlen(season);
  if (len == 3 && strcmp(season, "ANN") == 0)
    {
      for (size_t k = 0; k < 12; ++k) imonths[k + 1] = 1;
    }
  else
    {
      if (len > 12) cdoAbort("Too many months %d (limit=12)!", (int) len);
      char *const season_u = strdup(season);
      cstrToUpperCase(season_u);
      const char *const sstr = strstr(smons, season_u);
      free(season_u);
      if (sstr != nullptr)
        {
          const size_t ks = (size_t)(sstr - smons);
          const size_t ke = ks + len;
          for (size_t k = ks; k < ke; ++k) imonths[imons[k]]++;
        }
      else
        {
          cdoAbort("Season %s not available!", season);
        }
    }
}

double
datestrToDouble(const char *datestr, const int opt)
{
  int year = 1, month = 1, day = 1, hour = 0, minute = 0, second = 0;
  double fval = 0;

  const auto len = strlen(datestr);

  for (size_t i = 0; i < len; ++i)
    {
	  const int c = datestr[i];
      if (!(isdigit(c) || c == '-' || c == ':' || c == '.' || c == 'T'))
        cdoAbort("Date string >%s< contains invalid character at position %zu!", datestr, i + 1);
    }

  if (opt)
    {
      hour = 23;
      minute = 59;
      second = 59;
    }

  if (strchr(datestr, '-') == nullptr)
    {
      fval = parameter2double(datestr);
    }
  else if (strchr(datestr, 'T'))
    {
	  const int status = sscanf(datestr, "%d-%d-%dT%d:%d:%d", &year, &month, &day, &hour, &minute, &second);
      if (status != 6) cdoAbort("Invalid date string >%s<!", datestr);
      fval = cdiEncodeTime(hour, minute, second);
      if (std::fabs(fval) > 0) fval /= 1000000;
      fval += cdiEncodeDate(year, month, day);
    }
  else
    {
	  const int status = sscanf(datestr, "%d-%d-%d", &year, &month, &day);
      if (status != 3) cdoAbort("Invalid date string >%s<!", datestr);
      fval = cdiEncodeTime(hour, minute, second);
      if (std::fabs(fval) > 0) fval /= 1000000;
      fval += cdiEncodeDate(year, month, day);
    }

  return fval;
}

/* argv conversions ============================================== */
std::vector<int>
cdoArgvToInt(const std::vector<std::string> &argv)
{
  std::vector<int> intArgv;

  for (const auto &argument : argv)
    {
      int first, last, inc;
      split_intstring(argument, first, last, inc);

      if (inc >= 0)
        {
          for (int ival = first; ival <= last; ival += inc) intArgv.push_back(ival);
        }
      else
        {
          for (int ival = first; ival >= last; ival += inc) intArgv.push_back(ival);
        }
    }

  return intArgv;
}

std::vector<double>
cdoArgvToFlt(const std::vector<std::string> &argv)
{
  std::vector<double> fltArgv;
  for (const auto &argument : argv)
    {
	  const int len = (int) argument.size();
      int i;
      for (i = 0; i < len; i++)
        if (argument[i] != '/' && argument[i] != '-' && !isdigit(argument[i])) break;

      if (i != len)
        {
          fltArgv.push_back(parameter2double(argument));
        }
      else
        {
          int first, last, inc;
          split_intstring(argument, first, last, inc);

          if (inc >= 0)
            {
              for (int ival = first; ival <= last; ival += inc)  fltArgv.push_back((double) ival);
            }
          else
            {
              for (int ival = first; ival >= last; ival += inc)  fltArgv.push_back((double) ival);
            }
        }
    }

  return fltArgv;
}

void
split_intstring(const char *const intstr, int &first, int &last, int &inc)
{
  auto startptr = intstr;
  char *endptr = nullptr;
  int ival = (int) strtol(startptr, &endptr, 10);
  if (*endptr != 0 && *endptr != '/' && (endptr - startptr) == 0) parameterError("Integer", startptr, endptr);
  first = ival;
  last = ival;
  inc = 1;

  if (*endptr == '/')
    {
      startptr = endptr + 1;
      endptr = nullptr;
      ival = (int) strtol(startptr, &endptr, 10);
      if (*endptr != 0 && *endptr != '/' && (endptr - startptr) == 0) parameterError("Integer", startptr, endptr);
      last = ival;
      if (first > last) inc = -1;

      if (*endptr == '/')
        {
          startptr = endptr + 1;
          endptr = nullptr;
          ival = (int) strtol(startptr, &endptr, 10);
          if (*endptr != 0 && (endptr - startptr) == 0) parameterError("Integer", startptr, endptr);
          inc = ival;
        }
    }
}

void
split_intstring(const std::string &intstr, int &first, int &last, int &inc)
{
  split_intstring(intstr.c_str(), first, last, inc);
}
