/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Tstepcount  tstepcount  Count number of timesteps
*/

#include <cdi.h>

#include "process_int.h"
#include "param_conversion.h"
#include "cdo_options.h"
#include "cimdOmp.h"

#define NALLOC_INC 1024

static double
tstepcount(long nts, double missval1, const Varray<double> &array1, double refval)
{
  long j;
  long n = 0;

  if (DBL_IS_EQUAL(refval, missval1)) return missval1;

  for (j = 0; j < nts; j++)
    {
      n++;
      if (DBL_IS_EQUAL(array1[j], refval)) break;
    }

  return (j == nts) ? missval1 : (double) n;
}

void *
Tstepcount(void *process)
{
  int nrecs;
  int gridID, varID, levelID;
  size_t nmiss;
  int64_t vdate = 0;
  int vtime = 0;
  struct memory_t
  {
    Varray<double> array1;
  };

  cdoInitialize(process);

  double refval = (operatorArgc() == 1) ? parameter2double(cdoOperatorArgv(0)) : 0;

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  vlistDefNtsteps(vlistID2, 1);

  const auto nvars = vlistNvars(vlistID1);
  for (varID = 0; varID < nvars; varID++) vlistDefVarUnits(vlistID2, varID, "steps");

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  FieldVector3D vars;

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      if ((size_t) tsID >= vars.size()) vars.resize(vars.size() + NALLOC_INC);

      vdate = taxisInqVdate(taxisID1);
      vtime = taxisInqVtime(taxisID1);

      fieldsFromVlist(vlistID1, vars[tsID]);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          gridID = vlistInqVarGrid(vlistID1, varID);
          const auto gridsize = gridInqSize(gridID);
          vars[tsID][varID][levelID].resize(gridsize);
          cdoReadRecord(streamID1, vars[tsID][varID][levelID].vec_d.data(), &nmiss);
          vars[tsID][varID][levelID].nmiss = nmiss;
        }

      tsID++;
    }

  int nts = tsID;

  std::vector<memory_t> mem(Threading::ompNumThreads);
  for (int i = 0; i < Threading::ompNumThreads; i++) mem[i].array1.resize(nts);

  for (varID = 0; varID < nvars; varID++)
    {
      gridID = vlistInqVarGrid(vlistID1, varID);
      auto missval = vlistInqVarMissval(vlistID1, varID);
      auto gridsize = gridInqSize(gridID);
      const auto nlevels = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));
      for (levelID = 0; levelID < nlevels; levelID++)
        {
#ifdef _OPENMP
#pragma omp parallel for default(none) shared(gridsize, nts, refval, mem, vars, varID, levelID, missval) schedule(dynamic, 1)
#endif
          for (size_t i = 0; i < gridsize; i++)
            {
              const auto ompthID = cdo_omp_get_thread_num();

              for (int t = 0; t < nts; t++) mem[ompthID].array1[t] = vars[t][varID][levelID].vec_d[i];

              const auto count = tstepcount(nts, missval, mem[ompthID].array1, refval);
              vars[0][varID][levelID].vec_d[i] = count;
            }
        }
    }

  taxisDefVdate(taxisID2, vdate);
  taxisDefVtime(taxisID2, vtime);
  cdoDefTimestep(streamID2, 0);

  for (varID = 0; varID < nvars; varID++)
    {
      const auto nlevels = zaxisInqSize(vlistInqVarZaxis(vlistID2, varID));
      for (levelID = 0; levelID < nlevels; levelID++)
        {
          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, vars[0][varID][levelID].vec_d.data(), fieldNumMiss(vars[0][varID][levelID]));
        }
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
