/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public Liccense as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Splitsel   splitsel        Split time selection
*/

#include <cdi.h>

#include "cdo_options.h"
#include "process_int.h"
#include "param_conversion.h"
#include "util_files.h"

void *
Splitsel(void *process)
{
  int nrecs = 0;
  int varID, levelID;
  int tsID;
  size_t nmiss;
  int i2 = 0;
  char filesuffix[32];
  char filename[8192];

  cdoInitialize(process);

  const auto lcopy = unchangedRecord();

  cdoOperatorAdd("splitsel", 0, 0, nullptr);

  // operatorInputArg("nsets <noffset <nskip>>");

  const auto nargc = operatorArgc();
  if (nargc < 1) cdoAbort("Too few arguments! Need %d found %d.", 1, nargc);

  const auto ndates = parameter2double(cdoOperatorArgv(0));
  const auto noffset = (nargc > 1) ? parameter2double(cdoOperatorArgv(1)) : 0.0;
  const auto nskip = (nargc > 2) ? parameter2double(cdoOperatorArgv(2)) : 0.0;

  if (Options::cdoVerbose) cdoPrint("nsets = %f, noffset = %f, nskip = %f", ndates, noffset, nskip);

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  // int taxisID2 = cdoTaxisCreate(TAXIS_ABSOLUTE);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  strcpy(filename, cdoGetObase());
  const auto nchars = strlen(filename);

  const char *refname = cdoGetObase();
  filesuffix[0] = 0;
  cdoGenFileSuffix(filesuffix, sizeof(filesuffix), cdoInqFiletype(streamID1), vlistID1, refname);

  Varray<double> array;
  //  if ( ! lcopy )
  {
    auto gridsizemax = vlistGridsizeMax(vlistID1);
    if (vlistNumber(vlistID1) != CDI_REAL) gridsizemax *= 2;
    array.resize(gridsizemax);
  }

  const auto nvars = vlistNvars(vlistID1);
  int nconst = 0;
  for (varID = 0; varID < nvars; varID++)
    if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT) nconst++;

  FieldVector2D vars;
  if (nconst)
    {
      vars.resize(nvars);

      for (varID = 0; varID < nvars; varID++)
        {
          if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT)
            {
              const auto gridID = vlistInqVarGrid(vlistID1, varID);
              const auto nlevel = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));
              const auto gridsize = gridInqSize(gridID);

              vars[varID].resize(nlevel);

              for (levelID = 0; levelID < nlevel; levelID++)
                {
                  vars[varID][levelID].grid = gridID;
                  vars[varID][levelID].resize(gridsize);
                }
            }
        }
    }

  int index = 0;
  int nsets = 0;

  // offset
  for (tsID = 0; tsID < noffset; tsID++)
    {
      nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs == 0)
        {
          cdoWarning("noffset is larger than number of timesteps!");
          goto LABEL_END;
        }

      if (tsID == 0 && nconst)
        for (int recID = 0; recID < nrecs; recID++)
          {
            cdoInqRecord(streamID1, &varID, &levelID);
            if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT)
              {
                cdoReadRecord(streamID1, vars[varID][levelID].vec_d.data(), &nmiss);
                vars[varID][levelID].nmiss = nmiss;
              }
          }
    }

  while (true)
    {
      sprintf(filename + nchars, "%06d", index);
      sprintf(filename + nchars + 6, "%s", filesuffix);

      if (Options::cdoVerbose) cdoPrint("create file %s", filename);

      CdoStreamID streamID2 = CDO_STREAM_UNDEF;
      int tsID2 = 0;

      for (; nsets < (int) (ndates * (index + 1)); nsets++)
        {
          nrecs = cdoStreamInqTimestep(streamID1, tsID);
          if (nrecs == 0) break;

          taxisCopyTimestep(taxisID2, taxisID1);

          if (streamID2 == CDO_STREAM_UNDEF)
            {
              streamID2 = cdoOpenWrite(filename);
              cdoDefVlist(streamID2, vlistID2);
            }

          cdoDefTimestep(streamID2, tsID2);

          if (tsID > 0 && tsID2 == 0 && nconst)
            {
              for (varID = 0; varID < nvars; varID++)
                {
                  if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT)
                    {
                      const auto nlevels = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));
                      for (levelID = 0; levelID < nlevels; levelID++)
                        {
                          cdoDefRecord(streamID2, varID, levelID);
                          nmiss = vars[varID][levelID].nmiss;
                          cdoWriteRecord(streamID2, vars[varID][levelID].vec_d.data(), nmiss);
                        }
                    }
                }
            }

          for (int recID = 0; recID < nrecs; recID++)
            {
              cdoInqRecord(streamID1, &varID, &levelID);
              cdoDefRecord(streamID2, varID, levelID);
              if (lcopy && !(tsID == 0 && nconst))
                {
                  cdoCopyRecord(streamID2, streamID1);
                }
              else
                {
                  cdoReadRecord(streamID1, array.data(), &nmiss);
                  cdoWriteRecord(streamID2, array.data(), nmiss);

                  if (tsID == 0 && nconst)
                    {
                      if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT)
                        {
                          const auto gridID = vlistInqVarGrid(vlistID1, varID);
                          const auto gridsize = gridInqSize(gridID);
                          varrayCopy(gridsize, array, vars[varID][levelID].vec_d);
                          vars[varID][levelID].nmiss = nmiss;
                        }
                    }
                }
            }

          tsID++;
          tsID2++;
        }

      cdoStreamClose(streamID2);
      if (nrecs == 0) break;

      nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs == 0) break;

      for (; i2 < (int) (nskip * (index + 1)); i2++)
        {
          nrecs = cdoStreamInqTimestep(streamID1, tsID);
          if (nrecs == 0) break;
          tsID++;
        }

      nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs == 0) break;

      index++;
    }

LABEL_END:

  cdoStreamClose(streamID1);

  vlistDestroy(vlistID2);

  cdoFinish();

  return nullptr;
}
