/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Sort sortcode  Sort by code number
*/

#include <cdi.h>

#include "cdo_options.h"
#include "cdo_vlist.h"
#include "dmemory.h"
#include "process_int.h"
#include "param_conversion.h"
#include "cdo_zaxis.h"

struct levinfo_t
{
  int levelID;
  size_t nmiss;
  double level;
};

struct varinfo_t
{
  int varID;
  int nlevs;
  int code;
  char param[CDI_MAX_NAME];
  char name[CDI_MAX_NAME];
  levinfo_t *levInfo;
};

static int
cmpvarcode(const void *a, const void *b)
{
  const auto x = ((const varinfo_t *) a)->code;
  const auto y = ((const varinfo_t *) b)->code;
  return ((x > y) - (x < y)) * 2 + (x > y) - (x < y);
}

static int
cmpvarparam(const void *a, const void *b)
{
  const auto x = ((const varinfo_t *) a)->param;
  const auto y = ((const varinfo_t *) b)->param;
  return strcmp(x, y);
}

static int
cmpvarname(const void *a, const void *b)
{
  const auto x = ((const varinfo_t *) a)->name;
  const auto y = ((const varinfo_t *) b)->name;
  return strcmp(x, y);
}

static int
cmpvarlevel(const void *a, const void *b)
{
  const auto x = ((const levinfo_t *) a)->level;
  const auto y = ((const levinfo_t *) b)->level;
  return ((x > y) - (x < y)) * 2 + (x > y) - (x < y);
}

static int
cmpvarlevelrev(const void *a, const void *b)
{
  const auto x = ((const levinfo_t *) a)->level;
  const auto y = ((const levinfo_t *) b)->level;
  return -1 * (((x > y) - (x < y)) * 2 + (x > y) - (x < y));
}

static void
setNmiss(int varID, int levelID, int nvars, varinfo_t *varInfo, size_t nmiss)
{
  int vindex, lindex;

  for (vindex = 0; vindex < nvars; vindex++)
    if (varInfo[vindex].varID == varID) break;

  if (vindex == nvars) cdoAbort("Internal problem; varID not found!");

  const auto nlevs = varInfo[vindex].nlevs;
  for (lindex = 0; lindex < nlevs; lindex++)
    if (varInfo[vindex].levInfo[lindex].levelID == levelID) break;

  if (lindex == nlevs) cdoAbort("Internal problem; levelID not found!");

  varInfo[vindex].levInfo[lindex].nmiss = nmiss;
}

void
paramToStringLong(int param, char *paramstr, int maxlen)
{
  int len;

  int dis, cat, num;
  cdiDecodeParam(param, &num, &cat, &dis);

  size_t umaxlen = maxlen >= 0 ? (unsigned) maxlen : 0U;
  if (dis == 255 && (cat == 255 || cat == 0))
    len = snprintf(paramstr, umaxlen, "%03d", num);
  else if (dis == 255)
    len = snprintf(paramstr, umaxlen, "%03d.%03d", num, cat);
  else
    len = snprintf(paramstr, umaxlen, "%03d.%03d.%03d", num, cat, dis);

  if (len >= maxlen || len < 0) fprintf(stderr, "Internal problem (%s): size of input string is too small!\n", __func__);
}

void *
Sort(void *process)
{
  int varID, levelID, zaxisID;
  int vindex, lindex;
  int nrecs, nlevs;
  size_t nmiss;
  int (*cmpvarlev)(const void *, const void *) = cmpvarlevel;

  cdoInitialize(process);

  // clang-format off
  const auto SORTCODE  = cdoOperatorAdd("sortcode",  0, 0, nullptr);
  const auto SORTPARAM = cdoOperatorAdd("sortparam", 0, 0, nullptr);
  const auto SORTNAME  = cdoOperatorAdd("sortname",  0, 0, nullptr);
  const auto SORTLEVEL = cdoOperatorAdd("sortlevel", 0, 0, nullptr);
  // clang-format on

  const auto operatorID = cdoOperatorID();

  if (operatorArgc() > 1) cdoAbort("Too many arguments!");

  if (operatorID == SORTLEVEL && operatorArgc() == 1)
    {
      auto iarg = parameter2int(cdoOperatorArgv(0));
      if (iarg < 0) cmpvarlev = cmpvarlevelrev;
    }

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);
  /*
  if ( operatorID == SORTCODE )
      vlistSortCode(vlistID2);
   else if ( operatorID == SORTNAME )
      ;
   else if ( operatorID == SORTLEVEL )
      ;
  */

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  VarList varList1;
  varListInit(varList1, vlistID1);

  const auto nvars = vlistNvars(vlistID1);

  varinfo_t *varInfo = (varinfo_t *) Malloc(nvars * sizeof(varinfo_t));
  for (varID = 0; varID < nvars; ++varID)
    {
      nlevs = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));
      varInfo[varID].nlevs = nlevs;
      varInfo[varID].levInfo = (levinfo_t *) Malloc(nlevs * sizeof(levinfo_t));
    }

  double **vardata = (double **) Malloc(nvars * sizeof(double *));

  for (varID = 0; varID < nvars; varID++)
    {
      vardata[varID] = (double *) Malloc(varList1[varID].gridsize * varList1[varID].nlevels * sizeof(double));
    }

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          if (tsID == 0)
            {
              varInfo[varID].varID = varID;
              varInfo[varID].code = vlistInqVarCode(vlistID1, varID);
              const auto iparam = vlistInqVarParam(vlistID1, varID);
              paramToStringLong(iparam, varInfo[varID].param, sizeof(varInfo[varID].param));
              vlistInqVarName(vlistID1, varID, varInfo[varID].name);
              zaxisID = vlistInqVarZaxis(vlistID1, varID);
              varInfo[varID].levInfo[levelID].levelID = levelID;
              varInfo[varID].levInfo[levelID].level = cdoZaxisInqLevel(zaxisID, levelID);
            }

          const auto offset = varList1[varID].gridsize * levelID;
          auto single = vardata[varID] + offset;

          cdoReadRecord(streamID1, single, &nmiss);

          setNmiss(varID, levelID, nvars, varInfo, nmiss);
          // varInfo[varID].levInfo[levelID].nmiss = nmiss;
        }

      if (tsID == 0)
        {
          if (Options::cdoVerbose)
            for (vindex = 0; vindex < nvars; vindex++)
              {
                nlevs = varInfo[vindex].nlevs;
                for (lindex = 0; lindex < nlevs; ++lindex)
                  printf("sort in: %d %s %d %d %g\n", vindex, varInfo[vindex].name, varInfo[vindex].code, varInfo[vindex].nlevs,
                         varInfo[vindex].levInfo[lindex].level);
              }

          if (operatorID == SORTCODE)
            std::qsort(varInfo, nvars, sizeof(varinfo_t), cmpvarcode);
          else if (operatorID == SORTPARAM)
            std::qsort(varInfo, nvars, sizeof(varinfo_t), cmpvarparam);
          else if (operatorID == SORTNAME)
            std::qsort(varInfo, nvars, sizeof(varinfo_t), cmpvarname);
          else if (operatorID == SORTLEVEL)
            {
              for (vindex = 0; vindex < nvars; vindex++)
                {
                  nlevs = varInfo[vindex].nlevs;
                  std::qsort(varInfo[vindex].levInfo, nlevs, sizeof(levinfo_t), cmpvarlev);
                }
            }

          if (Options::cdoVerbose)
            for (vindex = 0; vindex < nvars; vindex++)
              {
                nlevs = varInfo[vindex].nlevs;
                for (lindex = 0; lindex < nlevs; ++lindex)
                  printf("sort out: %d %s %d %d %g\n", vindex, varInfo[vindex].name, varInfo[vindex].code, varInfo[vindex].nlevs,
                         varInfo[vindex].levInfo[lindex].level);
              }
        }

      for (vindex = 0; vindex < nvars; vindex++)
        {
          varID = varInfo[vindex].varID;
          nlevs = varInfo[vindex].nlevs;
          for (lindex = 0; lindex < nlevs; ++lindex)
            {
              levelID = varInfo[vindex].levInfo[lindex].levelID;
              nmiss = varInfo[vindex].levInfo[lindex].nmiss;

              if (tsID == 0 || vlistInqVarTimetype(vlistID1, varID) != TIME_CONSTANT)
                {
                  const auto offset = varList1[varID].gridsize * levelID;
                  auto single = vardata[varID] + offset;

                  cdoDefRecord(streamID2, varID, levelID);
                  cdoWriteRecord(streamID2, single, nmiss);
                }
            }
        }

      tsID++;
    }

  cdoStreamClose(streamID1);
  cdoStreamClose(streamID2);

  for (varID = 0; varID < nvars; varID++) Free(vardata[varID]);
  Free(vardata);

  for (vindex = 0; vindex < nvars; vindex++) Free(varInfo[vindex].levInfo);
  Free(varInfo);

  cdoFinish();

  return nullptr;
}
