/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Select      select         Select fields
*/

#include <cdi.h>

#include "cdo_options.h"
#include "functs.h"
#include "process_int.h"
#include "cdo_vlist.h"
#include "cdo_zaxis.h"
#include "datetime.h"
#include "sellist.h"
#include "printinfo.h"
#include "progress.h"
#include "cdi_lockedIO.h"

double datestrToDouble(const char *datestr, int opt);

int cdo_read_timestepmask(const char *maskfile, std::vector<bool> &imask);

static void
write_const_vars(CdoStreamID streamID2, const int vlistID2, const int nvars, Varray2D<double> &vardata2)
{
  for (int varID2c = 0; varID2c < nvars; ++varID2c)
    {
      if (vardata2[varID2c].size())
        {
          const auto missval = vlistInqVarMissval(vlistID2, varID2c);
          const auto gridsize = gridInqSize(vlistInqVarGrid(vlistID2, varID2c));
          const auto nlevel = zaxisInqSize(vlistInqVarZaxis(vlistID2, varID2c));
          for (int levelID2c = 0; levelID2c < nlevel; ++levelID2c)
            {
              auto pdata = &vardata2[varID2c][gridsize * levelID2c];
              const auto nmiss = arrayNumMV(gridsize, pdata, missval);
              cdoDefRecord(streamID2, varID2c, levelID2c);
              cdoWriteRecord(streamID2, pdata, nmiss);
            }
          vardata2[varID2c].clear();
          vardata2[varID2c].shrink_to_fit();
        }
    }
}

static void
eval_timestepmask(const char *maskfile, KVList &kvlist)
{
  std::vector<bool> imask;
  int n = cdo_read_timestepmask(maskfile, imask);
  if (n < 0) cdoAbort("Read of timestep mask failed!");

  int nvalues = 0;
  for (int i = 0; i < n; ++i)
    if (imask[i]) nvalues++;
  if (nvalues == 0)
    cdoPrint("timestepmask has no values!");
  else
    {
      KeyValues kv;
      kv.key = "timestep";
      kv.nvalues = nvalues;
      kv.values.resize(nvalues);

      std::vector<char> value;
      int j = 0;
      for (int i = 0; i < n; ++i)
        {
          if (imask[i])
            {
              const size_t length = (size_t) std::log10(j + 1) + 2;
              value.resize(length);
              sprintf(value.data(), "%d", i + 1);
              kv.values[j++] = value.data();
            }
        }

      kvlist.push_back(kv);
    }
}

const char *
steptypeName(const int tsteptype)
{
  const char *steptype;
  // clang-format off
  if      (tsteptype == TSTEP_INSTANT)  steptype = "instant";
  else if (tsteptype == TSTEP_INSTANT2) steptype = "instant";
  else if (tsteptype == TSTEP_INSTANT3) steptype = "instant";
  else if (tsteptype == TSTEP_MIN)      steptype = "min";
  else if (tsteptype == TSTEP_MAX)      steptype = "max";
  else if (tsteptype == TSTEP_AVG)      steptype = "avg";
  else if (tsteptype == TSTEP_ACCUM)    steptype = "accum";
  else if (tsteptype == TSTEP_RANGE)    steptype = "range";
  else if (tsteptype == TSTEP_DIFF)     steptype = "diff";
  else if (tsteptype == TSTEP_SUM)      steptype = "sum";
  else steptype = "unknown";
  // clang-format on
  return steptype;
}

void *
Select(void *process)
{
  bool lconstvars = true;
  CdoStreamID streamID2 = CDO_STREAM_UNDEF;
  int nrecs;
  int nvars, nvars2 = 0;
  int varID, levelID;
  int last_year = -999999999;
  char paramstr[32];
  char varname[CDI_MAX_NAME];
  char stdname[CDI_MAX_NAME];
  char gname[CDI_MAX_NAME];
  char zname[CDI_MAX_NAME];
  int vlistID0 = -1, vlistID2 = -1;
  int taxisID2 = CDI_UNDEFID;
  int ntsteps2 = 0;
  bool ltimsel = false;
  std::vector<bool> vars;
  Varray2D<double> vardata2;
  Varray<double> array;
  double fstartdate = -99999999999.;
  double fenddate = -99999999999.;

  cdoInitialize(process);

  const auto SELECT = cdoOperatorAdd("select", 0, 0, "parameter list");
  const auto DELETE = cdoOperatorAdd("delete", 0, 0, "parameter list");

  const auto lcopy = unchangedRecord();

  const auto operatorID = cdoOperatorID();

  operatorInputArg(cdoOperatorEnter(operatorID));

  const auto argc = operatorArgc();
  auto argnames = cdoGetOperArgv();

  if (argc == 0) cdoAbort("Parameter missing!");

  KVList kvlist;
  kvlist.name = "SELECT";
  if (kvlist.parseArguments(argc, argnames) != 0) cdoAbort("Parse error!");
  if (Options::cdoVerbose) kvlist.print();

  auto kv = kvlist.search("timestepmask");
  if (kv && kv->nvalues > 0)
    {
      if (kvlist.search("timestep")) cdoAbort("Parameter timestep and timestepmask can't be combined!");
      eval_timestepmask(kv->values[0].c_str(), kvlist);
    }

  SelectList sellist;
  sellist_init(sellist, kvlist);

  // clang-format off
  SELLIST_ADD_INT(timestep_of_year, "Timestep of year");
  SELLIST_ADD_INT(timestep,         "Timestep");
  SELLIST_ADD_INT(year,             "Year");
  SELLIST_ADD_INT(month,            "Month");
  SELLIST_ADD_INT(day,              "Day");
  SELLIST_ADD_INT(hour,             "Hour");
  SELLIST_ADD_INT(minute,           "Minute");
  SELLIST_ADD_INT(code,             "Code number");
  SELLIST_ADD_INT(levidx,           "Level index");
  SELLIST_ADD_INT(ltype,            "Level type");
  SELLIST_ADD_INT(zaxisnum,         "Zaxis number");
  SELLIST_ADD_INT(gridnum,          "Grid number");
  SELLIST_ADD_FLT(level,            "Level");
  SELLIST_ADD_WORD(name,            "Variable name");
  SELLIST_ADD_WORD(param,           "Parameter");
  SELLIST_ADD_WORD(zaxisname,       "Zaxis name");
  SELLIST_ADD_WORD(gridname,        "Grid name");
  SELLIST_ADD_WORD(steptype,        "Time step type");
  SELLIST_ADD_WORD(startdate,       "Start date");
  SELLIST_ADD_WORD(enddate,         "End date");
  SELLIST_ADD_WORD(season,          "Season");
  SELLIST_ADD_WORD(date,            "Date");
  SELLIST_ADD_WORD(timestepmask,    "Timestep mask");
  // clang-format on

  if (Options::cdoVerbose) sellist_print(sellist);

  sellist_verify(sellist);

  if (SELLIST_NVAL(timestepmask) > 1) cdoAbort("Key timestepmask has too many values!");
  (void) (timestepmask);  // CDO_UNUSED

  const auto nfiles = cdoStreamCnt() - 1;

  DateTimeList dtlist;

  if (!Options::cdoVerbose && nfiles > 1) progress::init();

  int tsmax = -1;

  timestep = 0;
  int tsID2 = 0;
  for (int indf = 0; indf < nfiles; ++indf)
    {
      if (!Options::cdoVerbose && nfiles > 1) progress::update(0, 1, (indf + 1.) / nfiles);
      if (Options::cdoVerbose) cdoPrint("Process file: %s", cdoGetStreamName(indf));

      const auto streamID1 = cdoOpenRead(indf);

      const auto vlistID1 = cdoStreamInqVlist(streamID1);
      const auto taxisID1 = vlistInqTaxis(vlistID1);

      bool lcopy_const = false;

      if (indf == 0)
        {
          bool xresult = true;

          // vlistID0 = vlistDuplicate(vlistID1);

          vlistClearFlag(vlistID1);
          nvars = vlistNvars(vlistID1);
          vars.resize(nvars);

          if (operatorID == DELETE)
            {
              xresult = false;
              for (varID = 0; varID < nvars; ++varID)
                {
                  const auto zaxisID = vlistInqVarZaxis(vlistID1, varID);
                  const auto nlevs = zaxisInqSize(zaxisID);
                  for (int levID = 0; levID < nlevs; ++levID) vlistDefFlag(vlistID1, varID, levID, true);
                }
            }

          const bool findVariable = SELLIST_NVAL(code) || SELLIST_NVAL(name) || SELLIST_NVAL(param);

          bool lvarsel = findVariable || SELLIST_NVAL(ltype) || SELLIST_NVAL(zaxisnum) || SELLIST_NVAL(gridnum)
                         || SELLIST_NVAL(zaxisname) || SELLIST_NVAL(gridname) || SELLIST_NVAL(steptype);

          bool llevsel = SELLIST_NVAL(level) || SELLIST_NVAL(levidx);

          ltimsel = SELLIST_NVAL(date) || SELLIST_NVAL(startdate) || SELLIST_NVAL(enddate) || SELLIST_NVAL(season)
                    || SELLIST_NVAL(timestep_of_year) || SELLIST_NVAL(timestep) || SELLIST_NVAL(year) || SELLIST_NVAL(month)
                    || SELLIST_NVAL(day) || SELLIST_NVAL(hour) || SELLIST_NVAL(minute);

          for (varID = 0; varID < nvars; ++varID)
            {
              const int iparam = vlistInqVarParam(vlistID1, varID);
              code = vlistInqVarCode(vlistID1, varID);
              vlistInqVarName(vlistID1, varID, varname);
              vlistInqVarStdname(vlistID1, varID, stdname);

              cdiParamToString(iparam, paramstr, sizeof(paramstr));

              name = varname;
              param = paramstr;

              const auto gridID = vlistInqVarGrid(vlistID1, varID);
              const auto zaxisID = vlistInqVarZaxis(vlistID1, varID);
              const auto nlevs = zaxisInqSize(zaxisID);
              ltype = zaxis2ltype(zaxisID);

              zaxisnum = vlistZaxisIndex(vlistID1, zaxisID) + 1;
              const auto zaxistype = zaxisInqType(zaxisID);
              zaxisName(zaxistype, zname);
              zaxisname = zname;

              gridnum = vlistGridIndex(vlistID1, gridID) + 1;
              const auto gridtype = gridInqType(gridID);
              gridName(gridtype, gname);
              gridname = gname;

              const auto tsteptype = vlistInqVarTsteptype(vlistID1, varID);
              steptype = steptypeName(tsteptype);

              vars[varID] = false;

              const auto found_code = SELLIST_CHECK(code);
              const auto found_name = SELLIST_CHECK(name);
              const auto found_param = SELLIST_CHECK(param);
              const auto found_grid = SELLIST_CHECK(gridnum);
              const auto found_gname = SELLIST_CHECK(gridname);
              const auto found_stype = SELLIST_CHECK(steptype);
              const auto found_ltype = SELLIST_CHECK(ltype);
              const auto found_zaxis = SELLIST_CHECK(zaxisnum);
              const auto found_zname = SELLIST_CHECK(zaxisname);

              bool lvar = found_code || found_name || found_param;
              bool lstep = SELLIST_NVAL(steptype) ? found_stype : true;
              bool lgrid = (SELLIST_NVAL(gridnum) || SELLIST_NVAL(gridname)) ? (found_grid || found_gname) : true;
              bool lvert = (SELLIST_NVAL(ltype) || SELLIST_NVAL(zaxisnum) || SELLIST_NVAL(zaxisname))
                               ? (found_ltype || found_zaxis || found_zname)
                               : true;

              if (lvar && lgrid && lvert && lstep) vars[varID] = true;

              if (!vars[varID] && !lvar && !findVariable)
                {
                  if (found_grid || found_gname)
                    vars[varID] = true;
                  else if (found_stype)
                    vars[varID] = true;
                  else if (found_ltype || found_zaxis || found_zname)
                    vars[varID] = true;
                  else if (!lvarsel && (SELLIST_NVAL(levidx) || SELLIST_NVAL(level)))
                    {
                      for (int levID = 0; levID < nlevs; ++levID)
                        {
                          levidx = levID + 1;
                          level = cdoZaxisInqLevel(zaxisID, levID);
                          if (!vars[varID] && SELLIST_CHECK(levidx)) vars[varID] = true;
                          if (!vars[varID] && SELLIST_CHECK(level)) vars[varID] = true;
                        }
                    }
                }
            }

          for (varID = 0; varID < nvars; ++varID)
            {
              if (vars[varID])
                {
                  const int zaxisID = vlistInqVarZaxis(vlistID1, varID);
                  if (zaxisInqType(zaxisID) == ZAXIS_HYBRID)
                    {
                      const int psvarid = vlist_get_psvarid(vlistID1, zaxisID);
                      if (psvarid != -1 && !vars[psvarid]) vars[psvarid] = true;
                    }
                }
            }

          for (varID = 0; varID < nvars; ++varID)
            {
              if (vars[varID])
                {
                  const auto zaxisID = vlistInqVarZaxis(vlistID1, varID);
                  const auto nlevs = zaxisInqSize(zaxisID);
                  for (int levID = 0; levID < nlevs; ++levID)
                    {
                      levidx = levID + 1;
                      level = cdoZaxisInqLevel(zaxisID, levID);

                      if (nlevs == 1 && IS_EQUAL(level, 0))
                        {
                          SELLIST_CHECK(level);
                          vlistDefFlag(vlistID1, varID, levID, xresult);
                        }
                      else
                        {
                          if (SELLIST_NVAL(levidx))
                            {
                              if (SELLIST_CHECK(levidx)) vlistDefFlag(vlistID1, varID, levID, xresult);
                            }
                          else if (SELLIST_NVAL(level))
                            {
                              if (SELLIST_CHECK(level)) vlistDefFlag(vlistID1, varID, levID, xresult);
                            }
                          else
                            {
                              vlistDefFlag(vlistID1, varID, levID, xresult);
                            }
                        }
                    }
                }
            }

          SELLIST_CHECK_FLAG(code);
          SELLIST_CHECK_FLAG(levidx);
          SELLIST_CHECK_FLAG(ltype);
          SELLIST_CHECK_FLAG(zaxisnum);
          SELLIST_CHECK_FLAG(gridnum);
          SELLIST_CHECK_FLAG(level);
          SELLIST_CHECK_FLAG(name);
          SELLIST_CHECK_FLAG(param);
          SELLIST_CHECK_FLAG(zaxisname);
          SELLIST_CHECK_FLAG(gridname);
          SELLIST_CHECK_FLAG(steptype);

          int npar = 0;
          for (varID = 0; varID < nvars; ++varID)
            {
              const auto zaxisID = vlistInqVarZaxis(vlistID1, varID);
              const auto nlevs = zaxisInqSize(zaxisID);
              for (int levID = 0; levID < nlevs; ++levID)
                if (vlistInqFlag(vlistID1, varID, levID) == true)
                  {
                    npar++;
                    break;
                  }
            }

          if (npar == 0)
            {
              if ((!lvarsel) && (!llevsel) && ltimsel)
                {
                  lcopy_const = true;

                  for (varID = 0; varID < nvars; ++varID)
                    {
                      vars[varID] = true;
                      const auto zaxisID = vlistInqVarZaxis(vlistID1, varID);
                      const auto nlevs = zaxisInqSize(zaxisID);
                      for (int levID = 0; levID < nlevs; ++levID) vlistDefFlag(vlistID1, varID, levID, true);
                    }
                }
              else
                {
                  cdoAbort("No variable selected!");
                }
            }
          else
            {
              if (lvarsel && ltimsel)
                {
                  for (varID = 0; varID < nvars; ++varID)
                    {
                      if (vars[varID] == true && vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT)
                        {
                          lcopy_const = true;
                          break;
                        }
                    }
                }
            }

          // if ( Options::cdoVerbose ) vlistPrint(vlistID1);

          vlistID0 = vlistDuplicate(vlistID1);
          for (varID = 0; varID < nvars; ++varID)
            {
              const auto zaxisID = vlistInqVarZaxis(vlistID1, varID);
              const auto nlevs = zaxisInqSize(zaxisID);
              for (int levID = 0; levID < nlevs; ++levID)
                vlistDefFlag(vlistID0, varID, levID, vlistInqFlag(vlistID1, varID, levID));
            }

          // if ( Options::cdoVerbose ) vlistPrint(vlistID0);

          vlistID2 = vlistCreate();
          cdoVlistCopyFlag(vlistID2, vlistID0);

          // if ( Options::cdoVerbose ) vlistPrint(vlistID2);

          taxisID2 = taxisDuplicate(taxisID1);

          auto ntsteps = vlistNtsteps(vlistID1);

          nvars2 = vlistNvars(vlistID2);

          if (ntsteps == 1 && nfiles == 1)
            {
              for (varID = 0; varID < nvars2; ++varID)
                if (vlistInqVarTimetype(vlistID2, varID) != TIME_CONSTANT) break;

              if (varID == nvars2) ntsteps = 0;
            }

          ntsteps2 = (operatorID == SELECT && SELLIST_NVAL(timestep) == 1) ? 1 : ntsteps;

          if (ntsteps2 == 0 && nfiles > 1)
            {
              lconstvars = false;
              for (varID = 0; varID < nvars2; ++varID) vlistDefVarTimetype(vlistID2, varID, TIME_VARYING);
            }

          const auto nsel = SELLIST_NVAL(timestep);

          // support for negative timestep values
          if (nsel > 0)
            {
              for (int i = 0; i < nsel; ++i)
                {
                  int ptimestep;
                  SELLIST_GET_VAL(timestep, i, &ptimestep);
                  if (ptimestep < 0)
                    {
                      if (nfiles != 1) cdoAbort("Negative timesteps only supported with one input stream!");
                      if (ntsteps < 0 && !cdoAssertFilesOnly() == false)
                        {
                          int tsID = 0;
                          while (cdoStreamInqTimestep(streamID1, tsID)) tsID++;
                          ntsteps = tsID;
                          if (Options::cdoVerbose) cdoPrint("Found %d timesteps", ntsteps);
                        }
                      if (ntsteps > 0)
                        {
                          if (Options::cdoVerbose) cdoPrint("timestep %d changed to %d", ptimestep, ptimestep + ntsteps + 1);
                          ptimestep += ntsteps + 1;
                          SELLIST_DEF_VAL(timestep, i, &ptimestep);
                        }
                    }
                }

              for (int i = 0; i < nsel; i++)
                {
                  int ptimestep;
                  SELLIST_GET_VAL(timestep, i, &ptimestep);
                  tsmax = std::max(tsmax, ptimestep);
                }
            }

          if (!lcopy)
            {
              auto gridsizemax = vlistGridsizeMax(vlistID1);
              if (vlistNumber(vlistID1) != CDI_REAL) gridsizemax *= 2;
              array.resize(gridsizemax);
            }

          SELLIST_GET_VAL(startdate, 0, &startdate);
          SELLIST_GET_VAL(enddate, 0, &enddate);
          if (SELLIST_NVAL(startdate)) fstartdate = datestrToDouble(startdate, 0);
          if (SELLIST_NVAL(enddate)) fenddate = datestrToDouble(enddate, 1);
        }
      else
        {
          vlistCompare(vlistID0, vlistID1, CMP_ALL);
        }

      if (nvars2 == 0)
        {
          cdoWarning("No variable selected!");
          goto END_LABEL;
        }

      if (lcopy_const) vardata2.resize(nvars2);

      bool lstop = false;
      int tsID1 = 0;
      while ((nrecs = cdoStreamInqTimestep(streamID1, tsID1)))
        {
          timestep++;
          bool copytimestep = true;

          if (ltimsel)
            {
              copytimestep = false;

              if (operatorID == SELECT && SELLIST_NVAL(timestep) > 0)
                {
                  if (timestep > tsmax)
                    {
                      lstop = true;
                      break;
                    }
                }

              dtlist.taxisInqTimestep(taxisID1, 0);
              const auto vdate = dtlist.getVdate(0);
              const auto vtime = dtlist.getVtime(0);
              int second;
              cdiDecodeDate(vdate, &year, &month, &day);
              cdiDecodeTime(vtime, &hour, &minute, &second);
              (void) (season);  // CDO_UNUSED

              if (year != last_year)
                {
                  timestep_of_year = 0;
                  last_year = year;
                }

              timestep_of_year++;

              if (SELLIST_CHECK(timestep)) copytimestep = true;
              if (SELLIST_CHECK(timestep_of_year)) copytimestep = true;

              if (!copytimestep && SELLIST_NVAL(date) == 0 && SELLIST_NVAL(timestep) == 0 && SELLIST_NVAL(timestep_of_year) == 0)
                {
                  bool lseason = false, lyear = false, lmonth = false, lday = false, lhour = false, lminute = false;

                  if (SELLIST_NVAL(season) == 0 || SELLIST_CHECK_SEASON(season, month)) lseason = true;
                  if (SELLIST_NVAL(year) == 0 || SELLIST_CHECK(year)) lyear = true;
                  if (SELLIST_NVAL(month) == 0 || SELLIST_CHECK(month)) lmonth = true;
                  if (SELLIST_NVAL(day) == 0 || SELLIST_CHECK(day)) lday = true;
                  if (SELLIST_NVAL(hour) == 0 || SELLIST_CHECK(hour)) lhour = true;
                  if (SELLIST_NVAL(minute) == 0 || SELLIST_CHECK(minute)) lminute = true;

                  if (lseason && lyear && lmonth && lday && lhour && lminute) copytimestep = true;
                }

              const double fdate = ((double) vdate) + ((double) vtime) / 1000000.;

              if (SELLIST_NVAL(enddate))
                {
                  copytimestep = (fdate <= fenddate);
                  if (fdate > fenddate)
                    {
                      SELLIST_DEF_FLAG(enddate, 0, true);
                      if (operatorID == SELECT)
                        {
                          lstop = true;
                          break;
                        }
                    }
                }

              if (SELLIST_NVAL(startdate))
                {
                  copytimestep = (fdate >= fstartdate);
                  if (fdate >= fstartdate) SELLIST_DEF_FLAG(startdate, 0, true);
                }

              if (SELLIST_NVAL(date))
                {
                  const auto datetimeString = datetimeToString(vdate, vtime);
                  date = datetimeString.c_str();
                  if (SELLIST_CHECK_DATE(date)) copytimestep = true;
                }

              if (operatorID == DELETE) copytimestep = !copytimestep;

              if (copytimestep && indf == 0 && tsID1 == 0) lcopy_const = false;
            }

          if (copytimestep)
            {
              taxisCopyTimestep(taxisID2, taxisID1);
              if (streamID2 == CDO_STREAM_UNDEF)
                {
                  const bool lasttimestep = (nfiles == 1) && (ntsteps2 > 1) && (ntsteps2 == (tsID1 + 1));
                  if (lasttimestep && tsID2 == 0) ntsteps2 = 1;
                  if (ntsteps2 == 0 || ntsteps2 == 1) vlistDefNtsteps(vlistID2, ntsteps2);
                  streamID2 = cdoOpenWrite(nfiles);
                  vlistDefTaxis(vlistID2, taxisID2);
                  cdoDefVlist(streamID2, vlistID2);
                }
              cdoDefTimestep(streamID2, tsID2);

              for (int recID = 0; recID < nrecs; ++recID)
                {
                  cdoInqRecord(streamID1, &varID, &levelID);
                  if (vlistInqFlag(vlistID0, varID, levelID) == true)
                    {
                      if (lconstvars && tsID2 > 0 && tsID1 == 0)
                        if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT) continue;

                      const auto varID2 = vlistFindVar(vlistID2, varID);
                      const auto levelID2 = vlistFindLevel(vlistID2, varID, levelID);
                      if (lcopy_const && tsID2 == 0) write_const_vars(streamID2, vlistID2, varID2, vardata2);

                      cdoDefRecord(streamID2, varID2, levelID2);
                      if (lcopy)
                        {
                          cdoCopyRecord(streamID2, streamID1);
                        }
                      else
                        {
                          size_t nmiss;
                          cdoReadRecord(streamID1, array.data(), &nmiss);
                          cdoWriteRecord(streamID2, array.data(), nmiss);
                        }
                    }
                }

              if (lcopy_const && tsID2 == 0) write_const_vars(streamID2, vlistID2, nvars2, vardata2);

              tsID2++;
            }
          else if (lcopy_const && indf == 0 && tsID1 == 0)
            {
              for (int recID = 0; recID < nrecs; ++recID)
                {
                  cdoInqRecord(streamID1, &varID, &levelID);
                  if (vlistInqFlag(vlistID0, varID, levelID) == true)
                    {
                      const auto varID2 = vlistFindVar(vlistID2, varID);
                      if (vlistInqVarTimetype(vlistID2, varID2) == TIME_CONSTANT)
                        {
                          const auto levelID2 = vlistFindLevel(vlistID2, varID, levelID);
                          const auto gridsize = gridInqSize(vlistInqVarGrid(vlistID2, varID2));
                          if (levelID == 0)
                            {
                              const auto nlevel = zaxisInqSize(vlistInqVarZaxis(vlistID2, varID2));
                              vardata2[varID2].resize(gridsize * nlevel);
                            }
                          size_t nmiss;
                          cdoReadRecord(streamID1, &vardata2[varID2][gridsize * levelID2], &nmiss);
                        }
                    }
                }
            }

          tsID1++;
        }

      cdoStreamClose(streamID1);

      if (lstop) break;
    }

END_LABEL:

  if (!Options::cdoVerbose && nfiles > 1) progress::update(0, 1, 1);

  SELLIST_CHECK_FLAG(timestep_of_year);
  SELLIST_CHECK_FLAG(timestep);
  SELLIST_CHECK_FLAG(year);
  SELLIST_CHECK_FLAG(month);
  SELLIST_CHECK_FLAG(day);
  SELLIST_CHECK_FLAG(hour);
  SELLIST_CHECK_FLAG(minute);
  SELLIST_CHECK_FLAG(startdate);
  //  SELLIST_CHECK_FLAG(enddate);
  SELLIST_CHECK_FLAG(season);
  SELLIST_CHECK_FLAG(date);

  if (streamID2 != CDO_STREAM_UNDEF) cdoStreamClose(streamID2);

  vlistDestroy(vlistID0);
  vlistDestroy(vlistID2);

  sellist_destroy(sellist);

  if (tsID2 == 0) cdoAbort("No timesteps selected!");

  cdoFinish();

  return nullptr;
}
