/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Intlevel   intlevel        Linear level interpolation
*/

#include <cdi.h>

#include "cdo_options.h"
#include "process_int.h"
#include "cdo_vlist.h"
#include "cdo_zaxis.h"
#include "param_conversion.h"

static double
vert_interp_lev_kernel(double w1, double w2, const double var1L1, const double var1L2, const double missval)
{
  auto var2 = missval;

  if (DBL_IS_EQUAL(var1L1, missval)) w1 = 0;
  if (DBL_IS_EQUAL(var1L2, missval)) w2 = 0;

  if (IS_EQUAL(w1, 0) && IS_EQUAL(w2, 0))
    {
      var2 = missval;
    }
  else if (IS_EQUAL(w1, 0))
    {
      var2 = (w2 >= 0.5) ? var1L2 : missval;
    }
  else if (IS_EQUAL(w2, 0))
    {
      var2 = (w1 >= 0.5) ? var1L1 : missval;
    }
  else
    {
      var2 = var1L1 * w1 + var1L2 * w2;
    }

  return var2;
}

static void
vert_interp_lev(size_t gridsize, double missval, const Varray<double> &vardata1, Varray<double> &vardata2, int nlev2,
                const Varray<int> &lev_idx1, const Varray<int> &lev_idx2, const Varray<double> &lev_wgt1, const Varray<double> &lev_wgt2)
{
  for (int ilev = 0; ilev < nlev2; ++ilev)
    {
      const auto idx1 = lev_idx1[ilev];
      const auto idx2 = lev_idx2[ilev];
      auto wgt1 = lev_wgt1[ilev];
      auto wgt2 = lev_wgt2[ilev];
      auto var2 = &vardata2[gridsize * ilev];

      auto var1L1 = &vardata1[gridsize * idx1];
      auto var1L2 = &vardata1[gridsize * idx2];

#ifdef _OPENMP
#pragma omp parallel for default(none) shared(gridsize, var2, var1L1, var1L2, missval, wgt1, wgt2)
#endif
      for (size_t i = 0; i < gridsize; ++i)
        {
          var2[i] = vert_interp_lev_kernel(wgt1, wgt2, var1L1[i], var1L2[i], missval);
        }
    }
}

/*
 * 3d vertical interpolation routine (see vert_interp_lev() in src/Intlevel.cc)
 */
void
vert_interp_lev3d(size_t gridsize, double missval, const Varray<double> &vardata1, Varray<double> &vardata2, int nlev2,
                  const Varray<int> &lev_idx1, const Varray<int> &lev_idx2, const Varray<double> &lev_wgt1, const Varray<double> &lev_wgt2)
{
  for (int ilev = 0; ilev < nlev2; ilev++)
    {
      auto offset = ilev * gridsize;
      auto var2 = &vardata2[offset];

#ifdef _OPENMP
#pragma omp parallel for default(none) shared(gridsize, offset, vardata1, var2, lev_idx1, lev_idx2, lev_wgt1, lev_wgt2, missval)
#endif
      for (size_t i = 0; i < gridsize; i++)
        {
          const auto idx1 = lev_idx1[offset + i];
          const auto idx2 = lev_idx2[offset + i];
          const auto wgt1 = lev_wgt1[offset + i];
          const auto wgt2 = lev_wgt2[offset + i];

          // upper/lower values from input field
          const auto var1L1 = vardata1[idx1];
          const auto var1L2 = vardata1[idx2];

          var2[i] = vert_interp_lev_kernel(wgt1, wgt2, var1L1, var1L2, missval);
        }
    }
}

void
vert_gen_weights(int expol, int nlev1, const Varray<double> &lev1, int nlev2, const Varray<double> &lev2, Varray<int> &lev_idx1, Varray<int> &lev_idx2,
                 Varray<double> &lev_wgt1, Varray<double> &lev_wgt2)
{
  int i1;
  int idx1 = 0, idx2 = 0;
  double val1, val2 = 0;

  for (int i2 = 0; i2 < nlev2; ++i2)
    {
      // Because 2 levels were added to the source vertical coordinate (one on top, one at the bottom), its loop starts at 1
      for (i1 = 1; i1 < nlev1; ++i1)
        {
          if (lev1[i1 - 1] < lev1[i1])
            {
              idx1 = i1 - 1;
              idx2 = i1;
            }
          else
            {
              idx1 = i1;
              idx2 = i1 - 1;
            }
          val1 = lev1[idx1];
          val2 = lev1[idx2];

          if (lev2[i2] > val1 && lev2[i2] <= val2) break;
        }

      if (i1 == nlev1) cdoAbort("Level %g not found!", lev2[i2]);

      if (i1 - 1 == 0)  // destination levels ios not covert by the first two input z levels
        {
          lev_idx1[i2] = 1;
          lev_idx2[i2] = 1;
          lev_wgt1[i2] = 0;
          lev_wgt2[i2] = (expol || IS_EQUAL(lev2[i2], val2));
        }
      else if (i1 == nlev1 - 1)  // destination level is beyond the last value of the input z field
        {
          lev_idx1[i2] = nlev1 - 2;
          lev_idx2[i2] = nlev1 - 2;
          lev_wgt1[i2] = (expol || IS_EQUAL(lev2[i2], val2));
          lev_wgt2[i2] = 0;
        }
      else  // target z values has two bounday values in input z field
        {
          lev_idx1[i2] = idx1;
          lev_idx2[i2] = idx2;
          lev_wgt1[i2] = (lev1[idx2] - lev2[i2]) / (lev1[idx2] - lev1[idx1]);
          lev_wgt2[i2] = (lev2[i2] - lev1[idx1]) / (lev1[idx2] - lev1[idx1]);
        }
      // backshift of the indices because of the two additional levels in input vertical coordinate
      lev_idx1[i2]--;
      lev_idx2[i2]--;
      /*
        printf("%d %g %d %d %g %g %d %d %g %g\n", i2, lev2[i2], idx1, idx2, lev1[idx1], lev1[idx2],
        lev_idx1[i2], lev_idx2[i2], lev_wgt1[i2], lev_wgt2[i2]);
      */
    }
}

static void
vert_gen_weights3d1d(bool expol, int nlev1, size_t gridsize, Varray<double> &xlev1, int nlev2, const Varray<double> &lev2,
                     Varray<int> &xlev_idx1, Varray<int> &xlev_idx2, Varray<double> &xlev_wgt1, Varray<double> &xlev_wgt2)
{
  Varray<double> lev1(nlev1);
  Varray<double> lev_wgt1(nlev2), lev_wgt2(nlev2);
  Varray<int> lev_idx1(nlev2), lev_idx2(nlev2);

  for (size_t i = 0; i < gridsize; i++)
    {
      for (int k = 0; k < nlev1; ++k) lev1[k] = xlev1[k * gridsize + i];

      vert_gen_weights(expol, nlev1, lev1, nlev2, lev2, lev_idx1, lev_idx2, lev_wgt1, lev_wgt2);

      for (int k = 0; k < nlev2; ++k) xlev_idx1[k * gridsize + i] = lev_idx1[k] * gridsize + i;
      for (int k = 0; k < nlev2; ++k) xlev_idx2[k * gridsize + i] = lev_idx2[k] * gridsize + i;
      for (int k = 0; k < nlev2; ++k) xlev_wgt1[k * gridsize + i] = lev_wgt1[k];
      for (int k = 0; k < nlev2; ++k) xlev_wgt2[k * gridsize + i] = lev_wgt2[k];
    }
}

static bool
levelDirUp(const int nlev, const double *const lev)
{
  const bool lup = (nlev > 1 && lev[1] > lev[0]);
  for (int i = 1; i < nlev - 1; ++i)
    if (lup && lev[i + 1] <= lev[i]) return false;

  return lup;
}

static bool
levelDirDown(const int nlev, const double *const lev)
{
  const bool ldown = (nlev > 1 && lev[1] < lev[0]);
  for (int i = 1; i < nlev - 1; ++i)
    if (ldown && lev[i + 1] >= lev[i]) return false;

  return ldown;
}

void
vert_init_level_0_and_N(int nlev, size_t gridsize, Varray<double> &zlevels)
{
  // Check monotony of vertical levels
  Varray<double> zlev(nlev);
  for (int i = 0; i < nlev; ++i) zlev[i] = zlevels[(i + 1) * gridsize];
  const auto lup = levelDirUp(nlev, &zlev[0]);
  const auto ldown = levelDirDown(nlev, &zlev[0]);

  // Add artificial values for indication of extrapolation areas (lowermost + upmost levels)
  if (lup)
    {
      for (size_t i = 0; i < gridsize; i++)
        {
          zlevels[i] = -1.e33;
          zlevels[(nlev + 1) * gridsize + i] = 1.e33;
        }
    }
  else if (ldown)
    {
      for (size_t i = 0; i < gridsize; i++)
        {
          zlevels[i] = 1.e33;
          zlevels[(nlev + 1) * gridsize + i] = -1.e33;
        }
    }
  else
    cdoWarning("Non monotonic zaxis!");

  if (Options::cdoVerbose)
    for (int i = 0; i < nlev + 2; ++i) cdoPrint("lev1 %d: %g", i, zlevels[i * gridsize]);
}

void *
Intlevel(void *process)
{
  int nrecs;
  int varID, levelID;
  int zaxisID1 = -1;
  int nlevel = 0;

  cdoInitialize(process);

  // clang-format off
  const auto INTLEVEL  = cdoOperatorAdd("intlevel",  0, 0, nullptr);
  const auto INTLEVELX = cdoOperatorAdd("intlevelx", 0, 0, nullptr);
  // clang-format on

  (void) (INTLEVEL);  // CDO_UNUSED

  const auto operatorID = cdoOperatorID();

  const bool expol = (operatorID == INTLEVELX);

  operatorInputArg("<zvar> levels");

  auto argc = operatorArgc();
  auto argv = cdoGetOperArgv();
  const char *zvarname = nullptr;
  if (argc > 1 && isalpha((int) argv[0][0]))
    {
      zvarname = argv[0].c_str();
      argc--;
      argv = std::vector<std::string>(argv.begin() + 1, argv.end());
      if (Options::cdoVerbose) cdoPrint("zvarname = %s", zvarname);
    }

  const auto lev2 = cdoArgvToFlt(argv);
  const int nlev2 = lev2.size();

  if (Options::cdoVerbose)
    for (int i = 0; i < nlev2; ++i) cdoPrint("lev2 %d: %g", i, lev2[i]);

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  VarList varList1;
  varListInit(varList1, vlistID1);

  const auto nvars = vlistNvars(vlistID1);

  // Find z-variable
  int zaxisID2 = CDI_UNDEFID;
  int nlev1 = 0;
  Varray<double> lev1;
  int zvarID = CDI_UNDEFID;
  bool zvarIsVarying = false;
  size_t zvarGridsize = 0;
  size_t wisize = 0;
  if (zvarname)
    {
      for (varID = 0; varID < nvars; varID++)
        {
          char varname[CDI_MAX_NAME];
          vlistInqVarName(vlistID1, varID, varname);
          if (cstrIsEqual(zvarname, varname))
            {
              zvarID = varID;
              break;
            }
        }

      if (zvarID == CDI_UNDEFID) cdoAbort("Variable %s not found!", zvarname);
      zvarIsVarying = varList1[zvarID].timetype == TIME_VARYING;
      zvarGridsize = varList1[zvarID].gridsize;
      nlev1 = varList1[zvarID].nlevels;

      lev1.resize(zvarGridsize * (nlev1 + 2));

      zaxisID2 = zaxisCreate(ZAXIS_GENERIC, nlev2);

      char str[CDI_MAX_NAME];
      strcpy(str, "zlev");
      zaxisDefName(zaxisID2, str);
      str[0] = 0;
      vlistInqVarLongname(vlistID1, zvarID, str);
      if (str[0]) zaxisDefLongname(zaxisID2, str);
      str[0] = 0;
      vlistInqVarUnits(vlistID1, zvarID, str);
      if (str[0]) zaxisDefUnits(zaxisID2, str);

      wisize = zvarGridsize * nlev2;
    }
  else
    {
      int i;
      const auto nzaxis = vlistNzaxis(vlistID1);
      for (i = 0; i < nzaxis; i++)
        {
          const auto zaxisID = vlistZaxis(vlistID1, i);
          nlevel = zaxisInqSize(zaxisID);
          //if (zaxisInqType(zaxisID) != ZAXIS_HYBRID && zaxisInqType(zaxisID) != ZAXIS_HYBRID_HALF)
            if (nlevel > 1)
              {
                zaxisID1 = zaxisID;
                break;
              }
        }
      if (i == nzaxis) cdoAbort("No processable variable found!");

      zaxisID2 = zaxisCreate(zaxisInqType(zaxisID1), nlev2);

      char str[CDI_MAX_NAME];
      zaxisInqName(zaxisID1, str);
      zaxisDefName(zaxisID2, str);
      str[0] = 0;
      zaxisInqLongname(zaxisID1, str);
      if (str[0]) zaxisDefLongname(zaxisID2, str);
      str[0] = 0;
      zaxisInqUnits(zaxisID1, str);
      if (str[0]) zaxisDefUnits(zaxisID2, str);

      zaxisDefDatatype(zaxisID2, zaxisInqDatatype(zaxisID1));

      nlev1 = nlevel;
      lev1.resize(nlev1 + 2);
      cdoZaxisInqLevels(zaxisID1, &lev1[1]);

      const auto lup = levelDirUp(nlev1, &lev1[1]);
      const auto ldown = levelDirDown(nlev1, &lev1[1]);

      if (lup)
        {
          lev1[0] = -1.e33;
          lev1[nlev1 + 1] = 1.e33;
        }
      else if (ldown)
        {
          lev1[0] = 1.e33;
          lev1[nlev1 + 1] = -1.e33;
        }
      else
        cdoWarning("Non monotonic zaxis!");

      if (Options::cdoVerbose)
        for (i = 0; i < nlev1 + 2; ++i) cdoPrint("lev1 %d: %g", i, lev1[i]);

      wisize = nlev2;
    }

  zaxisDefLevels(zaxisID2, lev2.data());
  const auto nzaxis = vlistNzaxis(vlistID1);
  for (int i = 0; i < nzaxis; i++)
    if (zaxisID1 == vlistZaxis(vlistID1, i)) vlistChangeZaxisIndex(vlistID2, i, zaxisID2);

  VarList varList2;
  varListInit(varList2, vlistID2);

  Varray<int> lev_idx1(wisize), lev_idx2(wisize);
  Varray<double> lev_wgt1(wisize), lev_wgt2(wisize);

  const auto streamID2 = cdoOpenWrite(1);

  cdoDefVlist(streamID2, vlistID2);

  std::vector<bool> vars(nvars);
  std::vector<bool> varinterp(nvars);
  std::vector<std::vector<size_t>> varnmiss(nvars);
  Varray2D<double> vardata1(nvars), vardata2(nvars);

  const int maxlev = nlev1 > nlev2 ? nlev1 : nlev2;

  for (varID = 0; varID < nvars; varID++)
    {
      const auto gridsize = varList1[varID].gridsize;
      const auto nlevels = varList1[varID].nlevels;

      vardata1[varID].resize(gridsize * nlevels);

      varinterp[varID] = varList1[varID].zaxisID == zaxisID1;

      if (varinterp[varID])
        {
          vardata2[varID].resize(gridsize * nlev2);
          varnmiss[varID].resize(maxlev, 0);
        }
      else
        {
          varnmiss[varID].resize(nlevels);
        }
    }

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      for (varID = 0; varID < nvars; ++varID) vars[varID] = false;

      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          const auto offset = varList1[varID].gridsize * levelID;
          cdoReadRecord(streamID1, &vardata1[varID][offset], &varnmiss[varID][levelID]);
          vars[varID] = true;
        }

      if (tsID == 0 || zvarIsVarying)
        {
          if (zvarname)
            {
              for (levelID = 0; levelID < nlev1; ++levelID)
                {
                  const auto offset1 = zvarGridsize * levelID;
                  const auto offset2 = zvarGridsize * (levelID + 1);
                  arrayCopy(zvarGridsize, &vardata1[zvarID][offset1], &lev1[offset2]);
                }

              vert_init_level_0_and_N(nlev1, zvarGridsize, lev1);
              vert_gen_weights3d1d(expol, nlev1 + 2, zvarGridsize, lev1, nlev2, lev2, lev_idx1, lev_idx2, lev_wgt1, lev_wgt2);
            }
          else
            vert_gen_weights(expol, nlev1 + 2, lev1, nlev2, lev2, lev_idx1, lev_idx2, lev_wgt1, lev_wgt2);
        }

      for (varID = 0; varID < nvars; varID++)
        {
          if (vars[varID] && varinterp[varID])
            {
              const auto missval = varList1[varID].missval;
              const auto gridsize = varList1[varID].gridsize;

              if (zvarname)
                vert_interp_lev3d(gridsize, missval, vardata1[varID], vardata2[varID], nlev2, lev_idx1, lev_idx2, lev_wgt1, lev_wgt2);
              else
                vert_interp_lev(gridsize, missval, vardata1[varID], vardata2[varID], nlev2, lev_idx1, lev_idx2, lev_wgt1, lev_wgt2);

              for (levelID = 0; levelID < nlev2; levelID++)
                {
                  const auto offset = gridsize * levelID;
                  varnmiss[varID][levelID] = arrayNumMV(gridsize, &vardata2[varID][offset], missval);
                }
            }
        }

      for (varID = 0; varID < nvars; varID++)
        {
          if (vars[varID])
            {
              for (levelID = 0; levelID < varList2[varID].nlevels; levelID++)
                {
                  const auto offset = varList2[varID].gridsize * levelID;
                  auto single = varinterp[varID] ? &vardata2[varID][offset] : &vardata1[varID][offset];
                  cdoDefRecord(streamID2, varID, levelID);
                  cdoWriteRecord(streamID2, single, varnmiss[varID][levelID]);
                }
            }
        }

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
