/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

*/

#include <cdi.h>

#include "cdo_options.h"
#include "process_int.h"
#include "cdo_cdi_wrapper.h"
#include "cdo_zaxis.h"

void *
Gengrid(void *process)
{
  int varID, levelID;
  size_t nmiss1, nmiss2;
  double missval = 0;

  cdoInitialize(process);

  operatorCheckArgc(0);

  const auto streamID1 = cdoOpenRead(0);
  const auto streamID2 = cdoOpenRead(1);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = cdoStreamInqVlist(streamID2);

  const auto gridID1 = vlistGrid(vlistID1, 0);
  const auto gridID2 = vlistGrid(vlistID2, 0);

  if (gridInqSize(gridID1) != gridInqSize(gridID2)) cdoAbort("Arrays have different grid size!");

  const auto gridsize = gridInqSize(gridID1);
  const auto xsize = gridInqXsize(gridID1);
  const auto ysize = gridInqYsize(gridID1);

  Varray<double> array1(gridsize), array2(gridsize), array3(gridsize);

  cdoStreamInqTimestep(streamID1, 0);
  cdoStreamInqTimestep(streamID2, 0);

  cdoInqRecord(streamID1, &varID, &levelID);
  cdoReadRecord(streamID1, array1.data(), &nmiss1);
  cdoInqRecord(streamID2, &varID, &levelID);
  cdoReadRecord(streamID2, array2.data(), &nmiss2);

  const auto datatype = vlistInqVarDatatype(vlistID1, 0);

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  if (nmiss1 || nmiss2) cdoAbort("Missing values unsupported!");

  const auto gridID3 = gridCreate(GRID_CURVILINEAR, gridsize);

  if (Options::cdoVerbose) cdoPrint("xsize %zu  ysize %zu", xsize, ysize);
  if (xsize * ysize != gridsize) cdoAbort("xsize*ysize != gridsize");

  gridDefXsize(gridID3, xsize);
  gridDefYsize(gridID3, ysize);
  gridDefXvals(gridID3, array1.data());
  gridDefYvals(gridID3, array2.data());

  gridDefDatatype(gridID3, (datatype == CDI_DATATYPE_FLT64) ? CDI_DATATYPE_FLT64 : CDI_DATATYPE_FLT32);

  const auto xmm = varrayMinMax(array1);
  const auto ymm = varrayMinMax(array2);

  if (Options::cdoVerbose)
    cdoPrint("xminval = %g, xmaxval = %g, yminval = %g, ymaxval = %g", xmm.min, xmm.max, ymm.min, ymm.max);

  /* check units */
  if (xmm.min > -4 && xmm.max < 8 && ymm.min > -2 && ymm.max < 2)
    {
      gridDefXunits(gridID3, "radians");
      gridDefYunits(gridID3, "radians");
    }
  else if (xmm.min > -181 && xmm.max < 361 && ymm.min > -91 && ymm.max < 91)
    {
      /* default is degrees */
    }
  else
    {
      cdoAbort("Units undefined!");
    }

  const auto zaxisID3 = zaxisFromName("surface");

  const auto vlistID3 = vlistCreate();
  vlistDefVar(vlistID3, gridID3, zaxisID3, TIME_CONSTANT);
  vlistDefVarMissval(vlistID3, 0, missval);
  vlistDefVarName(vlistID3, 0, "dummy");
  vlistDefVarDatatype(vlistID3, 0, CDI_DATATYPE_INT8);

  const auto taxisID3 = cdoTaxisCreate(TAXIS_ABSOLUTE);
  vlistDefTaxis(vlistID3, taxisID3);

  const auto streamID3 = cdoOpenWrite(2);

  cdoDefVlist(streamID3, vlistID3);

  int tsID = 0;
  cdoDefTimestep(streamID3, tsID);

  for (size_t i = 0; i < gridsize; ++i) array3[i] = missval;

  cdoDefRecord(streamID3, 0, 0);
  cdoWriteRecord(streamID3, array3.data(), gridsize);

  cdoStreamClose(streamID3);

  cdoFinish();

  return nullptr;
}
