/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Detrend    detrend         Detrend
*/

#include <cdi.h>

#include "array.h"
#include "process_int.h"
#include "cdo_vlist.h"
#include "cdo_options.h"
#include "datetime.h"
#include "cimdOmp.h"
#include "pmlist.h"
#include "param_conversion.h"

#define NALLOC_INC 1024

static void
detrend(const long nts, const Varray<double> &deltaTS0, const double missval1, const Varray<double> &array1, Varray<double> &array2)
{
  const auto missval2 = missval1;
  double sumj = 0, sumjj = 0;
  double sumx = 0, sumjx = 0;
  long n = 0;
  for (long j = 0; j < nts; j++)
    if (!DBL_IS_EQUAL(array1[j], missval1))
      {
        const auto zj = deltaTS0[j];
        sumj += zj;
        sumjj += zj * zj;
        sumjx += zj * array1[j];
        sumx += array1[j];
        n++;
      }

  const auto work1 = DIVMN(SUBMN(sumjx, DIVMN(MULMN(sumj, sumx), n)), SUBMN(sumjj, DIVMN(MULMN(sumj, sumj), n)));
  const auto work2 = SUBMN(DIVMN(sumx, n), MULMN(DIVMN(sumj, n), work1));

  for (long j = 0; j < nts; j++) array2[j] = SUBMN(array1[j], ADDMN(work2, MULMN(j, work1)));
}

static void
computeDeltaTS0(bool tstepIsEqual, int nts, int calendar, DateTimeList &dtlist, Varray<double> &deltaTS0)
{
  CheckTimeInc checkTimeInc;
  JulianDate juldate0;
  double deltat1 = 0;

  for (int tsID = 0; tsID < nts; ++tsID)
    {
      const auto vdate = dtlist.getVdate(tsID);
      const auto vtime = dtlist.getVtime(tsID);
      if (tstepIsEqual) checkTimeIncrement(tsID, calendar, vdate, vtime, checkTimeInc);
      deltaTS0[tsID] = tstepIsEqual ? (double) tsID : deltaTimeStep0(tsID, calendar, vdate, vtime, juldate0, deltat1);
    }
}

static void
detrendGetParameter(bool &tstepIsEqual)
{
  const auto pargc = operatorArgc();
  if (pargc)
    {
      const auto pargv = cdoGetOperArgv();

      KVList kvlist;
      kvlist.name = "TREND";
      if (kvlist.parseArguments(pargc, pargv) != 0) cdoAbort("Parse error!");
      if (Options::cdoVerbose) kvlist.print();

      for (const auto &kv : kvlist)
        {
          const auto &key = kv.key;
          if (kv.nvalues > 1) cdoAbort("Too many values for parameter key >%s<!", key.c_str());
          if (kv.nvalues < 1) cdoAbort("Missing value for parameter key >%s<!", key.c_str());
          const auto &value = kv.values[0];

          // clang-format off
          if      (key == "equal")  tstepIsEqual = parameter2bool(value);
          else cdoAbort("Invalid parameter key >%s<!", key.c_str());
          // clang-format on
        }
    }
}

void *
Detrend(void *process)
{
  int nrecs;
  int varID, levelID;
  size_t nmiss;
  DateTimeList dtlist;

  cdoInitialize(process);

  auto tstepIsEqual = true;
  detrendGetParameter(tstepIsEqual);

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  VarList varList;
  varListInit(varList, vlistID1);

  const auto nvars = vlistNvars(vlistID1);
  FieldVector3D vars;

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      if ((size_t) tsID >= vars.size()) vars.resize(vars.size() + NALLOC_INC);

      dtlist.taxisInqTimestep(taxisID1, tsID);

      fieldsFromVlist(vlistID1, vars[tsID]);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          vars[tsID][varID][levelID].resize(varList[varID].gridsize);
          cdoReadRecord(streamID1, vars[tsID][varID][levelID].vec_d.data(), &nmiss);
          vars[tsID][varID][levelID].nmiss = nmiss;
        }

      tsID++;
    }

  const auto nts = tsID;
  Varray<double> deltaTS0(nts);
  VARRAY_2D(double, array1, Threading::ompNumThreads, nts);
  VARRAY_2D(double, array2, Threading::ompNumThreads, nts);

  const auto calendar = taxisInqCalendar(taxisID1);
  computeDeltaTS0(tstepIsEqual, nts, calendar, dtlist, deltaTS0);

  for (varID = 0; varID < nvars; varID++)
    {
      auto nsteps = (varList[varID].timetype == TIME_CONSTANT) ? 1 : nts;
      auto missval = varList[varID].missval;
      auto gridsize = varList[varID].gridsize;
      const auto nlevels = varList[varID].nlevels;
      for (levelID = 0; levelID < nlevels; levelID++)
        {
#ifdef _OPENMP
#pragma omp parallel for default(none) shared(gridsize, nsteps, deltaTS0, missval, array1, array2, vars, varID, levelID)
#endif
          for (size_t i = 0; i < gridsize; i++)
            {
              const auto ompthID = cdo_omp_get_thread_num();

              for (int k = 0; k < nsteps; ++k) array1[ompthID][k] = vars[k][varID][levelID].vec_d[i];

              detrend(nsteps, deltaTS0, missval, array1[ompthID], array2[ompthID]);

              for (int k = 0; k < nsteps; ++k) vars[k][varID][levelID].vec_d[i] = array2[ompthID][k];
            }
        }
    }

  for (tsID = 0; tsID < nts; tsID++)
    {
      dtlist.taxisDefTimestep(taxisID2, tsID);
      cdoDefTimestep(streamID2, tsID);

      for (varID = 0; varID < nvars; varID++)
        {
          if (tsID && varList[varID].timetype == TIME_CONSTANT) continue;
          const auto nlevels = varList[varID].nlevels;
          for (levelID = 0; levelID < nlevels; levelID++)
            {
              nmiss = vars[tsID][varID][levelID].nmiss;
              cdoDefRecord(streamID2, varID, levelID);
              cdoWriteRecord(streamID2, vars[tsID][varID][levelID].vec_d.data(), nmiss);
            }
        }
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
