/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Derivepar     gheight          geopotential height
*/

#include <cdi.h>

#include "process_int.h"
#include "cdo_vlist.h"
#include "vertical_interp.h"
#include "stdnametable.h"
#include "util_string.h"
#include "const.h"
#include "cdo_zaxis.h"
#include "cdo_options.h"

void MakeGeopotHeight(double *geop, double *gt, double *gq, double *ph, int nhor, int nlev);

void vlist_hybrid_vct(int vlistID, int *rzaxisIDh, int *rnvct, Varray<double> &vct, int *rnhlevf);

void *
Derivepar(void *process)
{
  ModelMode mode(ModelMode::UNDEF);
  int nrecs;
  size_t offset;
  int varID, levelID;
  int surfaceID = -1;
  int sgeopotID = -1, geopotID = -1, tempID = -1, humID = -1, psID = -1, lnpsID = -1, presID = -1, gheightID = -1;
  int lnpsID2 = -1;
  // int clwcID = -1, ciwcID = -1;
  char paramstr[32];
  char varname[CDI_MAX_NAME], stdname[CDI_MAX_NAME];
  double *single2;
  // double *lwater = nullptr, *iwater = nullptr;
  size_t nmiss, nmissout = 0;
  gribcode_t gribcodes;
  memset(&gribcodes, 0, sizeof(gribcode_t));

  cdoInitialize(process);

  // clang-format off
  const auto GHEIGHT          = cdoOperatorAdd("gheight",            0, 0, nullptr);
  const auto SEALEVELPRESSURE = cdoOperatorAdd("sealevelpressure",   0, 0, nullptr);
  // clang-format on

  const auto operatorID = cdoOperatorID();

  operatorCheckArgc(0);

  const auto streamID1 = cdoOpenRead(0);
  const auto vlistID1 = cdoStreamInqVlist(streamID1);

  auto gridID0 = vlistGrid(vlistID1, 0);
  if (gridInqType(gridID0) == GRID_SPECTRAL) cdoAbort("Spectral data unsupported!");

  auto gridsize = vlist_check_gridsize(vlistID1);

  int zaxisIDh = -1;
  int nvct = 0;
  int nhlevf = 0;
  Varray<double> vct;
  vlist_hybrid_vct(vlistID1, &zaxisIDh, &nvct, vct, &nhlevf);

  if (Options::cdoVerbose)
    for (int i = 0; i < nvct / 2; ++i) cdoPrint("vct: %5d %25.17f %25.17f", i, vct[i], vct[nvct / 2 + i]);

  if (zaxisIDh == -1) cdoAbort("No 3D variable with hybrid sigma pressure coordinate found!");

  const auto nvars = vlistNvars(vlistID1);

  auto useTable = false;
  for (varID = 0; varID < nvars; varID++)
    {
      int tableNum = tableInqNum(vlistInqVarTable(vlistID1, varID));
      if (tableNum > 0 && tableNum != 255)
        {
          useTable = true;
          break;
        }
    }

  if (Options::cdoVerbose && useTable) cdoPrint("Using code tables!");

  for (varID = 0; varID < nvars; varID++)
    {
      const auto gridID = vlistInqVarGrid(vlistID1, varID);
      const auto zaxisID = vlistInqVarZaxis(vlistID1, varID);
      const auto nlevel = zaxisInqSize(zaxisID);
      const auto instNum = institutInqCenter(vlistInqVarInstitut(vlistID1, varID));
      const auto tableNum = tableInqNum(vlistInqVarTable(vlistID1, varID));

      auto code = vlistInqVarCode(vlistID1, varID);
      auto param = vlistInqVarParam(vlistID1, varID);

      cdiParamToString(param, paramstr, sizeof(paramstr));
      int pnum, pcat, pdis;
      cdiDecodeParam(param, &pnum, &pcat, &pdis);
      if (pdis >= 0 && pdis < 255) code = -1;

      if (useTable)
        {
          if (tableNum == 2)
            {
              mode = ModelMode::WMO;
              wmo_gribcodes(&gribcodes);
            }
          else if (tableNum == 128 || tableNum == 0)
            {
              mode = ModelMode::ECHAM;
              echam_gribcodes(&gribcodes);
            }
          //  KNMI: HIRLAM model version 7.2 uses tableNum=1    (LAMH_D11*)
          //  KNMI: HARMONIE model version 36 uses tableNum=1   (grib*)
          //  (opreational NWP version) KNMI: HARMONIE model version 38 uses
          //  tableNum=253 (grib,grib_md) and tableNum=1 (grib_sfx) (research version)
          else if (tableNum == 1 || tableNum == 253)
            {
              mode = ModelMode::HIRLAM;
              hirlam_harmonie_gribcodes(&gribcodes);
            }
        }
      else
        {
          mode = ModelMode::ECHAM;
          echam_gribcodes(&gribcodes);
        }

      if (Options::cdoVerbose) cdoPrint("Mode = %d  Center = %d  Param = %s", static_cast<int>(mode), instNum, paramstr);

      if (code <= 0 || code == 255)
        {
          vlistInqVarName(vlistID1, varID, varname);
          cstrToLowerCase(varname);

          vlistInqVarStdname(vlistID1, varID, stdname);
          cstrToLowerCase(stdname);

          code = echamcode_from_stdname(stdname);
          if (code < 0)
            {
              // clang-format off
              if      (sgeopotID == -1 && cstrIsEqual(varname, "geosp")) code = gribcodes.geopot;
              else if (psID == -1 && cstrIsEqual(varname, "aps")) code = gribcodes.ps;
              else if (psID == -1 && cstrIsEqual(varname, "ps")) code = gribcodes.ps;
              else if (lnpsID == -1 && cstrIsEqual(varname, "lsp")) code = gribcodes.lsp;
              else if (lnpsID2 == -1 && cstrIsEqual(varname, "lnps")) code = 777;
              else if (tempID == -1 && cstrIsEqual(varname, "t")) code = gribcodes.temp;
              else if (humID == -1 && cstrIsEqual(varname, "q")) code = gribcodes.hum;
              // else if ( geopotID  == -1 && cstrIsEqual(stdname, "geopotential_full")) code = gribcodes.geopot;
              // else if (cstrIsEqual(varname, "clwc")) code = 246;
              // else if (cstrIsEqual(varname, "ciwc")) code = 247;
              // clang-format on
            }
        }

      // clang-format off
      if      (code == gribcodes.geopot && nlevel == 1) sgeopotID = varID;
      else if (code == gribcodes.geopot && nlevel == nhlevf) geopotID = varID;
      else if (code == gribcodes.temp && nlevel == nhlevf) tempID = varID;
      else if (code == gribcodes.hum && nlevel == nhlevf) humID = varID;
      else if (code == gribcodes.ps && nlevel == 1) psID = varID;
      else if (code == gribcodes.lsp && nlevel == 1) lnpsID = varID;
      else if (code == 777 && nlevel == 1) lnpsID2 = varID;
      else if (code == gribcodes.gheight && nlevel == nhlevf) gheightID = varID;
      // else if (code == 246) clwcID    = varID;
      // else if (code == 247) ciwcID    = varID;
      // clang-format on

      if (operatorID == SEALEVELPRESSURE) humID = -1;

      if (gridInqType(gridID) == GRID_SPECTRAL && zaxisInqType(zaxisID) == ZAXIS_HYBRID)
        cdoAbort("Spectral data on model level unsupported!");

      if (gridInqType(gridID) == GRID_SPECTRAL) cdoAbort("Spectral data unsupported!");
    }

  if (Options::cdoVerbose)
    {
      cdoPrint("Found:");
      if (tempID != -1) cdoPrint("  %s", var_stdname(air_temperature));
      if (psID != -1) cdoPrint("  %s", var_stdname(surface_air_pressure));
      if (lnpsID != -1) cdoPrint("  LOG(%s)", var_stdname(surface_air_pressure));
      if (sgeopotID != -1) cdoPrint("  %s", var_stdname(surface_geopotential));
      if (geopotID != -1) cdoPrint("  %s", var_stdname(geopotential));
      if (gheightID != -1) cdoPrint("  %s", var_stdname(geopotential_height));
    }

  if (lnpsID != -1 && lnpsID2 != -1) cdoAbort("Found LOG(%s) twice: lsp and lnps!", var_stdname(surface_air_pressure));

  if (tempID == -1) cdoAbort("%s not found!", var_stdname(air_temperature));

  Varray<double> array(gridsize);
  Varray<double> sgeopot(gridsize), ps(gridsize);
  Varray<double> temp(gridsize * nhlevf);
  Varray<double> half_press(gridsize * (nhlevf + 1));
  Varray<double> hum, gheight;
  if (operatorID == GHEIGHT)
    {
      if (humID == -1)
        cdoWarning("%s not found - using algorithm without %s!", var_stdname(specific_humidity), var_stdname(specific_humidity));
      else
        hum.resize(gridsize * nhlevf);

      gheight.resize(gridsize * (nhlevf + 1));
    }

  Varray<double> full_press, sealevelpressure;
  if (operatorID == SEALEVELPRESSURE)
    {
      full_press.resize(gridsize * nhlevf);

      surfaceID = zaxisFromName("surface");
      sealevelpressure.resize(gridsize);
    }

  if (zaxisIDh != -1 && sgeopotID == -1)
    {
      if (geopotID == -1)
        cdoWarning("%s not found - set to zero!", var_stdname(surface_geopotential));
      else
        cdoPrint("%s not found - using bottom layer of %s!", var_stdname(surface_geopotential), var_stdname(geopotential));

      varrayFill(sgeopot, 0.0);
    }

  presID = lnpsID;
  if (zaxisIDh != -1 && lnpsID == -1)
    {
      if (psID == -1)
        cdoAbort("%s not found!", var_stdname(surface_air_pressure));
      else
        presID = psID;
    }

  if (Options::cdoVerbose)
    {
      if (presID == lnpsID)
        cdoPrint("using LOG(%s)", var_stdname(surface_air_pressure));
      else
        cdoPrint("using %s", var_stdname(surface_air_pressure));
    }

  const auto vlistID2 = vlistCreate();

  int var_id = -1;

  if (operatorID == GHEIGHT)
    {
      var_id = geopotential_height;
      varID = vlistDefVar(vlistID2, gridID0, zaxisIDh, TIME_VARYING);
    }
  else if (operatorID == SEALEVELPRESSURE)
    {
      var_id = air_pressure_at_sea_level;
      varID = vlistDefVar(vlistID2, gridID0, surfaceID, TIME_VARYING);
    }
  else
    cdoAbort("Internal problem, invalid operatorID: %d!", operatorID);

  vlistDefVarParam(vlistID2, varID, cdiEncodeParam(var_echamcode(var_id), 128, 255));
  vlistDefVarName(vlistID2, varID, var_name(var_id));
  vlistDefVarStdname(vlistID2, varID, var_stdname(var_id));
  vlistDefVarUnits(vlistID2, varID, var_units(var_id));

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          offset = gridsize * levelID;
          cdoReadRecord(streamID1, array.data(), &nmiss);

          if (zaxisIDh != -1)
            {
              if (varID == sgeopotID)
                {
                  varrayCopy(gridsize, array, sgeopot);
                }
              else if (varID == geopotID && sgeopotID == -1 && (levelID + 1) == nhlevf)
                {
                  varrayCopy(gridsize, array, sgeopot);
                }
              else if (varID == presID)
                {
                  if (lnpsID != -1)
                    for (size_t i = 0; i < gridsize; ++i) ps[i] = std::exp(array[i]);
                  else if (psID != -1)
                    varrayCopy(gridsize, array, ps);
                }
              else if (varID == tempID)
                arrayCopy(gridsize, array.data(), &temp[offset]);
              else if (varID == humID)
                arrayCopy(gridsize, array.data(), &hum[offset]);
            }
        }

      if (zaxisIDh != -1)
        {
          // check range of ps_prog
          auto mm = varrayMinMax(ps);
          if (mm.min < MIN_PS || mm.max > MAX_PS)
            cdoWarning("Surface pressure out of range (min=%g max=%g)!", mm.min, mm.max);

          // check range of surface geopot
          mm = varrayMinMax(sgeopot);
          if (mm.min < MIN_FIS || mm.max > MAX_FIS)
            cdoWarning("Orography out of range (min=%g max=%g)!", mm.min, mm.max);
        }

      {
        varID = tempID;
        const auto nlevel = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));
        for (levelID = 0; levelID < nlevel; levelID++)
          {
            offset = gridsize * levelID;
            single2 = &temp[offset];

            const auto mm = varrayMinMax(gridsize, single2);
            if (mm.min < MIN_T || mm.max > MAX_T)
              cdoWarning("Input temperature at level %d out of range (min=%g max=%g)!", levelID + 1, mm.min, mm.max);
          }
      }

      if (humID != -1)
        {
          varID = humID;
          const auto nlevel = zaxisInqSize(vlistInqVarZaxis(vlistID1, varID));
          for (levelID = 0; levelID < nlevel; levelID++)
            {
              offset = gridsize * levelID;
              single2 = &hum[offset];

              // corr_hum(gridsize, single2, MIN_Q);

              const auto mm = varrayMinMax(gridsize, single2);
              if (mm.min < -0.1 || mm.max > MAX_Q)
                cdoWarning("Input humidity at level %d out of range (min=%g max=%g)!", levelID + 1, mm.min, mm.max);
            }
        }

      if (operatorID == GHEIGHT)
        {
          presh((double*)nullptr, half_press.data(), vct.data(), ps.data(), nhlevf, gridsize);
          arrayCopy(gridsize, sgeopot.data(), gheight.data() + gridsize * nhlevf);
          MakeGeopotHeight(gheight.data(), temp.data(), hum.data(), half_press.data(), gridsize, nhlevf);

          nmissout = 0;
          varID = 0;
          const auto nlevel = nhlevf;
          for (levelID = 0; levelID < nlevel; levelID++)
            {
              cdoDefRecord(streamID2, varID, levelID);
              cdoWriteRecord(streamID2, gheight.data() + levelID * gridsize, nmissout);
            }
        }
      else if (operatorID == SEALEVELPRESSURE)
        {
          presh(full_press.data(), half_press.data(), vct.data(), ps.data(), nhlevf, gridsize);

          extrapolate_P(sealevelpressure.data(), &half_press[gridsize * (nhlevf)], &full_press[gridsize * (nhlevf - 1)], sgeopot.data(),
                  &temp[gridsize * (nhlevf - 1)], gridsize);

          cdoDefRecord(streamID2, 0, 0);
          cdoWriteRecord(streamID2, sealevelpressure.data(), 0);
        }
      else
        cdoAbort("Internal error");

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  vlistDestroy(vlistID2);

  cdoFinish();

  return nullptr;
}
