/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include <string.h>
#include <stdlib.h>
#include <vector>

#include "pmlist.h"
#include "namelist.h"
#include "cdo_output.h"


static void
keyValuesPrint(const KeyValues &keyValues)
{
  printf("  %s =", keyValues.key.c_str());
  for (int i = 0; i < keyValues.nvalues; ++i) printf(" '%s'", keyValues.values[i].c_str());
  printf("\n");
}

void
KVList::print() const
{
  for (const auto &keyValues : *this) keyValuesPrint(keyValues);
}

int
KVList::parseArguments(const int argc, const char * const *argv)
{
  // Assume key = value pairs. That is, if argv[i] contains no '=' then treat it as if it belongs to the values of argv[i-1].
  char key[256];
  int i = 0;
  while (i < argc)
    {
      const char *end = strchr(argv[i], '=');
      if (end == nullptr)
        {
          fprintf(stderr, "Missing '=' in key/value string: >%s<\n", argv[i]);
          return -1;
        }

      snprintf(key, sizeof(key), "%.*s", (int) (end - argv[i]), argv[i]);
      key[sizeof(key) - 1] = 0;

      int j = 1;
      while (i + j < argc && strchr(argv[i + j], '=') == nullptr) j++;

      int nvalues = j;

      KeyValues kv;
      kv.values.resize(1);
      kv.values[0] = end + 1;
      if (kv.values[0][0] == 0) nvalues = 0;

      kv.key = key;
      kv.nvalues = nvalues;
      kv.values.resize(nvalues);

      for (j = 1; j < nvalues; ++j) kv.values[j] = argv[i + j];
      this->push_back(kv);

      i += j;
    }

  return 0;
}

const KeyValues *
KVList::search(const std::string &key) const
{
  for (const auto &kv : *this)
    {
      if (kv.key == key) return &kv;
    }

  return nullptr;
}

void
KVList::append(const char *key, const char * const *values, int nvalues)
{
  KeyValues kv;
  kv.key = strdup(key);
  kv.nvalues = nvalues;
  kv.values.resize(nvalues);
  for (int i = 0; i < nvalues; ++i) kv.values[i] = values[i];
  this->push_back(kv);
}

void
KVList::remove(const KeyValues *kv)
{
  std::list<KeyValues>::iterator i;
  for (i=this->begin(); i->key!=kv->key; i++) ;
  this->erase(i);
}

const KVList *
PMList::searchKVListVentry(const std::string &key, const std::string &value, const std::vector<std::string> &entry)
{
  for (const auto &kvlist : *this)
    {
      for (const auto &s : entry)
        if (kvlist.name == s)
          {
             const KeyValues *kv = kvlist.search(key);
            if (kv && kv->nvalues > 0 && kv->values[0] == value) return &kvlist;
          }
    }

  return nullptr;
}

const KVList *
PMList::getKVListVentry(const std::vector<std::string> &entry)
{
  for (const auto &kvlist : *this)
    {
      for (const auto &s : entry)
        if (kvlist.name == s) return &kvlist;
    }

  return nullptr;
}

static void
KVListAppendNamelist(KVList &kvlist, const char *key, const char *buffer, NamelistToken *t, int nvalues)
{
  char vbuf[4096];
  std::vector<char> value;
  KeyValues kv;
  kv.key = key;
  kv.nvalues = nvalues;
  if (nvalues > 0) kv.values.resize(nvalues);

  for (int i = 0; i < nvalues; ++i)
    {
      const size_t len = t[i].end - t[i].start;
      value.resize(len + 1);
      // printf(" value[%d] >%.*s<\n", i, (int)len, buffer+t[i].start);
      const char *pval = buffer + t[i].start;
      if (len < sizeof(vbuf))  // snprintf seems to call strlen(pval)
        {
          memcpy(vbuf, buffer + t[i].start, len);
          vbuf[len] = 0;
          pval = vbuf;
        }
      snprintf(value.data(), len + 1, "%.*s", (int) len, pval);
      value[len] = 0;
      kv.values[i] = value.data();
    }

  kvlist.push_back(kv);
}

static int
getNumberOfValues(const int ntok, const NamelistToken *tokens)
{
  int it;

  for (it = 0; it < ntok; ++it)
    {
      auto type = tokens[it].type;
      if (type != NamelistType::WORD && type != NamelistType::STRING) break;
    }

  return it;
}

static int
parseNamelist(PMList &pmlist, NamelistParser &parser, char *buf)
{
  char name[4096];
  KVList kvlist;
  auto &tokens = parser.tokens;
  const int ntok = parser.toknext;
  // printf("Number of tokens %d\n", ntok);

  for (int it = 0; it < ntok; ++it)
    {
      const auto &t = tokens[it];
      // printf("Token %u", it+1);
      if (t.type == NamelistType::OBJECT)
        {
          name[0] = 0;
          if (it + 1 < ntok && tokens[it + 1].type == NamelistType::WORD)
            {
              it++;
              const auto &t = tokens[it];
              snprintf(name, sizeof(name), "%.*s", t.end - t.start, buf + t.start);
              name[sizeof(name) - 1] = 0;
            }

          if (kvlist.size())
            {
              pmlist.push_back(kvlist);
              kvlist.clear();
            }

          kvlist.name = name;
        }
      else if (t.type == NamelistType::KEY)
        {
          // printf(" key >%.*s<\n", t.end - t.start, buf+t.start);
          snprintf(name, sizeof(name), "%.*s", t.end - t.start, buf + t.start);
          name[sizeof(name) - 1] = 0;

          const auto nvalues = getNumberOfValues(ntok - it - 1, &tokens[it + 1]);
          // printf("nvalues %d\n", nvalues);
          KVListAppendNamelist(kvlist, name, buf, &tokens[it + 1], nvalues);
          it += nvalues;
        }
      else
        {
          // printf(" token >%.*s<\n", t.end - t.start, buf+t.start);
          break;
        }
    }

  if (kvlist.size()) pmlist.push_back(kvlist);

  return 0;
}

int
parseListBuffer(PMList &pmlist, ListBuffer &listBuffer)
{
  const char *errMsg = "Namelist error";
  const auto name = listBuffer.name.c_str();

  NamelistParser p;
  const auto status = p.parse(listBuffer.buffer.data(), listBuffer.buffer.size());
  if (status != NamelistError::UNDEFINED)
    {
      switch (status)
        {
        case NamelistError::INVAL:
          fprintf(stderr, "%s: Invalid character in %s (line=%d character='%c')!\n", errMsg, name, p.lineno,
                  listBuffer.buffer[p.pos]);
          break;
        case NamelistError::PART: fprintf(stderr, "%s: End of string not found in %s (line=%d)!\n", errMsg, name, p.lineno); break;
        case NamelistError::INKEY: fprintf(stderr, "%s: Invalid key word in %s (line=%d)!\n", errMsg, name, p.lineno); break;
        case NamelistError::INTYP: fprintf(stderr, "%s: Invalid key word type in %s (line=%d)!\n", errMsg, name, p.lineno); break;
        case NamelistError::INOBJ: fprintf(stderr, "%s: Invalid object in %s (line=%d)!\n", errMsg, name, p.lineno); break;
        case NamelistError::EMKEY: fprintf(stderr, "%s: Emtry key name in %s (line=%d)!\n", errMsg, name, p.lineno); break;
        default: fprintf(stderr, "%s in %s (line=%d)!\n", errMsg, name, p.lineno); break;
        }
      cdoAbort("%s!", errMsg);
    }

  // p.dump(listBuffer.buffer.data());
  if (p.verify())
    {
      fprintf(stderr, "%s: Invalid contents in %s!\n", errMsg, name);
      cdoAbort("Namelist error!");
    }

  parseNamelist(pmlist, p, listBuffer.buffer.data());

  return 0;
}

void
PMList::readNamelist(FILE *fp, const char *name)
{
  ListBuffer listBuffer;
  if (listBuffer.read(fp, name)) cdoAbort("Read error on namelist %s!", name);

  const auto status = parseListBuffer(*this, listBuffer);
  if (status) cdoAbort("Namelist not found!");
}

void
PMList::print()
{
  for (const auto &kvlist : *this)
    {
      const auto listname = kvlist.name.c_str();
      printf("\nFound %s list with %zu key/values: \n", listname ? listname : "", kvlist.size());
      kvlist.print();
    }
}
