#include "param_conversion.h"

#include <inttypes.h>
#include <cstdlib>
#include <cstddef>
#include <ctype.h>

#include "cdo_output.h"
#include "compare.h"
#include "util_string.h"
#include "constants.h"
#include "const.h"


const char *
parameter2word(const char *cstring)
{
  const size_t len = strlen(cstring);

  for (size_t i = 0; i < len; ++i)
    {
      int c = cstring[i];
      if (iscntrl(c) || isblank(c)) cdoAbort("Word parameter >%s< contains invalid character at position %zu!", cstring, i + 1);
    }

  if (len == 0) cdoAbort("Word parameter >%s< is empty!", cstring);

  return cstring;
}

static inline void
parameterError(const char *name, const char *cstring, const char *endptr)
{
  cdoAbort("%s parameter >%s< contains invalid character at position %d!", name, cstring, (int) (endptr - cstring + 1));
}

double
parameter2double(const char *cstring)
{
  char *endptr = nullptr;
  const double fval = strtod(cstring, &endptr);
  if (*endptr) parameterError("Float", cstring, endptr);
  return fval;
}

int
parameter2int(const char *cstring)
{
  char *endptr = nullptr;
  const int ival = (int) strtol(cstring, &endptr, 10);
  if (*endptr) parameterError("Integer", cstring, endptr);
  return ival;
}

long
parameter2long(const char *cstring)
{
  char *endptr = nullptr;
  const long ival = strtol(cstring, &endptr, 10);
  if (*endptr) parameterError("Integer", cstring, endptr);
  return ival;
}

size_t
parameter2sizet(const char *cstring)
{
  char *endptr = nullptr;
  const size_t ival = (size_t) strtoimax(cstring, &endptr, 10);
  if (*endptr) parameterError("Integer", cstring, endptr);
  return ival;
}

int
parameter2intlist(const char *cstring)
{
  char *endptr = nullptr;
  const int ival = (int) strtol(cstring, &endptr, 10);
  if (*endptr && *endptr != '/' && (endptr - cstring) == 0) parameterError("Integer", cstring, endptr);
  return ival;
}


const std::string &
parameter2word(const std::string &string)
{
  const size_t len = string.size();

  for (size_t i = 0; i < len; ++i)
    {
      int c = string[i];
      if (iscntrl(c) || isblank(c)) cdoAbort("Word parameter >%s< contains invalid character at position %zu!", string.c_str(), i + 1);
    }

  if (len == 0) cdoAbort("Word parameter >%s< is empty!", string.c_str());

  return string;
}

bool
parameter2bool(const std::string &string)
{
  const auto lstring = stringToLower(string);

  if (lstring == "1" || lstring == "t" || lstring == "true") return true;
  if (lstring == "0" || lstring == "f" || lstring == "false") return false;

  cdoAbort("Boolean parameter >%s< contains invalid characters!", string.c_str());

  return false;

}

double
parameter2double(const std::string &string)
{
  return parameter2double(string.c_str());
}

int
parameter2int(const std::string &string)
{
  return parameter2int(string.c_str());
}

long
parameter2long(const std::string &string)
{
  return parameter2long(string.c_str());
}

size_t
parameter2sizet(const std::string &string)
{
  return parameter2sizet(string.c_str());
}

int
parameter2intlist(const std::string &string)
{
  return parameter2intlist(string.c_str());
}

double
radius_str_to_deg(const char *string)
{
  char *endptr = nullptr;
  auto radius = strtod(string, &endptr);

  if (*endptr != 0)
    {
      if (strncmp(endptr, "km", 2) == 0)
        radius = 360 * ((radius * 1000) / (2 * PlanetRadius * M_PI));
      else if (strncmp(endptr, "m", 1) == 0)
        radius = 360 * ((radius) / (2 * PlanetRadius * M_PI));
      else if (strncmp(endptr, "deg", 3) == 0)
        ;
      else if (strncmp(endptr, "rad", 3) == 0)
        radius *= RAD2DEG;
      else
        cdoAbort("Float parameter >%s< contains invalid character at position %d!", string, (int) (endptr - string + 1));
    }

  if (radius > 180.) radius = 180.;

  return radius;
}
