/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/
#include <sys/stat.h> /* stat */
#include <cdi.h>

#include "fileStream.h"

#include "cdi_lockedIO.h"
#include "cdo_options.h"
#include "cdo_output.h"
#include "cdo_defaultValues.h"
#include "cdo_history.h"

#include "timer.h"
#include "commandline.h"

extern int processNum;

bool FileStream::TimerEnabled = false;

FileStream::FileStream(std::string p_fileName) : m_filename(p_fileName)
{
  m_name = p_fileName;
  m_fileID = CDI_UNDEFID;
}

int
FileStream::openRead()
{
  openLock();

  CdiStreamID fileID = streamOpenRead(m_filename.c_str());
  if (fileID < 0) cdiOpenError(fileID, "Open failed on >%s<", m_filename.c_str());
  isopen = true;

  m_filetype = streamInqFiletype(fileID);
  if (CdoDefault::FileType == CDI_UNDEFID) CdoDefault::FileType = m_filetype;
  cdoInqHistory(fileID);
  m_fileID = fileID;

  Debug(FILE_STREAM, "Set number of worker to %d", Options::numStreamWorker);
  if (Options::numStreamWorker > 0) streamDefNumWorker(fileID, Options::numStreamWorker);

  openUnlock();

  return fileID;
}

int
FileStream::openWrite(int p_filetype)
{

  if (Options::cdoInteractive)
    {
      struct stat stbuf;
      const int rstatus = stat(m_name.c_str(), &stbuf);
      /* If permanent file already exists, query user whether to overwrite or exit */
      if (rstatus != -1) query_user_exit(m_name.c_str());
    }
  if (p_filetype == CDI_UNDEFID) p_filetype = CDI_FILETYPE_GRB;

  // TODO FIX THIS: if(FileStream::timersEnabled()) timer_start(timer_write);

  openLock();
  int fileID = streamOpenWrite(m_filename.c_str(), p_filetype);
  openUnlock();

  // TODO FIX THIS: if(FileStream::timersEnabled()) timer_stop(timer_write);
  if (fileID < 0) cdiOpenError(fileID, "Open failed on >%s<", m_name.c_str());
  isopen = true;

  cdoDefHistory(fileID, commandLine());

  if (CdoDefault::Byteorder != CDI_UNDEFID) streamDefByteorder(fileID, CdoDefault::Byteorder);

  set_comp(fileID, p_filetype);

  m_fileID = fileID;
  m_filetype = p_filetype;

  return m_cdoStreamID;
}
int
FileStream::openAppend()
{
  if (FileStream::timersEnabled()) timer_start(timer_write);

  openLock();
  int fileID = streamOpenAppend(m_filename.c_str());
  openUnlock();

  if (FileStream::timersEnabled()) timer_stop(timer_write);

  if (fileID < 0) cdiOpenError(fileID, "Open failed on >%s<", m_filename.c_str());

  isopen = true;

  m_filetype = streamInqFiletype(fileID);
  set_comp(fileID, m_filetype);

  m_fileID = fileID;

  return m_fileID;
}

void
FileStream::defVlist(int p_vlistID)
{
  if (CdoDefault::DataType != CDI_UNDEFID)
    {
      int varID;
      int nvars = vlistNvars(p_vlistID);

      for (varID = 0; varID < nvars; ++varID) vlistDefVarDatatype(p_vlistID, varID, CdoDefault::DataType);

      if (CdoDefault::DataType == CDI_DATATYPE_FLT64 || CdoDefault::DataType == CDI_DATATYPE_FLT32)
        {
          for (varID = 0; varID < nvars; varID++)
            {
              vlistDefVarAddoffset(p_vlistID, varID, 0.0);
              vlistDefVarScalefactor(p_vlistID, varID, 1.0);
            }
        }
    }

  if (Options::cdoChunkType != CDI_UNDEFID)
    {
      const int nvars = vlistNvars(p_vlistID);
      for (int varID = 0; varID < nvars; ++varID) vlistDefVarChunkType(p_vlistID, varID, Options::cdoChunkType);
    }

  if (Options::CMOR_Mode)
    {
      cdo_def_tracking_id(p_vlistID, "tracking_id");
      cdo_def_creation_date(p_vlistID);
    }

  if (Options::VersionInfo) cdiDefAttTxt(p_vlistID, CDI_GLOBAL, "CDO", (int) strlen(cdoComment()), cdoComment());

#ifdef _OPENMP
  if (Threading::ompNumThreads > 1)
    cdiDefAttInt(p_vlistID, CDI_GLOBAL, "cdo_openmp_thread_number", CDI_DATATYPE_INT32, 1, &Threading::ompNumThreads);
#endif
  defVarList(p_vlistID);

  if (FileStream::timersEnabled()) timer_start(timer_write);
  streamDefVlistLocked(m_fileID, p_vlistID);
  if (FileStream::timersEnabled()) timer_stop(timer_write);
}

int
FileStream::inqVlist()
{
  int vlistID;
  if (FileStream::timersEnabled()) timer_start(timer_read);
  vlistID = streamInqVlistLocked(m_fileID);
  if (FileStream::timersEnabled()) timer_stop(timer_read);
  if (vlistID == -1)
    {
      cdoAbort("Couldn't read data from input fileID %d!", m_fileID);
    }

  const int nsubtypes = vlistNsubtypes(vlistID);
  if (nsubtypes > 1) cdoWarning("Subtypes are unsupported, the processing results are possibly wrong!");

  if (CdoDefault::TaxisType != CDI_UNDEFID) taxisDefType(vlistInqTaxis(vlistID), CdoDefault::TaxisType);

  m_vlistID = vlistID;
  return m_vlistID;
}

void
FileStream::inqRecord(int *varID, int *levelID)
{
  if (FileStream::timersEnabled()) timer_start(timer_read);
  streamInqRecLocked(m_fileID, varID, levelID);
  if (FileStream::timersEnabled()) timer_stop(timer_read);
  m_varID = *varID;
}

void
FileStream::defRecord(int varID, int levelID)
{
  if (FileStream::timersEnabled()) timer_start(timer_write);
  streamDefRecLocked(m_fileID, varID, levelID);
  if (FileStream::timersEnabled()) timer_stop(timer_write);
  m_varID = varID;
}

void
FileStream::readRecord(double *data, size_t *nmiss)
{
  if (FileStream::timersEnabled()) timer_start(timer_read);
  streamReadrecordDoubleLocked(m_fileID, data, nmiss);
  if (FileStream::timersEnabled()) timer_stop(timer_read);
}

void
FileStream::readRecord(Field *data, size_t *nmiss)
{
  readRecord(data->vec.data(), nmiss);
}

void
FileStream::readRecord(float *data, size_t *nmiss)
{
  if (FileStream::timersEnabled()) timer_start(timer_read);
  streamReadrecordFloatLocked(m_fileID, data, nmiss);
  if (FileStream::timersEnabled()) timer_stop(timer_read);
}

void
FileStream::writeRecord(double *p_data, size_t p_nmiss)
{
  int varID = m_varID;
  if (FileStream::timersEnabled()) timer_start(timer_write);

  if (varID < (int) m_varlist.size())
    if (m_varlist[varID].check_datarange) m_varlist[varID].checkDatarange(p_data, p_nmiss);

  streamWriteRecordDoubleLocked(m_fileID, p_data, p_nmiss);

  if (FileStream::timersEnabled()) timer_stop(timer_write);
}
void
FileStream::writeRecord(Field *p_data, size_t p_nmiss)
{
  writeRecord(p_data->vec.data(), p_nmiss);
}

void
FileStream::writeRecord(float *p_data, size_t p_nmiss)
{
  if (FileStream::timersEnabled()) timer_start(timer_write);
  streamWriteRecordFloatLocked(m_fileID, p_data, p_nmiss);
  if (FileStream::timersEnabled()) timer_stop(timer_write);
}

void
FileStream::copyRecord(CdoStreamID p_destination)
{
  FileStream *fStream = dynamic_cast<FileStream *>(p_destination.get());
  streamCopyRecordLocked(m_fileID, fStream->getFileID());
}
/*
 * FileStream::inqTimestep(int p_tsID)
 * stets internal state of the cdi datastructure to work on the given timestep (p_tsID) and returns the number of records that the
 * timestep contains.
 * Inquires and defines the time axis type if the timestep ID is 0 AND the taxis type is yet to be defined.
 * When the timestep inquiry was successfull m_tsID is set to the wanted p_tsID IF p_tsID != m_tsID
 * When only one process is running the timers are enabled.
 * -- last Documentation update(2019-06-14) --
 */
int
FileStream::inqTimestep(int p_tsID)
{

  if (FileStream::timersEnabled()) timer_start(timer_read);
  int nrecs = streamInqTimeStepLocked(m_fileID, p_tsID);
  if (FileStream::timersEnabled()) timer_stop(timer_read);

  if (p_tsID == 0 && CdoDefault::TaxisType != CDI_UNDEFID) taxisDefType(vlistInqTaxis(m_vlistID), CdoDefault::TaxisType);

  if (nrecs && p_tsID != m_tsID)
    {
      m_tsID = p_tsID;
    }
  return nrecs;
}

void
FileStream::defTimestep(int p_tsID)
{
  if (FileStream::timersEnabled()) timer_start(timer_write);
  // don't use sync -> very slow on GPFS
  //  if ( p_tsID > 0 ) streamSync(fileID);

  streamDefTimeStepLocked(m_fileID, p_tsID);

  if (FileStream::timersEnabled()) timer_stop(timer_write);
}

int
FileStream::inqFileType()
{
  return streamInqFiletype(m_fileID);
}

int
FileStream::inqByteorder()
{
  return streamInqByteorder(m_fileID);
}

size_t
FileStream::getNvals()
{
  return m_nvals;
}

int
FileStream::getFileID()
{
  return m_fileID;
}

void
FileStream::close()
{
  Debug(FILE_STREAM, "%s fileID %d", m_name, m_fileID);

  streamCloseLocked(m_fileID);
  isopen = false;

  if (m_varlist.size())
    {
      m_varlist.clear();
      m_varlist.shrink_to_fit();
    }
}
