/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include <cdi.h>

#include "functs.h"
#include "process_int.h"
#include "percentiles.h"

static size_t
fillReducedPoints(int gridID, size_t ny, std::vector<int> &reducedPoints, std::vector<int> &cumreducedPoints)
{
  reducedPoints.resize(ny);
  cumreducedPoints.resize(ny);
  gridInqReducedPoints(gridID, reducedPoints.data());
  cumreducedPoints[0] = 0;
  for (size_t j = 1; j < ny; j++) cumreducedPoints[j] = cumreducedPoints[j - 1] + reducedPoints[j - 1];
  size_t nx = reducedPoints[0];
  for (size_t j = 1; j < ny; j++)
    if (reducedPoints[j] > (int) nx) nx = reducedPoints[j];
  return nx;
}

void
zonmin(const Field &field1, Field &field2)
{
  size_t rnmiss = 0;

  const auto ny = gridInqYsize(field1.grid);
  const bool lreduced = gridInqType(field1.grid) == GRID_GAUSSIAN_REDUCED;
  std::vector<int> reducedPoints, cumreducedPoints;
  auto nx = lreduced ? fillReducedPoints(field1.grid, ny, reducedPoints, cumreducedPoints) : gridInqXsize(field1.grid);

  std::vector<double> v(nx);

  for (size_t j = 0; j < ny; ++j)
    {
      if (lreduced) nx = reducedPoints[j];
      const size_t offset = lreduced ? cumreducedPoints[j] : j * nx;
      std::copy(field1.vec.begin() + offset, field1.vec.begin() + offset + nx, v.begin());

      const double rmin = field1.nmiss ? arrayMinMV(nx, v, field1.missval) : arrayMin(nx, v);

      if (DBL_IS_EQUAL(rmin, field1.missval)) rnmiss++;

      field2.vec[j] = rmin;
    }

  field2.nmiss = rnmiss;
}

void
zonmax(const Field &field1, Field &field2)
{
  size_t rnmiss = 0;

  const auto ny = gridInqYsize(field1.grid);
  const bool lreduced = gridInqType(field1.grid) == GRID_GAUSSIAN_REDUCED;
  std::vector<int> reducedPoints, cumreducedPoints;
  auto nx = lreduced ? fillReducedPoints(field1.grid, ny, reducedPoints, cumreducedPoints) : gridInqXsize(field1.grid);

  std::vector<double> v(nx);

  for (size_t j = 0; j < ny; ++j)
    {
      if (lreduced) nx = reducedPoints[j];
      const size_t offset = lreduced ? cumreducedPoints[j] : j * nx;
      std::copy(field1.vec.begin() + offset, field1.vec.begin() + offset + nx, v.begin());

      const double rmax = field1.nmiss ? arrayMaxMV(nx, v, field1.missval) : arrayMax(nx, v);

      if (DBL_IS_EQUAL(rmax, field1.missval)) rnmiss++;

      field2.vec[j] = rmax;
    }

  field2.nmiss = rnmiss;
}

void
zonrange(const Field &field1, Field &field2)
{
  size_t rnmiss = 0;

  const auto ny = gridInqYsize(field1.grid);
  const bool lreduced = gridInqType(field1.grid) == GRID_GAUSSIAN_REDUCED;
  std::vector<int> reducedPoints, cumreducedPoints;
  auto nx = lreduced ? fillReducedPoints(field1.grid, ny, reducedPoints, cumreducedPoints) : gridInqXsize(field1.grid);

  std::vector<double> v(nx);

  for (size_t j = 0; j < ny; ++j)
    {
      if (lreduced) nx = reducedPoints[j];
      const size_t offset = lreduced ? cumreducedPoints[j] : j * nx;
      std::copy(field1.vec.begin() + offset, field1.vec.begin() + offset + nx, v.begin());

      const double range = field1.nmiss ? arrayRangeMV(nx, v, field1.missval) : arrayRange(nx, v);

      if (DBL_IS_EQUAL(range, field1.missval)) rnmiss++;

      field2.vec[j] = range;
    }

  field2.nmiss = rnmiss;
}

void
zonsum(const Field &field1, Field &field2)
{
  size_t rnmiss = 0;

  const auto ny = gridInqYsize(field1.grid);
  const bool lreduced = gridInqType(field1.grid) == GRID_GAUSSIAN_REDUCED;
  std::vector<int> reducedPoints, cumreducedPoints;
  auto nx = lreduced ? fillReducedPoints(field1.grid, ny, reducedPoints, cumreducedPoints) : gridInqXsize(field1.grid);

  std::vector<double> v(nx);

  for (size_t j = 0; j < ny; ++j)
    {
      if (lreduced) nx = reducedPoints[j];
      const size_t offset = lreduced ? cumreducedPoints[j] : j * nx;
      std::copy(field1.vec.begin() + offset, field1.vec.begin() + offset + nx, v.begin());

      const double rsum = field1.nmiss ? arraySumMV(nx, v, field1.missval) : arraySum(nx, v);

      if (DBL_IS_EQUAL(rsum, field1.missval)) rnmiss++;

      for (size_t i = 0; i < nx; i++) v[i] = field1.vec[j * nx + i];

      field2.vec[j] = rsum;
    }

  field2.nmiss = rnmiss;
}

void
zonmean(const Field &field1, Field &field2)
{
  size_t rnmiss = 0;

  const auto ny = gridInqYsize(field1.grid);
  const bool lreduced = gridInqType(field1.grid) == GRID_GAUSSIAN_REDUCED;
  std::vector<int> reducedPoints, cumreducedPoints;
  auto nx = lreduced ? fillReducedPoints(field1.grid, ny, reducedPoints, cumreducedPoints) : gridInqXsize(field1.grid);

  std::vector<double> v(nx);

  for (size_t j = 0; j < ny; ++j)
    {
      if (lreduced) nx = reducedPoints[j];
      const size_t offset = lreduced ? cumreducedPoints[j] : j * nx;
      std::copy(field1.vec.begin() + offset, field1.vec.begin() + offset + nx, v.begin());

      const double rmean = field1.nmiss ? arrayMeanMV(nx, v, field1.missval) : arrayMean(nx, v);

      if (DBL_IS_EQUAL(rmean, field1.missval)) rnmiss++;

      field2.vec[j] = rmean;
    }

  field2.nmiss = rnmiss;
}

void
zonavg(const Field &field1, Field &field2)
{
  size_t rnmiss = 0;

  const auto ny = gridInqYsize(field1.grid);
  const bool lreduced = gridInqType(field1.grid) == GRID_GAUSSIAN_REDUCED;
  std::vector<int> reducedPoints, cumreducedPoints;
  auto nx = lreduced ? fillReducedPoints(field1.grid, ny, reducedPoints, cumreducedPoints) : gridInqXsize(field1.grid);

  std::vector<double> v(nx);

  for (size_t j = 0; j < ny; ++j)
    {
      if (lreduced) nx = reducedPoints[j];
      const size_t offset = lreduced ? cumreducedPoints[j] : j * nx;
      std::copy(field1.vec.begin() + offset, field1.vec.begin() + offset + nx, v.begin());

      const double ravg = field1.nmiss ? arrayAvgMV(nx, v, field1.missval) : arrayMean(nx, v);

      if (DBL_IS_EQUAL(ravg, field1.missval)) rnmiss++;

      field2.vec[j] = ravg;
    }

  field2.nmiss = rnmiss;
}

static void
prevarsum_zon(const std::vector<double> v, size_t nx, size_t nmiss, double missval, double &rsum, double &rsumw, double &rsumq,
              double &rsumwq)
{
  const double w = 1. / nx;

  rsum = 0;
  rsumq = 0;
  rsumw = 0;
  rsumwq = 0;

  if (nmiss)
    {
      for (size_t i = 0; i < nx; i++)
        if (!DBL_IS_EQUAL(v[i], missval))
          {
            rsum += w * v[i];
            rsumq += w * v[i] * v[i];
            rsumw += w;
            rsumwq += w * w;
          }
    }
  else
    {
      for (size_t i = 0; i < nx; i++)
        {
          rsum += w * v[i];
          rsumq += w * v[i] * v[i];
          rsumw += w;
          rsumwq += w * w;
        }
    }
}

void
zonvar(const Field &field1, Field &field2)
{
  size_t rnmiss = 0;
  size_t nmiss = field1.nmiss;
  double missval1 = field1.missval;
  double rsum = 0, rsumw = 0;
  double rsumq = 0, rsumwq = 0;

  const auto ny = gridInqYsize(field1.grid);
  const bool lreduced = gridInqType(field1.grid) == GRID_GAUSSIAN_REDUCED;
  std::vector<int> reducedPoints, cumreducedPoints;
  auto nx = lreduced ? fillReducedPoints(field1.grid, ny, reducedPoints, cumreducedPoints) : gridInqXsize(field1.grid);

  std::vector<double> v(nx);

  for (size_t j = 0; j < ny; j++)
    {
      if (lreduced) nx = reducedPoints[j];
      const size_t offset = lreduced ? cumreducedPoints[j] : j * nx;
      std::copy(field1.vec.begin() + offset, field1.vec.begin() + offset + nx, v.begin());

      prevarsum_zon(v, nx, nmiss, missval1, rsum, rsumw, rsumq, rsumwq);

      double rvar = IS_NOT_EQUAL(rsumw, 0) ? (rsumq * rsumw - rsum * rsum) / (rsumw * rsumw) : missval1;
      if (rvar < 0 && rvar > -1.e-5) rvar = 0;
      if (DBL_IS_EQUAL(rvar, missval1)) rnmiss++;
      field2.vec[j] = rvar;
    }

  field2.nmiss = rnmiss;
}

void
zonvar1(const Field &field1, Field &field2)
{
  size_t rnmiss = 0;
  size_t nmiss = field1.nmiss;
  double missval1 = field1.missval;
  double rsum = 0, rsumw = 0;
  double rsumq = 0, rsumwq = 0;

  const auto ny = gridInqYsize(field1.grid);
  const bool lreduced = gridInqType(field1.grid) == GRID_GAUSSIAN_REDUCED;
  std::vector<int> reducedPoints, cumreducedPoints;
  auto nx = lreduced ? fillReducedPoints(field1.grid, ny, reducedPoints, cumreducedPoints) : gridInqXsize(field1.grid);

  std::vector<double> v(nx);

  for (size_t j = 0; j < ny; j++)
    {
      if (lreduced) nx = reducedPoints[j];
      const size_t offset = lreduced ? cumreducedPoints[j] : j * nx;
      std::copy(field1.vec.begin() + offset, field1.vec.begin() + offset + nx, v.begin());

      prevarsum_zon(v, nx, nmiss, missval1, rsum, rsumw, rsumq, rsumwq);

      double rvar = (rsumw * rsumw > rsumwq) ? (rsumq * rsumw - rsum * rsum) / (rsumw * rsumw - rsumwq) : missval1;
      if (rvar < 0 && rvar > -1.e-5) rvar = 0;
      if (DBL_IS_EQUAL(rvar, missval1)) rnmiss++;
      field2.vec[j] = rvar;
    }

  field2.nmiss = rnmiss;
}

void
zonstd(const Field &field1, Field &field2)
{
  size_t rnmiss = 0;
  const double missval = field1.missval;
  const auto ny = gridInqYsize(field1.grid);

  zonvar(field1, field2);

  for (size_t j = 0; j < ny; j++)
    {
      const double rstd = varToStd(field2.vec[j], missval);
      if (DBL_IS_EQUAL(rstd, missval)) rnmiss++;
      field2.vec[j] = rstd;
    }

  field2.nmiss = rnmiss;
}

void
zonstd1(const Field &field1, Field &field2)
{
  size_t rnmiss = 0;
  const double missval = field1.missval;
  const auto ny = gridInqYsize(field1.grid);

  zonvar1(field1, field2);

  for (size_t j = 0; j < ny; j++)
    {
      const double rstd = varToStd(field2.vec[j], missval);
      if (DBL_IS_EQUAL(rstd, missval)) rnmiss++;
      field2.vec[j] = rstd;
    }

  field2.nmiss = rnmiss;
}

void
zonpctl(Field &field1, Field &field2, int p)
{
  size_t rnmiss = 0;
  const double missval = field1.missval;

  const auto ny = gridInqYsize(field1.grid);
  const bool lreduced = gridInqType(field1.grid) == GRID_GAUSSIAN_REDUCED;
  std::vector<int> reducedPoints, cumreducedPoints;
  auto nx = lreduced ? fillReducedPoints(field1.grid, ny, reducedPoints, cumreducedPoints) : gridInqXsize(field1.grid);

  std::vector<double> v(nx);

  if (field1.nmiss)
    {
      std::vector<double> array2(nx);

      for (size_t j = 0; j < ny; j++)
        {
          if (lreduced) nx = reducedPoints[j];
          const size_t offset = lreduced ? cumreducedPoints[j] : j * nx;
          std::copy(field1.vec.begin() + offset, field1.vec.begin() + offset + nx, v.begin());

          size_t l = 0;
          for (size_t i = 0; i < nx; i++)
            if (!DBL_IS_EQUAL(field1.vec[j * nx + i], missval)) array2[l++] = v[i];

          if (l > 0)
            {
              field2.vec[j] = percentile(array2.data(), l, p);
            }
          else
            {
              field2.vec[j] = missval;
              rnmiss++;
            }
        }
    }
  else
    {
      for (size_t j = 0; j < ny; j++)
        {
          if (lreduced) nx = reducedPoints[j];
          const size_t offset = lreduced ? cumreducedPoints[j] : j * nx;
          std::copy(field1.vec.begin() + offset, field1.vec.begin() + offset + nx, v.begin());

          if (nx > 0)
            {
              field2.vec[j] = percentile(v.data(), nx, p);
            }
          else
            {
              field2.vec[j] = missval;
              rnmiss++;
            }
        }
    }

  field2.nmiss = rnmiss;
}

void
zonfun(const Field &field1, Field &field2, int function)
{
  // clang-format off
  switch (function)
    {
    case func_min:   zonmin(field1, field2);    break;
    case func_max:   zonmax(field1, field2);    break;
    case func_range: zonrange(field1, field2);  break;
    case func_sum:   zonsum(field1, field2);    break;
    case func_mean:  zonmean(field1, field2);   break;
    case func_avg:   zonavg(field1, field2);    break;
    case func_std:   zonstd(field1, field2);    break;
    case func_std1:  zonstd1(field1, field2);   break;
    case func_var:   zonvar(field1, field2);    break;
    case func_var1:  zonvar1(field1, field2);   break;
    default: cdoAbort("function %d not implemented!", function);
    }
  // clang-format on
}
