/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include <cdi.h>

#include "functs.h"
#include "process_int.h"
#include "percentiles.h"

static void
mermin(const Field &field1, Field &field2)
{
  size_t rnmiss = 0;
  double rmin;

  const auto nx = gridInqXsize(field1.grid);
  const auto ny = gridInqYsize(field1.grid);

  std::vector<double> v(ny);

  for (size_t i = 0; i < nx; ++i)
    {
      for (size_t j = 0; j < ny; j++) v[j] = field1.vec[j * nx + i];

      if (field1.nmiss)
        {
          rmin = arrayMinMV(ny, v, field1.missval);
          if (DBL_IS_EQUAL(rmin, field1.missval)) rnmiss++;
        }
      else
        {
          rmin = arrayMin(ny, v);
        }

      field2.vec[i] = rmin;
    }

  field2.nmiss = rnmiss;
}

static void
mermax(const Field &field1, Field &field2)
{
  size_t rnmiss = 0;
  double rmax;

  const auto nx = gridInqXsize(field1.grid);
  const auto ny = gridInqYsize(field1.grid);

  std::vector<double> v(ny);

  for (size_t i = 0; i < nx; ++i)
    {
      for (size_t j = 0; j < ny; j++) v[j] = field1.vec[j * nx + i];

      if (field1.nmiss)
        {
          rmax = arrayMaxMV(ny, v, field1.missval);
          if (DBL_IS_EQUAL(rmax, field1.missval)) rnmiss++;
        }
      else
        {
          rmax = arrayMax(ny, v);
        }

      field2.vec[i] = rmax;
    }

  field2.nmiss = rnmiss;
}

static void
merrange(const Field &field1, Field &field2)
{
  size_t rnmiss = 0;
  double range;

  const auto nx = gridInqXsize(field1.grid);
  const auto ny = gridInqYsize(field1.grid);

  std::vector<double> v(ny);

  for (size_t i = 0; i < nx; ++i)
    {
      for (size_t j = 0; j < ny; j++) v[j] = field1.vec[j * nx + i];

      if (field1.nmiss)
        {
          range = arrayRangeMV(ny, v, field1.missval);
          if (DBL_IS_EQUAL(range, field1.missval)) rnmiss++;
        }
      else
        {
          range = arrayRange(ny, v);
        }

      field2.vec[i] = range;
    }

  field2.nmiss = rnmiss;
}

static void
mersum(const Field &field1, Field &field2)
{
  size_t rnmiss = 0;
  double rsum = 0;

  const auto nx = gridInqXsize(field1.grid);
  const auto ny = gridInqYsize(field1.grid);

  std::vector<double> v(ny);

  for (size_t i = 0; i < nx; ++i)
    {
      for (size_t j = 0; j < ny; j++) v[j] = field1.vec[j * nx + i];

      if (field1.nmiss)
        {
          rsum = arraySumMV(ny, v, field1.missval);
          if (DBL_IS_EQUAL(rsum, field1.missval)) rnmiss++;
        }
      else
        {
          rsum = arraySum(ny, v);
        }

      field2.vec[i] = rsum;
    }

  field2.nmiss = rnmiss;
}

static void
mermeanw(const Field &field1, Field &field2)
{
  const double missval = field1.missval;
  const auto nx = gridInqXsize(field1.grid);
  const auto ny = gridInqYsize(field1.grid);

  std::vector<double> v(ny), w(ny);

  size_t rnmiss = 0;
  for (size_t i = 0; i < nx; ++i)
    {
      for (size_t j = 0; j < ny; j++) v[j] = field1.vec[j * nx + i];
      for (size_t j = 0; j < ny; j++) w[j] = field1.weightv[j * nx + i];

      const double rmean = field1.nmiss ? arrayWeightedMeanMV(ny, v, w, missval) : arrayWeightedMean(ny, v, w, missval);

      if (DBL_IS_EQUAL(rmean, missval)) rnmiss++;

      field2.vec[i] = rmean;
    }

  field2.nmiss = rnmiss;
}

static void
meravgw(const Field &field1, Field &field2)
{
  const double missval = field1.missval;
  const auto nx = gridInqXsize(field1.grid);
  const auto ny = gridInqYsize(field1.grid);

  std::vector<double> v(ny), w(ny);

  size_t rnmiss = 0;
  for (size_t i = 0; i < nx; ++i)
    {
      for (size_t j = 0; j < ny; j++) v[j] = field1.vec[j * nx + i];
      for (size_t j = 0; j < ny; j++) w[j] = field1.weightv[j * nx + i];

      const double ravg = field1.nmiss ? arrayWeightedAvgMV(ny, v, w, missval) : arrayWeightedMean(ny, v, w, missval);

      if (DBL_IS_EQUAL(ravg, missval)) rnmiss++;

      field2.vec[i] = ravg;
    }

  field2.nmiss = rnmiss;
}

static void
prevarsum_merw(const std::vector<double> &v, const std::vector<double> &w, size_t ny, size_t nmiss, double missval, double &rsum,
               double &rsumw, double &rsumq, double &rsumwq)
{
  rsum = 0;
  rsumq = 0;
  rsumw = 0;
  rsumwq = 0;

  if (nmiss)
    {
      for (size_t j = 0; j < ny; j++)
        if (!DBL_IS_EQUAL(v[j], missval) && !DBL_IS_EQUAL(w[j], missval))
          {
            rsum += w[j] * v[j];
            rsumq += w[j] * v[j] * v[j];
            rsumw += w[j];
            rsumwq += w[j] * w[j];
          }
    }
  else
    {
      for (size_t j = 0; j < ny; j++)
        {
          rsum += w[j] * v[j];
          rsumq += w[j] * v[j] * v[j];
          rsumw += w[j];
          rsumwq += w[j] * w[j];
        }
    }
}

static void
mervarw(const Field &field1, Field &field2)
{
  size_t rnmiss = 0;
  size_t nmiss = field1.nmiss;
  double missval = field1.missval;
  double rsum = 0, rsumw = 0;
  double rsumq = 0, rsumwq = 0;

  const auto nx = gridInqXsize(field1.grid);
  const auto ny = gridInqYsize(field1.grid);

  std::vector<double> v(ny), w(ny);

  for (size_t i = 0; i < nx; i++)
    {
      for (size_t j = 0; j < ny; j++) v[j] = field1.vec[j * nx + i];
      for (size_t j = 0; j < ny; j++) w[j] = field1.weightv[j * nx + i];

      prevarsum_merw(v, w, ny, nmiss, missval, rsum, rsumw, rsumq, rsumwq);

      double rvar = IS_NOT_EQUAL(rsumw, 0) ? (rsumq * rsumw - rsum * rsum) / (rsumw * rsumw) : missval;
      if (rvar < 0 && rvar > -1.e-5) rvar = 0;
      if (DBL_IS_EQUAL(rvar, missval)) rnmiss++;
      field2.vec[i] = rvar;
    }

  field2.nmiss = rnmiss;
}

static void
mervar1w(const Field &field1, Field &field2)
{
  size_t rnmiss = 0;
  size_t nmiss = field1.nmiss;
  double missval = field1.missval;
  double rsum = 0, rsumw = 0;
  double rsumq = 0, rsumwq = 0;

  const auto nx = gridInqXsize(field1.grid);
  const auto ny = gridInqYsize(field1.grid);

  std::vector<double> v(ny), w(ny);

  for (size_t i = 0; i < nx; i++)
    {
      for (size_t j = 0; j < ny; j++) v[j] = field1.vec[j * nx + i];
      for (size_t j = 0; j < ny; j++) w[j] = field1.weightv[j * nx + i];

      prevarsum_merw(v, w, ny, nmiss, missval, rsum, rsumw, rsumq, rsumwq);

      double rvar = (rsumw * rsumw > rsumwq) ? (rsumq * rsumw - rsum * rsum) / (rsumw * rsumw - rsumwq) : missval;
      if (rvar < 0 && rvar > -1.e-5) rvar = 0;
      if (DBL_IS_EQUAL(rvar, missval)) rnmiss++;
      field2.vec[i] = rvar;
    }

  field2.nmiss = rnmiss;
}

static void
merstdw(const Field &field1, Field &field2)
{
  size_t rnmiss = 0;
  double missval = field1.missval;

  const auto nx = gridInqXsize(field1.grid);

  mervarw(field1, field2);

  for (size_t i = 0; i < nx; i++)
    {
      const double rstd = varToStd(field2.vec[i], missval);
      if (DBL_IS_EQUAL(rstd, missval)) rnmiss++;
      field2.vec[i] = rstd;
    }

  field2.nmiss = rnmiss;
}

static void
merstd1w(const Field &field1, Field &field2)
{
  size_t rnmiss = 0;
  double missval = field1.missval;

  const auto nx = gridInqXsize(field1.grid);

  mervar1w(field1, field2);

  for (size_t i = 0; i < nx; i++)
    {
      const double rstd = varToStd(field2.vec[i], missval);
      if (DBL_IS_EQUAL(rstd, missval)) rnmiss++;
      field2.vec[i] = rstd;
    }

  field2.nmiss = rnmiss;
}

void
merpctl(Field &field1, Field &field2, int p)
{
  size_t rnmiss = 0;
  double missval = field1.missval;

  const auto nx = gridInqXsize(field1.grid);
  const auto ny = gridInqYsize(field1.grid);

  std::vector<double> array2(nx);

  if (field1.nmiss)
    {
      for (size_t i = 0; i < nx; i++)
        {
          size_t l = 0;
          for (size_t j = 0; j < ny; j++)
            if (!DBL_IS_EQUAL(field1.vec[j * nx + i], missval)) array2[l++] = field1.vec[j * nx + i];

          if (l > 0)
            {
              field2.vec[i] = percentile(array2.data(), l, p);
            }
          else
            {
              field2.vec[i] = missval;
              rnmiss++;
            }
        }
    }
  else
    {
      for (size_t i = 0; i < nx; i++)
        {
          if (ny > 0)
            {
              for (size_t j = 0; j < ny; j++) array2[j] = field1.vec[j * nx + i];
              field2.vec[i] = percentile(array2.data(), ny, p);
            }
          else
            {
              field2.vec[i] = missval;
              rnmiss++;
            }
        }
    }

  field2.nmiss = rnmiss;
}

void
merfun(const Field &field1, Field &field2, int function)
{
  // clang-format off
  switch (function)
    {
    case func_min:   mermin(field1, field2);    break;
    case func_max:   mermax(field1, field2);    break;
    case func_range: merrange(field1, field2);  break;
    case func_sum:   mersum(field1, field2);    break;
    case func_meanw: mermeanw(field1, field2);  break;
    case func_avgw:  meravgw(field1, field2);   break;
    case func_stdw:  merstdw(field1, field2);   break;
    case func_std1w: merstd1w(field1, field2);  break;
    case func_varw:  mervarw(field1, field2);   break;
    case func_var1w: mervar1w(field1, field2);  break;
    default: cdoAbort("function %d not implemented!", function);
    }
  // clang-format on
}
