/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2006 Brockmann Consult
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Ydrunstat    ydrunmin          Multi-year daily running minimum
      Ydrunstat    ydrunmax          Multi-year daily running maximum
      Ydrunstat    ydrunsum          Multi-year daily running sum
      Ydrunstat    ydrunmean         Multi-year daily running mean
      Ydrunstat    ydrunavg          Multi-year daily running average
      Ydrunstat    ydrunvar          Multi-year daily running variance
      Ydrunstat    ydrunvar1         Multi-year daily running variance [Normalize by (n-1)]
      Ydrunstat    ydrunstd          Multi-year daily running standard deviation
      Ydrunstat    ydrunstd1         Multi-year daily running standard deviation [Normalize by (n-1)]
*/

#include <cdi.h>


#include "dmemory.h"
#include "functs.h"
#include "process_int.h"
#include "param_conversion.h"
#include "calendar.h"
#include "datetime.h"

#define NDAY 373

struct YDAY_STATS
{
  int64_t vdate[NDAY];
  int vtime[NDAY];
  FieldVector2D vars1[NDAY];
  FieldVector2D vars2[NDAY];
  int nsets[NDAY];
  int vlist;
};

static YDAY_STATS *
ydstatCreate(int vlistID)
{
  YDAY_STATS *stats = (YDAY_STATS *) Malloc(sizeof(YDAY_STATS));

  for (int dayoy = 0; dayoy < NDAY; dayoy++)
    {
      stats->vdate[dayoy] = 0;
      stats->vtime[dayoy] = 0;
      stats->nsets[dayoy] = 0;
    }

  stats->vlist = vlistID;

  return stats;
}

static void
ydstatDestroy(YDAY_STATS *stats)
{
  if (stats != nullptr) Free(stats);
}

static void
ydstatUpdate(YDAY_STATS *stats, int64_t vdate, int vtime, const FieldVector2D &vars1, const FieldVector2D &vars2, int nsets,
             int operfunc)
{
  bool lvarstd = vars2.size();

  int nvars = vlistNvars(stats->vlist);

  int year, month, day;
  cdiDecodeDate(vdate, &year, &month, &day);

  const int dayoy = (month >= 1 && month <= 12) ? (month - 1) * 31 + day : 0;
  if (dayoy < 0 || dayoy >= NDAY) cdoAbort("day %d out of range!", dayoy);

  stats->vdate[dayoy] = vdate;
  stats->vtime[dayoy] = vtime;

  if (!stats->vars1[dayoy].size())
    {
      fieldsFromVlist(stats->vlist, stats->vars1[dayoy], FIELD_VEC);
      if (lvarstd) fieldsFromVlist(stats->vlist, stats->vars2[dayoy], FIELD_VEC);
    }

  for (int varID = 0; varID < nvars; varID++)
    {
      if (vlistInqVarTimetype(stats->vlist, varID) == TIME_CONSTANT) continue;

      const int nlevels = zaxisInqSize(vlistInqVarZaxis(stats->vlist, varID));
      for (int levelID = 0; levelID < nlevels; levelID++)
        {
          if (stats->nsets[dayoy] == 0)
            {
              stats->vars1[dayoy][varID][levelID].vec = vars1[varID][levelID].vec;
              stats->vars1[dayoy][varID][levelID].nmiss = vars1[varID][levelID].nmiss;

              if (lvarstd)
                {
                  stats->vars2[dayoy][varID][levelID].vec = vars2[varID][levelID].vec;
                  stats->vars2[dayoy][varID][levelID].nmiss = vars2[varID][levelID].nmiss;
                }
            }
          else
            {
              if (lvarstd)
                {
                  vfarsum(stats->vars1[dayoy][varID][levelID], vars1[varID][levelID]);
                  vfarsum(stats->vars2[dayoy][varID][levelID], vars2[varID][levelID]);
                }
              else
                {
                  vfarfun(stats->vars1[dayoy][varID][levelID], vars1[varID][levelID], operfunc);
                }
            }
        }
    }

  stats->nsets[dayoy] += nsets;
}

static void
ydstatFinalize(YDAY_STATS *stats, int operfunc)
{
  int varID, levelID, nlevels;
  const int divisor = operfunc == func_std1 || operfunc == func_var1;

  const int nvars = vlistNvars(stats->vlist);

  for (int dayoy = 0; dayoy < NDAY; dayoy++)
    if (stats->nsets[dayoy])
      {
        switch (operfunc)
          {
          case func_avg:
          case func_mean:
            for (varID = 0; varID < nvars; varID++)
              {
                if (vlistInqVarTimetype(stats->vlist, varID) == TIME_CONSTANT) continue;
                nlevels = zaxisInqSize(vlistInqVarZaxis(stats->vlist, varID));
                for (levelID = 0; levelID < nlevels; levelID++)
                  vfarcdiv(stats->vars1[dayoy][varID][levelID], (double) stats->nsets[dayoy]);
              }
            break;

          case func_std:
          case func_std1:
            for (varID = 0; varID < nvars; varID++)
              {
                if (vlistInqVarTimetype(stats->vlist, varID) == TIME_CONSTANT) continue;
                nlevels = zaxisInqSize(vlistInqVarZaxis(stats->vlist, varID));
                for (levelID = 0; levelID < nlevels; levelID++)
                  vfarcstd(stats->vars1[dayoy][varID][levelID], stats->vars2[dayoy][varID][levelID], stats->nsets[dayoy], divisor);
              }
            break;

          case func_var:
          case func_var1:
            for (varID = 0; varID < nvars; varID++)
              {
                if (vlistInqVarTimetype(stats->vlist, varID) == TIME_CONSTANT) continue;
                nlevels = zaxisInqSize(vlistInqVarZaxis(stats->vlist, varID));
                for (levelID = 0; levelID < nlevels; levelID++)
                  vfarcvar(stats->vars1[dayoy][varID][levelID], stats->vars2[dayoy][varID][levelID], stats->nsets[dayoy], divisor);
              }
            break;
          }
      }
}

void *
Ydrunstat(void *process)
{
  int varID;
  int nrecs;
  int levelID;
  int tsID;
  int inp, its;

  cdoInitialize(process);

  // clang-format off
  cdoOperatorAdd("ydrunmin",   func_min,   0, nullptr);
  cdoOperatorAdd("ydrunmax",   func_max,   0, nullptr);
  cdoOperatorAdd("ydrunsum",   func_sum,   0, nullptr);
  cdoOperatorAdd("ydrunmean",  func_mean,  0, nullptr);
  cdoOperatorAdd("ydrunavg",   func_avg,   0, nullptr);
  cdoOperatorAdd("ydrunvar",   func_var,   0, nullptr);
  cdoOperatorAdd("ydrunvar1",  func_var1,  0, nullptr);
  cdoOperatorAdd("ydrunstd",   func_std,   0, nullptr);
  cdoOperatorAdd("ydrunstd1",  func_std1,  0, nullptr);
  // clang-format on

  const int operatorID = cdoOperatorID();
  const int operfunc = cdoOperatorF1(operatorID);

  operatorInputArg("number of timesteps");
  const int ndates = parameter2int(operatorArgv()[0]);

  const bool lvarstd = operfunc == func_std || operfunc == func_var || operfunc == func_std1 || operfunc == func_var1;

  const auto streamID1 = cdoOpenRead(0);

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = vlistDuplicate(vlistID1);

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = taxisDuplicate(taxisID1);
  if (taxisHasBounds(taxisID2)) taxisDeleteBounds(taxisID2);
  vlistDefTaxis(vlistID2, taxisID2);

  const int calendar = taxisInqCalendar(taxisID1);
  const int dpy = calendar_dpy(calendar);

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  const int maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  std::vector<CdoDateTime> datetime(ndates + 1);

  YDAY_STATS *stats = ydstatCreate(vlistID1);
  FieldVector3D vars1(ndates + 1), vars2(ndates + 1);

  for (its = 0; its < ndates; its++)
    {
      fieldsFromVlist(vlistID1, vars1[its], FIELD_VEC);
      if (lvarstd) fieldsFromVlist(vlistID1, vars2[its], FIELD_VEC);
    }

  for (tsID = 0; tsID < ndates; tsID++)
    {
      nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs == 0) cdoAbort("File has less then %d timesteps!", ndates);

      datetime[tsID].date = taxisInqVdate(taxisID1);
      datetime[tsID].time = taxisInqVtime(taxisID1);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          if (tsID == 0)
            {
              recList[recID].varID = varID;
              recList[recID].levelID = levelID;
              recList[recID].lconst = vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT;
            }

          Field &rvars1 = vars1[tsID][varID][levelID];

          cdoReadRecord(streamID1, rvars1.vec.data(), &rvars1.nmiss);

          if (lvarstd)
            {
              vfarmoq(vars2[tsID][varID][levelID], vars1[tsID][varID][levelID]);
              for (int inp = 0; inp < tsID; inp++)
                {
                  vfarsumq(vars2[inp][varID][levelID], rvars1);
                  vfarsum(vars1[inp][varID][levelID], rvars1);
                }
            }
          else
            {
              for (int inp = 0; inp < tsID; inp++)
                {
                  vfarfun(vars1[inp][varID][levelID], rvars1, operfunc);
                }
            }
        }
    }

  while (true)
    {
      datetime_avg(dpy, ndates, datetime.data());

      const auto vdate = datetime[ndates].date;
      const auto vtime = datetime[ndates].time;

      ydstatUpdate(stats, vdate, vtime, vars1[0], vars2[0], ndates, operfunc);

      datetime[ndates] = datetime[0];
      vars1[ndates] = vars1[0];
      if (lvarstd) vars2[ndates] = vars2[0];

      for (inp = 0; inp < ndates; inp++)
        {
          datetime[inp] = datetime[inp + 1];
          vars1[inp] = vars1[inp + 1];
          if (lvarstd) vars2[inp] = vars2[inp + 1];
        }

      nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs == 0) break;

      datetime[ndates - 1].date = taxisInqVdate(taxisID1);
      datetime[ndates - 1].time = taxisInqVtime(taxisID1);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          Field &rvars1 = vars1[ndates - 1][varID][levelID];

          cdoReadRecord(streamID1, rvars1.vec.data(), &rvars1.nmiss);

          if (lvarstd)
            {
              for (inp = 0; inp < ndates - 1; inp++)
                {
                  vfarsumq(vars2[inp][varID][levelID], rvars1);
                  vfarsum(vars1[inp][varID][levelID], rvars1);
                }
              vfarmoq(vars2[ndates - 1][varID][levelID], rvars1);
            }
          else
            {
              for (inp = 0; inp < ndates - 1; inp++)
                {
                  vfarfun(vars1[inp][varID][levelID], rvars1, operfunc);
                }
            }
        }

      tsID++;
    }

  /*
  // set the year to the minimum of years found on output timestep
  int outyear = 1e9;
  int year, month, day;
  for ( int dayoy = 0; dayoy < NDAY; dayoy++ )
    if ( stats->nsets[dayoy] )
      {
        cdiDecodeDate(stats->vdate[dayoy], &year, &month, &day);
        if ( year < outyear ) outyear = year;
      }
  for ( int dayoy = 0; dayoy < NDAY; dayoy++ )
    if ( stats->nsets[dayoy] )
      {
        cdiDecodeDate(stats->vdate[dayoy], &year, &month, &day);
        // printf("vdates[%d] = %d  nsets = %d\n", dayoy, stats->vdate[dayoy],
  stats->nsets[dayoy]); if ( year > outyear ) stats->vdate[dayoy] =
  cdiEncodeDate(outyear, month, day);
      }
  */
  ydstatFinalize(stats, operfunc);

  int otsID = 0;

  for (int dayoy = 0; dayoy < NDAY; dayoy++)
    if (stats->nsets[dayoy])
      {
        taxisDefVdate(taxisID2, stats->vdate[dayoy]);
        taxisDefVtime(taxisID2, stats->vtime[dayoy]);
        cdoDefTimestep(streamID2, otsID);

        for (int recID = 0; recID < maxrecs; recID++)
          {
            if (otsID && recList[recID].lconst) continue;

            int varID = recList[recID].varID;
            int levelID = recList[recID].levelID;
            Field &rvars1 = stats->vars1[dayoy][varID][levelID];

            cdoDefRecord(streamID2, varID, levelID);
            cdoWriteRecord(streamID2, rvars1.vec.data(), rvars1.nmiss);
          }

        otsID++;
      }

  ydstatDestroy(stats);

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
