/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Timstat    timmin          Time minimum
      Timstat    timmax          Time maximum
      Timstat    timsum          Time sum
      Timstat    timmean         Time mean
      Timstat    timavg          Time average
      Timstat    timvar          Time variance
      Timstat    timvar1         Time variance [Normalize by (n-1)]
      Timstat    timstd          Time standard deviation
      Timstat    timstd1         Time standard deviation [Normalize by (n-1)]
      Hourstat   hourmin         Hourly minimum
      Hourstat   hourmax         Hourly maximum
      Hourstat   hoursum         Hourly sum
      Hourstat   hourmean        Hourly mean
      Hourstat   houravg         Hourly average
      Hourstat   hourvar         Hourly variance
      Hourstat   hourvar1        Hourly variance [Normalize by (n-1)]
      Hourstat   hourstd         Hourly standard deviation
      Hourstat   hourstd1        Hourly standard deviation [Normalize by (n-1)]
      Daystat    daymin          Daily minimum
      Daystat    daymax          Daily maximum
      Daystat    daysum          Daily sum
      Daystat    daymean         Daily mean
      Daystat    dayavg          Daily average
      Daystat    dayvar          Daily variance
      Daystat    dayvar1         Daily variance [Normalize by (n-1)]
      Daystat    daystd          Daily standard deviation
      Daystat    daystd1         Daily standard deviation [Normalize by (n-1)]
      Monstat    monmin          Monthly minimum
      Monstat    monmax          Monthly maximum
      Monstat    monsum          Monthly sum
      Monstat    monmean         Monthly mean
      Monstat    monavg          Monthly average
      Monstat    monvar          Monthly variance
      Monstat    monvar1         Monthly variance [Normalize by (n-1)]
      Monstat    monstd          Monthly standard deviation
      Monstat    monstd1         Monthly standard deviation [Normalize by (n-1)]
      Yearstat   yearmin         Yearly minimum
      Yearstat   yearmax         Yearly maximum
      Yearstat   yearsum         Yearly sum
      Yearstat   yearmean        Yearly mean
      Yearstat   yearavg         Yearly average
      Yearstat   yearvar         Yearly variance
      Yearstat   yearvar1        Yearly variance [Normalize by (n-1)]
      Yearstat   yearstd         Yearly standard deviation
      Yearstat   yearstd1        Yearly standard deviation [Normalize by (n-1)]
*/

#include <cdi.h>


#include "cdo_options.h"
#include "functs.h"
#include "process_int.h"
#include "cdo_task.h"
#include "timer.h"
#include "datetime.h"
#include "printinfo.h"
#include "util_date.h"

#define USE_CDI_STREAM 1

enum
{
  HOUR_LEN = 4,
  DAY_LEN = 6,
  MON_LEN = 8,
  YEAR_LEN = 10
};

class ReadArguments
{
public:
  ReadArguments(FieldVector2D &input_vars) : vars(input_vars){};
  ~ReadArguments(){};
  int tsIDnext;
#ifdef USE_CDI_STREAM
  int streamID;
#else
  CdoStreamID streamID;
#endif
  int nrecs;
  RecordInfo *recList;
  FieldVector2D &vars;
};

static int num_recs = 0;

static void *
cdoReadTimestep(void *rarg)
{
  ReadArguments *readarg = (ReadArguments *) rarg;
  auto &input_vars = readarg->vars;
  RecordInfo *recList = readarg->recList;
  const auto streamID = readarg->streamID;
  int tsIDnext = readarg->tsIDnext;
  int nrecs = readarg->nrecs;

  // timer_start(timer_read);

  for (int recID = 0; recID < nrecs; ++recID)
    {
      int varID, levelID;
#ifdef USE_CDI_STREAM
      streamInqRecord(streamID, &varID, &levelID);
#else
      cdoInqRecord(streamID, &varID, &levelID);
#endif

      if (tsIDnext == 1 && recList)
        {
          recList[recID].varID = varID;
          recList[recID].levelID = levelID;
        }

      size_t nmiss;

#ifdef USE_CDI_STREAM
      if (Options::CDO_Memtype == MEMTYPE_FLOAT)
        streamReadRecordF(streamID, input_vars[varID][levelID].vecf.data(), &nmiss);
      else
        streamReadRecord(streamID, input_vars[varID][levelID].vec.data(), &nmiss);
#else
      if (Options::CDO_Memtype == MEMTYPE_FLOAT)
        cdoReadRecordF(streamID, input_vars[varID][levelID].vecf.data(), &nmiss);
      else
        cdoReadRecord(streamID, input_vars[varID][levelID].vec.data(), &nmiss);
#endif
      input_vars[varID][levelID].nmiss = nmiss;
    }

  // timer_stop(timer_read);

#ifdef USE_CDI_STREAM
  num_recs = streamInqTimestep(streamID, tsIDnext);
#else
  num_recs = cdoStreamInqTimestep(streamID, tsIDnext);
#endif

  return ((void *) &num_recs);
}

static void
vlistSetFrequency(int vlistID, int comparelen)
{
  const char *freq = nullptr;
  // clang-format off
  if      (comparelen == DAY_LEN)  freq = "day";
  else if (comparelen == MON_LEN)  freq = "mon";
  else if (comparelen == YEAR_LEN) freq = "year";
  // clang-format on
  if (freq) cdiDefAttTxt(vlistID, CDI_GLOBAL, "frequency", (int) strlen(freq), freq);
}

static void
addOperators(void)
{
  // clang-format off
  cdoOperatorAdd("xtimmin",    func_min,   DATE_LEN, nullptr);
  cdoOperatorAdd("xtimmax",    func_max,   DATE_LEN, nullptr);
  cdoOperatorAdd("xtimsum",    func_sum,   DATE_LEN, nullptr);
  cdoOperatorAdd("xtimmean",   func_mean,  DATE_LEN, nullptr);
  cdoOperatorAdd("xtimavg",    func_avg,   DATE_LEN, nullptr);
  cdoOperatorAdd("xtimvar",    func_var,   DATE_LEN, nullptr);
  cdoOperatorAdd("xtimvar1",   func_var1,  DATE_LEN, nullptr);
  cdoOperatorAdd("xtimstd",    func_std,   DATE_LEN, nullptr);
  cdoOperatorAdd("xtimstd1",   func_std1,  DATE_LEN, nullptr);
  cdoOperatorAdd("xyearmin",   func_min,   YEAR_LEN, nullptr);
  cdoOperatorAdd("xyearmax",   func_max,   YEAR_LEN, nullptr);
  cdoOperatorAdd("xyearsum",   func_sum,   YEAR_LEN, nullptr);
  cdoOperatorAdd("xyearmean",  func_mean,  YEAR_LEN, nullptr);
  cdoOperatorAdd("xyearavg",   func_avg,   YEAR_LEN, nullptr);
  cdoOperatorAdd("xyearvar",   func_var,   YEAR_LEN, nullptr);
  cdoOperatorAdd("xyearvar1",  func_var1,  YEAR_LEN, nullptr);
  cdoOperatorAdd("xyearstd",   func_std,   YEAR_LEN, nullptr);
  cdoOperatorAdd("xyearstd1",  func_std1,  YEAR_LEN, nullptr);
  cdoOperatorAdd("xmonmin",    func_min,   MON_LEN, nullptr);
  cdoOperatorAdd("xmonmax",    func_max,   MON_LEN, nullptr);
  cdoOperatorAdd("xmonsum",    func_sum,   MON_LEN, nullptr);
  cdoOperatorAdd("xmonmean",   func_mean,  MON_LEN, nullptr);
  cdoOperatorAdd("xmonavg",    func_avg,   MON_LEN, nullptr);
  cdoOperatorAdd("xmonvar",    func_var,   MON_LEN, nullptr);
  cdoOperatorAdd("xmonvar1",   func_var1,  MON_LEN, nullptr);
  cdoOperatorAdd("xmonstd",    func_std,   MON_LEN, nullptr);
  cdoOperatorAdd("xmonstd1",   func_std1,  MON_LEN, nullptr);
  // clang-format on
}

void *
XTimstat(void *process)
{
  TimeStat timestat_date = TimeStat::MEAN;
  int64_t vdate = 0, vdate0 = 0;
  int vtime = 0, vtime0 = 0;
  int varID;
  CdoStreamID streamID3 = CDO_STREAM_UNDEF;
  int vlistID3, taxisID3 = -1;
  bool lvfrac = false;
  char indate1[DATE_LEN + 1], indate2[DATE_LEN + 1];
  double vfrac = 1;

  cdoInitialize(process);

  addOperators();

  // clang-format off
  const auto operatorID = cdoOperatorID();
  auto operfunc = cdoOperatorF1(operatorID);
  const auto comparelen = cdoOperatorF2(operatorID);

  bool lmean   = operfunc == func_mean || operfunc == func_avg;
  bool lstd    = operfunc == func_std || operfunc == func_std1;
  bool lvarstd = operfunc == func_std || operfunc == func_var || operfunc == func_std1 || operfunc == func_var1;
  int divisor  = operfunc == func_std1 || operfunc == func_var1;
  // clang-format on
  bool lvars2 = lvarstd;

  if (operfunc == func_mean)
    {
      const auto oargc = operatorArgc();
      const auto oargv = operatorArgv();

      if (oargc == 1)
        {
          lvfrac = true;
          vfrac = atof(oargv[0]);
          if (Options::cdoVerbose) cdoPrint("Set vfrac to %g", vfrac);
          if (vfrac < 0 || vfrac > 1) cdoAbort("vfrac out of range!");
        }
      else if (oargc > 1)
        cdoAbort("Too many arguments!");
    }

  const int cmplen = DATE_LEN - comparelen;

#ifdef USE_CDI_STREAM
  const auto streamID1 = streamOpenRead(cdoGetStreamName(0));
  auto vlistID1 = streamInqVlist(streamID1);
#else
  const auto streamID1 = cdoOpenRead(0);
  auto vlistID1 = cdoStreamInqVlist(streamID1);
#endif

  const auto vlistID2 = vlistDuplicate(vlistID1);

  if (cmplen == 0) vlistDefNtsteps(vlistID2, 1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  taxisWithBounds(taxisID2);
  if (taxisInqType(taxisID2) == TAXIS_FORECAST) taxisDefType(taxisID2, TAXIS_RELATIVE);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto nvars = vlistNvars(vlistID1);

  vlistSetFrequency(vlistID2, comparelen);

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  if (Options::cdoDiag)
    {
      char filename[8192];
      strcpy(filename, cdoOperatorName(operatorID));
      strcat(filename, "_");
      strcat(filename, cdoGetStreamName(1));
      streamID3 = cdoOpenWrite(filename);

      vlistID3 = vlistDuplicate(vlistID1);

      for (varID = 0; varID < nvars; ++varID)
        {
          vlistDefVarDatatype(vlistID3, varID, CDI_DATATYPE_INT32);
          vlistDefVarMissval(vlistID3, varID, -1);
          vlistDefVarUnits(vlistID3, varID, "");
          vlistDefVarAddoffset(vlistID3, varID, 0);
          vlistDefVarScalefactor(vlistID3, varID, 1);
        }

      taxisID3 = taxisDuplicate(taxisID1);
      taxisWithBounds(taxisID3);
      vlistDefTaxis(vlistID3, taxisID3);

      cdoDefVlist(streamID3, vlistID3);
    }

  DateTimeList dtlist;
  dtlist.setStat(timestat_date);
  dtlist.setCalendar(taxisInqCalendar(taxisID1));

  int FIELD_MEMTYPE = 0;
  if (operfunc == func_mean && Options::CDO_Memtype == MEMTYPE_FLOAT) FIELD_MEMTYPE = FIELD_FLT;
  FieldVector3D input_vars(2);
  fieldsFromVlist(vlistID1, input_vars[0], FIELD_VEC | FIELD_MEMTYPE);
  fieldsFromVlist(vlistID1, input_vars[1], FIELD_VEC | FIELD_MEMTYPE);
  FieldVector2D samp1, vars1, vars2;
  fieldsFromVlist(vlistID1, samp1, FIELD_NONE);
  fieldsFromVlist(vlistID1, vars1, FIELD_VEC);
  if (lvars2) fieldsFromVlist(vlistID1, vars2, FIELD_VEC);

  int curFirst = 0;
  int curSecond = 1;
  (void) curSecond;

  ReadArguments readarg(input_vars[curFirst]);
  readarg.streamID = streamID1;

  bool lparallelread = Options::CDO_Parallel_Read > 0;
  bool ltsfirst = true;
  void *read_task = nullptr;
  void *readresult = nullptr;

  if (lparallelread)
    {
      read_task = cdo_task_new();
      if (read_task == nullptr)
        {
          lparallelread = false;
          cdoWarning("CDO tasks not available!");
        }
    }

  int tsID = 0;
  int otsID = 0;
#ifdef USE_CDI_STREAM
  auto nrecs = streamInqTimestep(streamID1, tsID);
#else
  auto nrecs = cdoStreamInqTimestep(streamID1, tsID);
#endif
  int maxrecs = nrecs;
  std::vector<RecordInfo> recList(maxrecs);

  tsID++;
  while (true)
    {
      int nsets = 0;
      while (nrecs > 0)
        {
          dtlist.taxisInqTimestep(taxisID1, nsets);
          vdate = dtlist.getVdate(nsets);
          vtime = dtlist.getVtime(nsets);

          if (nsets == 0) SET_DATE(indate2, vdate, vtime);
          SET_DATE(indate1, vdate, vtime);

          if (DATE_IS_NEQ(indate1, indate2, cmplen)) break;

          readarg.tsIDnext = tsID;
          readarg.nrecs = nrecs;
          readarg.recList = recList.data();
          readarg.vars = input_vars[curFirst];

          if (ltsfirst || !lparallelread)
            {
              ltsfirst = false;
              readresult = cdoReadTimestep(&readarg);
            }
          else
            {
              readresult = cdo_task_wait(read_task);
            }

          nrecs = *(int *) readresult;

          // std::swap(curFirst, curSecond);

          if (nrecs && lparallelread)
            {
              readarg.vars = input_vars[curFirst];
              readarg.tsIDnext = tsID + 1;
              cdo_task_start(read_task, cdoReadTimestep, &readarg);
            }

          if (nsets == 0)
            {
#ifdef _OPENMP
#pragma omp parallel for default(none) shared(maxrecs, recList, curFirst, input_vars, vars1, samp1) if (maxrecs > 1)
#endif
              for (int recID = 0; recID < maxrecs; recID++)
                {
                  const auto varID = recList[recID].varID;
                  const auto levelID = recList[recID].levelID;

                  auto &rvars1 = vars1[varID][levelID];
                  auto &rinput_var = input_vars[curFirst][varID][levelID];

                  const auto nmiss = rinput_var.nmiss;

                  vfarcpy(rvars1, rinput_var);
                  rvars1.nmiss = nmiss;
                  if (nmiss || !samp1[varID][levelID].empty())
                    {
                      const auto fieldsize = rvars1.size;
                      if (samp1[varID][levelID].empty()) samp1[varID][levelID].resize(fieldsize);

                      for (size_t i = 0; i < fieldsize; i++)
                        samp1[varID][levelID].vec[i] = !DBL_IS_EQUAL(rvars1.vec[i], rvars1.missval);
                    }
                }
            }
          else
            {
#ifdef _OPENMP
#pragma omp parallel for default(none) \
    shared(lvarstd, nsets, maxrecs, recList, curFirst, input_vars, vars1, samp1, vars2, operfunc) if (maxrecs > 1)
#endif
              for (int recID = 0; recID < maxrecs; recID++)
                {
                  const auto varID = recList[recID].varID;
                  const auto levelID = recList[recID].levelID;

                  auto &rvars1 = vars1[varID][levelID];
                  auto &rinput_var = input_vars[curFirst][varID][levelID];

                  const auto nmiss = rinput_var.nmiss;

                  if (nmiss || !samp1[varID][levelID].empty())
                    {
                      const auto fieldsize = rvars1.size;
                      if (samp1[varID][levelID].empty()) samp1[varID][levelID].resize(fieldsize, nsets);

                      for (size_t i = 0; i < fieldsize; i++)
                        if (!DBL_IS_EQUAL(rinput_var.vec[i], rvars1.missval)) samp1[varID][levelID].vec[i]++;
                    }

                  if (lvarstd)
                    {
                      vfarsumq(vars2[varID][levelID], rinput_var);
                      vfarsum(rvars1, rinput_var);
                    }
                  else
                    {
                      vfarfun(rvars1, rinput_var, operfunc);
                    }
                }
            }

          if (nsets == 0 && lvarstd)
            for (int recID = 0; recID < maxrecs; recID++)
              {
                const auto varID = recList[recID].varID;
                const auto levelID = recList[recID].levelID;

                if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT) continue;

                vfarmoq(vars2[varID][levelID], vars1[varID][levelID]);
              }

          vdate0 = vdate;
          vtime0 = vtime;
          nsets++;
          tsID++;
        }

      if (nrecs == 0 && nsets == 0) break;

      if (lmean)
        {
#ifdef _OPENMP
#pragma omp parallel for default(none) shared(vlistID1, nsets, maxrecs, recList, vars1, samp1) if (maxrecs > 1)
#endif
          for (int recID = 0; recID < maxrecs; recID++)
            {
              const auto varID = recList[recID].varID;
              const auto levelID = recList[recID].levelID;

              if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT) continue;

              if (!samp1[varID][levelID].empty())
                vfardiv(vars1[varID][levelID], samp1[varID][levelID]);
              else
                vfarcdiv(vars1[varID][levelID], (double) nsets);
            }
        }
      else if (lvarstd)
        {
          for (int recID = 0; recID < maxrecs; recID++)
            {
              const auto varID = recList[recID].varID;
              const auto levelID = recList[recID].levelID;

              if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT) continue;

              auto &rsamp1 = samp1[varID][levelID];
              auto &rvars1 = vars1[varID][levelID];
              auto &rvars2 = vars2[varID][levelID];

              if (!rsamp1.empty())
                {
                  if (lstd)
                    vfarstd(rvars1, rvars2, rsamp1, divisor);
                  else
                    vfarvar(rvars1, rvars2, rsamp1, divisor);
                }
              else
                {
                  if (lstd)
                    vfarcstd(rvars1, rvars2, nsets, divisor);
                  else
                    vfarcvar(rvars1, rvars2, nsets, divisor);
                }
            }
        }

      if (Options::cdoVerbose)
        cdoPrint("%s %s  vfrac = %g, nsets = %d", dateToString(vdate0).c_str(), timeToString(vtime0).c_str(), vfrac, nsets);

      if (lvfrac && operfunc == func_mean)
        for (int recID = 0; recID < maxrecs; recID++)
          {
            const auto varID = recList[recID].varID;
            const auto levelID = recList[recID].levelID;
            auto &rvars1 = vars1[varID][levelID];

            if (vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT) continue;

            const auto missval = rvars1.missval;
            if (!samp1[varID][levelID].empty())
              {
                const auto fieldsize = rvars1.size;
                size_t irun = 0;
                for (size_t i = 0; i < fieldsize; ++i)
                  {
                    if ((samp1[varID][levelID].vec[i] / nsets) < vfrac)
                      {
                        rvars1.vec[i] = missval;
                        irun++;
                      }
                  }

                if (irun) rvars1.nmiss = fieldNumMiss(rvars1);
              }
          }

      dtlist.statTaxisDefTimestep(taxisID2, nsets);
      cdoDefTimestep(streamID2, otsID);

      if (Options::cdoDiag)
        {
          dtlist.statTaxisDefTimestep(taxisID3, nsets);
          cdoDefTimestep(streamID3, otsID);
        }

      for (int recID = 0; recID < maxrecs; recID++)
        {
          const auto varID = recList[recID].varID;
          const auto levelID = recList[recID].levelID;
          auto &rvars1 = vars1[varID][levelID];

          if (otsID && vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT) continue;

          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, rvars1.vec.data(), rvars1.nmiss);

          if (Options::cdoDiag)
            {
              if (!samp1[varID][levelID].empty())
                {
                  cdoDefRecord(streamID3, varID, levelID);
                  cdoWriteRecord(streamID3, samp1[varID][levelID].vec.data(), 0);
                }
            }
        }

      if (nrecs == 0) break;
      otsID++;
    }

  if (Options::cdoDiag) cdoStreamClose(streamID3);
  cdoStreamClose(streamID2);
#ifdef USE_CDI_STREAM
  streamClose(streamID1);
#else
  cdoStreamClose(streamID1);
#endif

  cdoFinish();

  return nullptr;
}
